/*
	Copyright (C) 1994 Sean Luke

	COWSProtocols.h
	Version 1.0
	Sean Luke
	
*/



#import "COWSStringNode.h"
#import "COWSArgumentList.h"



@protocol LibraryControl
	// The interpreter adheres to this protocol.
	// Used to load libraries, and to allow libraries to
	// top the interpreter, if need be, until they get an answer
	// back to give to it.

- addLibraryFunction:(const char*) this_name
	selector: (SEL) this_selector
	target: this_target;
- makeMeALibraryDelegate:sender;
- (BOOL) canPause;

// pauseInterpreting removes the interpreter's driving function.
// resumeInterpreting puts it back.  resumeInterpreting is rarely used,
// only when it's possible a timed entry is serviced while _still_in_the_scope_
// of the library function.
// resumeInterpretingWithValue is usually used instead, when the interpreter
// needs to be given a return value and started up outside the scope of the
// library function.
// All three of these functions should only be used if the library can be
// paused in the first place (see canPause above).

- pauseInterpreting;
- resumeInterpreting;
- resumeInterpretingWithValue:(COWSStringNode*)this_value;
@end



@protocol InterpreterToLibrary
	// Libraries adhere to this protocol.
	// Used to load libraries.

- loadLibrary:sender;

@end


@protocol InterpreterToLibraryDelegate
	// After pausing an interpreter, a library might need to know if the
	// interpreter was stopped by the user manually.  For this, the interpreter
	// can set itself up as a lightweight delegate (libraryDelegate)
	// to just receive the following messages:

- interpreterDidStop:sender;		// these are different from below because
- interpreterDidStart:sender;		// it's possible for a library to be both
									// a delegate of an interpreter and a
									// library delegate of an interpreter!
@end



@protocol InterpreterToDelegate
	// Delegates of the interpreter adhere to this protocol.
	// This protocol consists of delegate messages from the interpreter.

- interpreterStarted:sender;
- finishedInterpreting:(const char*)returnValue:(int)thisMessage:sender;
- errorInterpreting:(int) thisError:(const char*)thisFunction:
	(int)thisPosition:(const char*)thisString:sender;
- interpreterStopped:sender;		// sent in addition to either of the
									// two above, or alone if the 
									// interpreter was cancelled from outside.
@end





@protocol InterpreterControl
	// The interpreter adheres to this protocol.
	// This protocol gives the application control over the interpreter itself.

- addLibrary:this_library;
- clearLibraryFunctions;
- (int) setProgram:(const char*) this_string;	
- delegate;
- setDelegate:this_delegate;
- tempDelegate;
- setTempDelegate:this_delegate;
- interpretFunction:(const char*) this_name 
	arguments:(COWSArgumentList*)these_arguments;
- stopInterpreting;
- setLocked:(BOOL) yes_or_no;
- setForeground:(BOOL) yes_or_no;
- printErrors:(BOOL) yes_or_no;
- (BOOL) printingErrors;
- (BOOL) locked;
- (BOOL) working;
- (BOOL) running;
- (BOOL) foreground;
@end


@protocol InterpreterIPC
	// The IPC Library adheres to this protocol.
	// This protocol gives one app's interpreter control over
	// another app's interpreter.
	
- (oneway void) sendFunction:(const char*) this_name;
	// synchronous communication
- (const char*) result;
- (BOOL) resultIsError;
- (int) state;

- (void) addArgument:(const char*)this_argument;	// returns NULL
- (oneway void) sendOutFunction:(const char*) this_name;
	// asynchronous communication
- (BOOL) isForeground;
@end
