/*
 * Copyright (c) 1996 Awwwesome Webware. All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this software
 * and its documentation for NON-COMMERCIAL purposes and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies. 
 *
 * AWWWESOME WEBWARE MAKES NO REPRESENTATIONS OR WARRANTIES 
 * ABOUT THE SUITABILITY OF THE SOFTWARE, EITHER EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR 
 * PURPOSE, OR NON-INFRINGEMENT. AWWWESOME WEBWARE SHALL 
 * NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY LICENSEE AS A 
 * RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE 
 * OR ITS DERIVATIVES.
 */
package awwwesome.visualworld;

import java.awt.Graphics;
import java.awt.Polygon;
import java.awt.Color;


/**
 * A cube is a six sided polyhedron.   Cubes may in fact
 * be scaled in one axis or another.  Although they are
 * no longer cubes in the mathematical sense, they remain
 * valid members of this cube class.
 */
public class Cube extends Polyhedron {
    static int[][] faces= new int[6][4];

    static {
        //      0---1        Y   
        //     /|  /|        |  /
        //    3---2 |        | /
        //    | | | |        |/
        //    | 4-|-5   -----+------ X
        //    |/  |/        /|
        //    7---6        / |
        //                /  |
        //               Z
        //
        // The origin is at the center of the cube.
        // Positive Z is into the page.
        //
        //
        faces[0][0] = 0;
        faces[0][1] = 1;
        faces[0][2] = 2;
        faces[0][3] = 3;
    
        faces[1][0] = 7;
        faces[1][1] = 6;
        faces[1][2] = 5;
        faces[1][3] = 4;

        faces[2][0] = 1;
        faces[2][1] = 0;
        faces[2][2] = 4;
        faces[2][3] = 5;

        faces[3][0] = 2;
        faces[3][1] = 1;
        faces[3][2] = 5;
        faces[3][3] = 6;

        faces[4][0] = 3;
        faces[4][1] = 2;
        faces[4][2] = 6;
        faces[4][3] = 7;

        faces[5][0] = 0;
        faces[5][1] = 3;
        faces[5][2] = 7;
        faces[5][3] = 4;
    }
    
    public Cube(double x, double y, double z)
	{
        super(8);

        points[0].v[0] = -x/2; points[0].v[1] =  y/2; points[0].v[2] = -z/2;
        points[1].v[0] =  x/2; points[1].v[1] =  y/2; points[1].v[2] = -z/2;
        points[2].v[0] =  x/2; points[2].v[1] =  y/2; points[2].v[2] =  z/2;
        points[3].v[0] = -x/2; points[3].v[1] =  y/2; points[3].v[2] =  z/2;
        points[4].v[0] = -x/2; points[4].v[1] = -y/2; points[4].v[2] = -z/2;
        points[5].v[0] =  x/2; points[5].v[1] = -y/2; points[5].v[2] = -z/2;
        points[6].v[0] =  x/2; points[6].v[1] = -y/2; points[6].v[2] =  z/2;
        points[7].v[0] = -x/2; points[7].v[1] = -y/2; points[7].v[2] =  z/2;
	}

    public Cube(double w) {
        super(8);

        double w2 = w/2;

        points[0].v[0] = -w2; points[0].v[1] =  w2; points[0].v[2] = -w2;
        points[1].v[0] =  w2; points[1].v[1] =  w2; points[1].v[2] = -w2;
        points[2].v[0] =  w2; points[2].v[1] =  w2; points[2].v[2] =  w2;
        points[3].v[0] = -w2; points[3].v[1] =  w2; points[3].v[2] =  w2;
        points[4].v[0] = -w2; points[4].v[1] = -w2; points[4].v[2] = -w2;
        points[5].v[0] =  w2; points[5].v[1] = -w2; points[5].v[2] = -w2;
        points[6].v[0] =  w2; points[6].v[1] = -w2; points[6].v[2] =  w2;
        points[7].v[0] = -w2; points[7].v[1] = -w2; points[7].v[2] =  w2;
    }

    public Cube() {
        this(1.f);    
    }

    public int getPolygonCount() {
        return 6;
    }

    public Polygon3D transformPolygonToEye(int f, ViewTransformer viewTransformer) {
        Polygon3D poly = new Polygon3D(color, 4);

        poly.addPoint(viewTransformer.transform(points[faces[f][0]]));
        poly.addPoint(viewTransformer.transform(points[faces[f][1]]));
        poly.addPoint(viewTransformer.transform(points[faces[f][2]]));
        poly.addPoint(viewTransformer.transform(points[faces[f][3]]));

        return poly;
    }
}


