
package awwwesome.visualworld;

import java.awt.Graphics;
import java.awt.Polygon;
import java.awt.Color;
import java.awt.FontMetrics;
import java.util.Enumeration;
import neurodata.Colors;

/** A line in space, stretching between two points.  
These points are defined by two Point3D objects which are passed 
to Line3D when it is initially created.  Line3D does not actually
contain any coordinate definitions of its own, but depends on the
Point3D objects to tell it where it's located.   A line can have
a label and/or two different colors (each for half of the line).  */
public class Line3D extends VisualObject {
    public Color color;
    public Color othercolor;

    Point3D from;
    Point3D to;
    public String text;

    

    /** Construct a line with a label txt, and two Point3D objects
     defining where the line is "from" and where it goes "to". */
    public Line3D(String txt,Point3D f, Point3D t)
    {
        color = Color.black;
	othercolor=Color.black;
	from=f; to=t;
	text=txt;
    }


    /** Construct a line with no label, and two Point3D objects
     defining where the line is "from" and where it goes "to". */
    public Line3D(Point3D f, Point3D t) {this("",f,t); }


    /** You should not call this constructor -- it will cause later
      null pointer exceptions. */
    public Line3D() {this("",null,null); }  


    /** Set the color of the line segment drawn from the center of
      the line to the "to" point. */
    public void setOtherColor(Color c)
	{
	othercolor=c;
	}

    /** Set the color of the line segment drawn from the "from" point
      to the center of the line */
    public void setColor(Color c) {
        color = c;
    }


    /** Transform the line by multiplying it by m.  Actually, nothing
     is done here, since the line contains no coordinates. */
    public VisualObject mul(Matrix4x4 m) {
    /* Nothing done here -- we rely on our points to multiply for us */
        return this;
    }
    

    /** Return the line's Z coordinate value for sorting. */
    public double getEyeZ() {
    /* This is only done while sorting, which means that at this
       point, the transform to Eye coordinates has already been performed.
       So we assume this, and return the Eye coordinate of one of our 
       points. */
        return from.getEyeZ();   
	}


    /** Transform the line to eye coordinates.  Actually, nothing
      is done here, since the line contains no coordinates of its own. */
    public void transformToEye(ViewTransformer viewTransformer) {
    /* Nothing done here -- we rely on our points to transform for us */
    }

    public void project(Graphics g, ViewTransformer viewTransformer, double distance)
	{
	if (from.eyepoint.v[2]<distance && to.eyepoint.v[2]<distance)
	    project(g,viewTransformer);
	}

    /** Project that sucker. */
    public void project(Graphics g, ViewTransformer viewTransformer) {
    Vector4 middle = viewTransformer.transform(from.point.copy().add(to.point).scale((double).5));
    IntVector2 v1= viewTransformer.project(from.eyepoint);
    IntVector2 v2= viewTransformer.project(to.eyepoint);
    IntVector2 v3= viewTransformer.project(middle);

   /* Draw half-way */
    g.setColor(color);
    g.drawLine(v1.v[0],v1.v[1],v3.v[0],v3.v[1]);
    
    /* Draw other half */
    g.setColor(othercolor);
    g.drawLine(v3.v[0],v3.v[1],v2.v[0],v2.v[1]);
    
    /* Draw Label? */
    if (!(text.equals("")))    // not empty
	{
	FontMetrics f=g.getFontMetrics();
	int ascent=f.getMaxAscent();
	int descent=f.getMaxDecent(); /* Misspelling, but works with old Java */
	int width=f.stringWidth(text);
	
	/* Draw Box */
	g.setColor(Colors.backgroundColor);
	g.fillRect(v3.v[0]-width/2-1, v3.v[1]-(ascent+descent)/2-1, 
		   width+2,ascent+descent+2);
	g.setColor(Colors.labelColor);
	g.drawRect(v3.v[0]-width/2-2, v3.v[1]-(ascent+descent)/2-2, 
		   width+4,ascent+descent+4);

	/* Draw Text (in black)*/
    
	g.drawString(text,v3.v[0]-width/2,v3.v[1]+(ascent+descent)/2-descent);
	}
    }

    public void print() {
        String s = "Line[" + from.point +"," + to.point + ",\"" + text + "\"]";
        System.out.println(s);
    }
}


