/*
 * Copyright (c) 1996 Awwwesome Webware. All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this software
 * and its documentation for NON-COMMERCIAL purposes and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies. 
 *
 * AWWWESOME WEBWARE MAKES NO REPRESENTATIONS OR WARRANTIES 
 * ABOUT THE SUITABILITY OF THE SOFTWARE, EITHER EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR 
 * PURPOSE, OR NON-INFRINGEMENT. AWWWESOME WEBWARE SHALL 
 * NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY LICENSEE AS A 
 * RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE 
 * OR ITS DERIVATIVES.
 */
package awwwesome.visualworld;

import java.awt.Graphics;
import java.awt.Polygon;
import java.awt.Color;
import java.util.Enumeration;


/**
 * A polyhedron visual object.  A polyhedron is a closed
 * figure constucted from polygons.
 *
 */
public abstract class Polyhedron extends VisualObject {
    Color color;
    Color othercolor; 

    int n;
    Vector4[] points;

    VectorPolygon3D eyePolygons;

    /**
     * Construct a polyhedron with space for n points.
     * The deriving classes constructor should initialize
     * these points.
     * <p>
     * The default color is red.
     */    
    public Polyhedron(int n)
    {
        this.n = n;

        color = Color.red;
	othercolor = Color.black;

        points = new Vector4[n];

        for(int i=0; i<n; ++i)
            points[i] = new Vector4();
    }

    public void setColor(Color c) {
        color = c;
    }

    public void setOtherColor(Color c)
	{
	othercolor=c;
	}

    /**
     * Transform every point of the polyhedron.
     */
    public VisualObject mul(Matrix4x4 m) {
        for(int i=0; i<n; ++i) {
            points[i].mul(m);    
        }
        return this;
    }
    
    /**
     * Return the number of polygonal faces in the polyhedron.
     */
    protected abstract int getPolygonCount();
    
    protected abstract Polygon3D transformPolygonToEye(int f, ViewTransformer viewTransformer);

    public double getEyeZ() {
        if (eyePolygons.size() == 0)
            return 0.0f;

        Polygon3D p = (Polygon3D) eyePolygons.elementAt(0);
        return p.verticies[0].v[2];
    }

    public void transformToEye(ViewTransformer viewTransformer) {
        eyePolygons = new VectorPolygon3D();

        for(int i=0; i<getPolygonCount(); ++i) {

            Polygon3D p = transformPolygonToEye(i, viewTransformer);

            Vector4 v1 = (Vector4)(p.verticies[0].copy().sub(p.verticies[1]));
            Vector4 v2 = (Vector4)(p.verticies[2].copy().sub(p.verticies[1]));

            Vector4 n = ((Vector4)(v1.copy())).cross(v2);

            //System.out.println("");
            //System.out.println(p.toString());
            //System.out.println("  v1=" + v1.toString());
            //System.out.println("  v2=" + v2.toString());
            //System.out.println("  n=" + n.toString());

            /*
             * Visible if normal faces in Z direction.
             */
            if (n.v[2] > 0) {

                /*
                 * Shade
                 */
                double l = n.v[2];
                double m = (n.v[0] * n.v[0] + n.v[1]*n.v[1] + n.v[2]*n.v[2]);
                double r = (double)(l/Math.sqrt(m));

                //System.out.println("l=" + l + " r="  + r);

                //if (r < .5f) r = .5f;

                int red = (int)(p.color.getRed()*r);
                int grn = (int)(p.color.getGreen()*r);
                int blu = (int)(p.color.getBlue()*r);

                p.color = new Color(red, grn, blu);

                eyePolygons.addElement(p);        
            }
        }
    }

    /**
     * Paint a single polyhedron.
     */				
    public void project(Graphics g, ViewTransformer viewTransformer, double distance)
	{
        for(Enumeration e = eyePolygons.elements(); e.hasMoreElements(); ) {
                Polygon3D p = (Polygon3D)e.nextElement();

		/* heck to determine if item is behind me */
		boolean should_draw=true;
		for(int x=0;x<p.vCount;x++)
		    if (p.verticies[x].v[2] > distance)
			{ should_draw=false; break; }
		if (should_draw)
		    {
		    g.setColor(p.color);
		    
		    Polygon poly = new Polygon();
		    IntVector2 v0 = new IntVector2();
		    for(int j=0; j<p.n; ++j) {
                    IntVector2 v = viewTransformer.project(p.verticies[j]);
		    if (j==0) v0=v;
                    poly.addPoint(v.v[0], v.v[1]);
		    }
		    if (p.n!=0) poly.addPoint(v0.v[0],v0.v[1]);
		    /*g.fillPolygon(poly);*/
		    g.setColor(othercolor);
		    g.drawPolygon(poly);
		    }
		}
	}

    public void project(Graphics g, ViewTransformer viewTransformer) {

        for(Enumeration e = eyePolygons.elements(); e.hasMoreElements(); ) {
                Polygon3D p = (Polygon3D)e.nextElement();

                g.setColor(p.color);

                Polygon poly = new Polygon();
		IntVector2 v0 = new IntVector2();
                for(int j=0; j<p.n; ++j) {
                    IntVector2 v = viewTransformer.project(p.verticies[j]);
		    if (j==0) v0=v;
                    poly.addPoint(v.v[0], v.v[1]);
                }
		if (p.n!=0) poly.addPoint(v0.v[0],v0.v[1]);
                /*g.fillPolygon(poly);*/
		g.setColor(othercolor);
		g.drawPolygon(poly);
        }
    }

    public void print() {
        String s = "Polyhedron[";

        for(int i=0; i<n; ++i) {
            s += points[i] + " ";            
        }
        s += "]";

        System.out.println(s);
    }
}


