/*
 * Copyright (c) 1996 Awwwesome Webware. All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this software
 * and its documentation for NON-COMMERCIAL purposes and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies. 
 *
 * AWWWESOME WEBWARE MAKES NO REPRESENTATIONS OR WARRANTIES 
 * ABOUT THE SUITABILITY OF THE SOFTWARE, EITHER EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR 
 * PURPOSE, OR NON-INFRINGEMENT. AWWWESOME WEBWARE SHALL 
 * NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY LICENSEE AS A 
 * RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE 
 * OR ITS DERIVATIVES.
 */
package awwwesome.visualworld;

import java.util.Vector;
import java.util.Enumeration;
import java.awt.Graphics;
import java.awt.Color;
import java.awt.Polygon;


/**
 *A visual world model.  VisualObjects may be added to the
 *world.  The world may be painted.
 */
public class VisualWorld {
    boolean debugFlag = false;
    Color bgColor = Color.lightGray;

    public void debug(boolean flag) {this.debugFlag = flag;}

    public void setBgColor(Color c)
    {
       bgColor = c;
    }

    /**
     * All the objects in the world, except the camera.
     */
    public Vector objects = new Vector();

    private VisualObject objectAt(int i) {
        return (VisualObject)(objects.elementAt(i));
    }

    private void setObjectAt(VisualObject o, int i) {
        objects.setElementAt(o, i);
    }

    /*--------------------- Camera variables and functions --------------------*/
    /**
     * The view transformation derived from camera settings
     */
    ViewTransformer viewTransformer;
    
    /**
     * When set, the next paint operation will cause the camera parameters
     * to be recomputed.  It's set by any operation that changes the
     * camera location or  orientation.
     */
    boolean newCameraSettings = true;

    /**
     * The camera location.  Default: (100, 100, -100)
     */
    Vector4 cameraLocation = new Vector4(100, 100, -100, 1);

    /** 
     * The forward direction of the camera. Default (-1, -1, 1).
     */
    Vector4 cameraDirection = new Vector4(-1, -1, -1, 1);
    
    /**
     * The camera's top faces in as closely as possible to 
     * this direction.  Default: Y direction.
     */
    Vector4 cameraUp = new Vector4(0, 1, 0, 1);
    
    /**
     * The camera looks at this point. No default.
     */
    Vector4 cameraSubject = null;

    /*
     * Point the camera at the given point.  The camera
     * will continue to look at this point, no matter where
     * it is moved, until another call is made to cameraLookAt()
     * or cameraDirection().
     */
    public void setCameraSubject(Vector4 subject) {
        newCameraSettings = true;
        cameraSubject = subject.copy();
    }

    /*
     * Face the camera in the given direction.  The camera
     * will continue to point in this direction, no matter where
     * it is moved, until another call is made to cmaeraLookAt()
     * or cameraDirection().
     */
    public void setCameraDirection(Vector4 direction) {
        newCameraSettings = true;
        cameraSubject = null;
        cameraDirection = direction.copy();
    }


    /**
     * Move the camera to the specified location.
     */
    public void setCameraLocation(Vector4 location) {
        newCameraSettings = true;
        cameraLocation = location.copy();
    }

    /** 
     * Set the direction of the cameras up vector.
     * The camera is rotated about it's direction
     * vector until its top faces as close as
     * possible to the specified up vector.
     */
    public void setCameraUp(Vector4 up) {
        newCameraSettings = true;
        cameraUp = up.copy();
    }

    /*--------------------- End Camera variables and functions --------------------*/

    /**
     * Add an object to the world.
     */
    public void add(VisualObject object) {
        objects.addElement(object);
    }

    /**
     * Paint the world.
     */
    synchronized public void paint(Graphics g, int w, int h) {
        int w2 = w/2;
        int h2 = h/2;

        computeView(w, h);

        Enumeration e = objects.elements();
        VisualObject vo;

        /*
         * Paint a background
         */
        Polygon bg = new Polygon();
        bg.addPoint(0, 0);
        bg.addPoint(w, 0);
        bg.addPoint(w, h);
        bg.addPoint(0, h);

        g.setColor(bgColor);
        g.fillPolygon(bg);

        /*
         * Project the objects using the viewTransform.
         */
        for(e = objects.elements(); e.hasMoreElements();) {
            vo = (VisualObject)e.nextElement();
            vo.transformToEye(viewTransformer);
        }

        /*
         * Sort the objects by Z order
         */
        sortObjects();

        /* 
         * Draw the objects
         */
	double magnitude=cameraLocation.magnitude();  /* assumes we're always looking at the origin! */
        for(e = objects.elements(); e.hasMoreElements();) {
            vo = (VisualObject)e.nextElement();
            {
	    vo.project(g, viewTransformer);
	    }
        }
    }

    void removeObjects()
	{
	objects.removeAllElements();
	}

    void sortObjects() {
        sortRange(0, objects.size()-1, 0);
    }

    void sortRange(int start, int end, int depth) {
        double highZ, lowZ;
        int high, low;

        if (start >= end) return;

        //System.out.println("----");
        //for(int i=start; i<=end; ++i) {
        //    System.out.println("          ".substring(0, depth) + i + ": " + objectAt(i).getEyeZ());
        //}

        int middle = (start + end)/2;
        double middleZ = ((VisualObject)objects.elementAt(middle)).getEyeZ();

        low = start;
        high= end;

        while(low <= high) {

            while(low <= high && objectAt(low).getEyeZ() <= middleZ)
                ++low;

            while(low <= high && objectAt(high).getEyeZ() > middleZ)
                --high;

            if (low <= high) {
                VisualObject o = objectAt(high);
                setObjectAt(objectAt(low), high);
                setObjectAt(o, low);
            }
        }

        if (low > end) {
            VisualObject o = objectAt(high);
            setObjectAt(objectAt(middle), high);
            setObjectAt(o, middle);
    
            sortRange(start, end-1, depth+1);
        } else {
            sortRange(start, high, depth+1);
            sortRange(low, end, depth+1);
        }
    }

    /**
     * If the camera settings have changed, recompute
     * the view tranformation.
     */
    void computeView(int w, int h) {
        Matrix4x4 viewTransform;
        double screenDistance = 1.f;
        double m, sin, cos;
        Matrix4x4 rotx, roty, rotz;
                
        if (!newCameraSettings)
            return;
        newCameraSettings = false;

        /*
         * Convert the cameraSubject to cameraDirection.
         */
        if (cameraSubject != null) {
            cameraDirection = cameraSubject.copy().sub(cameraLocation);
        }

        /*
         * Position the camera to the origin
         */
        Matrix4x4 trn =new Matrix4x4().move(cameraLocation.copy().neg());

        /*
         * Rotate the world about the Y axis so the camera 
         * is in the Y-Z plane.
         */
        m = (double) Math.sqrt(cameraDirection.v[0] * cameraDirection.v[0]
                            + cameraDirection.v[2] * cameraDirection.v[2]);
        if (m != 0.f) {
            sin =  cameraDirection.v[0]/m;
            cos = -cameraDirection.v[2]/m;

            roty = new Matrix4x4().rotatey(sin, cos);
        } else
            roty = new Matrix4x4();

        /*
         * Rotate the world about the X axis so the camera 
         * points along the positive -Z axis.
         */
        Vector4 dir = cameraDirection.copy().mul(roty);

        m = (double) Math.sqrt(dir.v[1] * dir.v[1]
                            + dir.v[2] * dir.v[2]);
        if (m != 0.f) {
            sin = -dir.v[1]/m;
            cos = -dir.v[2]/m;

            rotx = new Matrix4x4().rotatex(sin, cos);
        } else
            rotx = new Matrix4x4();

        Matrix4x4 rot  = roty.copy().mul(rotx);

        /*
         * Rotate the camera about the Z axis until the top
         * is pointing in the positive Y direction.
         */
        Vector4 up = cameraUp.copy().mul(rot);

        m = (double) Math.sqrt(up.v[0] * up.v[0]
                            + up.v[1] * up.v[1]);
        if (m != 0.f) {
            sin = up.v[0]/m;
            cos = up.v[1]/m;

            rotz = new Matrix4x4().rotatez(sin, cos);
        } else
            rotz = new Matrix4x4();

        rot.mul(rotz);

/*
        System.out.println("");
        System.out.println("dir");
        cameraDirection.print();

        System.out.println("up");
        cameraUp.print();

        System.out.println("trn:");
        trn.print();

        System.out.println("rox:");
        rotx.print();

        System.out.println("roty:");
        roty.print();         

        System.out.println("rotz:");
        rotz.print();

        System.out.println("");
*/
        viewTransform = trn.copy().mul(rot);
        viewTransformer = new ViewTransformer(viewTransform, w/2, h/2);

        viewTransformer.setScreenDistanceAndHeight(200, 100);
    }
}
