package neurodata;

import neurodata.GL.*;
import awwwesome.visualworld.ViewTransformer;
import java.awt.Graphics;
import awwwesome.visualworld.Point3D;
import awwwesome.visualworld.Canvas3D;
import java.awt.Color;
import java.util.*;

/** CellPoint is the object which actually draws a cell's representation in the Canvas3D object.  If Java3D is being used, CellPoint contains a GLCellPoint as well; the GLCellPoint handles drawing the cell's representation in Java3D.
<p>
CellPoints also display the synapses in the Canvas3D object.  They do this by querying
their respective Cells for synapses, then draw a line for each synapse.
<p>
CellPoints also handle hit-testing as well as drawing.

*/

public class CellPoint extends Point3D
    {
    public static int SHOW_NONE=0;
    public static int SHOW_INCOMING=1;
    public static int SHOW_OUTGOING=2;
    public static int SHOW_GAP_JUNCTIONS=3;

    /** The color I display when I'm <i>not</i> selected. */
    public Color displayColor;
    /** The cell I'm responsible for drawing */
    public Cell cell;
    /** Am I currently displayed, given the current timestamp? */
    public boolean displayed=false;
    /** Should I display my label? */
    public boolean display_labels=false;
    /** Should my synapses display their labels? */
    public boolean display_synapse_labels=false;
    /** Even if I <i>can</i> display myself, should I -- according to the Highlight panel? */
    public boolean display_when_possible=true;
    /** Regarding my synapses: do I SHOW_NONE, SHOW_INCOMING synapses, SHOW_OUTGOING synapses, or SHOW_GAP_JUNCTIONS? */
    public int synapses_to_show=SHOW_NONE;

    public GLCellPoint glcellpoint;




/** How close am I in 2D to the point (x,y) in the Canvas3D? Used to determine who
 was hit */

    public int closenessToHit(int x, int y)
	/* returns -1.0 if not shown, else distance to v squared */
	{
	if (!projection_shown || !display_when_possible || !displayed) return -1;
	else return 
	    (x-projected_point.v[0])*(x-projected_point.v[0])+
	    (y-projected_point.v[1])*(y-projected_point.v[1]);
	}

    public CellPoint(Cell c)
	{
	super("",4);
	cell=c;
	cell.cp=this;
	displayColor=(Colors.cellColor);
	othercolor=(Colors.selectedCellColor);
	color=displayColor;
	}


    /** Sets my display color */
    public void setColor(Color c)
	{
	super.setColor(c);
	if (glcellpoint!=null) glcellpoint.setColor(c);
	}


    /** Selects me or unselects me.  This may require me to add or remove my synapse lines (which are objects in the Canvas3D).  It's not an efficient method. */

    public void setSelected(boolean selected, Canvas3D canvas)
	{
	if (selected)
	    {
	    this.setColor(Colors.selectedCellColor);
	    text=cell.official_name;

	    // first remove synapses
	    Vector tmp = new Vector(300);
	    Enumeration e =canvas.vw.objects.elements();
	    while(e.hasMoreElements())
		{
		Object o=e.nextElement();
		if (o instanceof SynapseLine)
		    tmp.addElement(o);
		}
	    e = tmp.elements();
	    while(e.hasMoreElements())
		{
		/* EXTREMELY inefficient!  O(n^2).  Better would
		   be to make a hash table conversion perhaps. */
		canvas.vw.objects.removeElement(e.nextElement());
		}
	    

	    // now add synapses

	    e = cell.synapses.elements();
	    while(e.hasMoreElements())
		{
		Synapse s=(Synapse)e.nextElement();
		if ((synapses_to_show==SHOW_GAP_JUNCTIONS &&
		    s.type==Synapse.type_gap) ||
		    (synapses_to_show==SHOW_OUTGOING &&
		     s.type==Synapse.type_chemical &&
		     s.from==cell) ||
		    (synapses_to_show==SHOW_INCOMING &&
		     s.type==Synapse.type_chemical &&
		     s.to==cell))
		    {
		    // show that sucker 
		    if (s.from.cp.displayed && s.to.cp.displayed)	
			// you can see both synapses
			{
			SynapseLine sl=new SynapseLine(s);
			canvas.addObject(sl);
			s.from.cp.text=s.from.official_name;
			s.to.cp.text=s.to.official_name;
			if (display_synapse_labels) sl.text=""+s.number;  /* display number of synapses */
			}
		    }
		}
	    }
	else
	    {
	    if (display_labels) text=cell.official_name;
	    else text="";
	    this.setColor(displayColor);
	    }
	}


    /** sets my display color */
    public void setDisplayColor(Color this_color)
	{
	displayColor=this_color;
	if (!color.equals(Colors.selectedCellColor))  /* not selected right now */
	    this.setColor(displayColor);
 }

    /** projects me.  I only actually get projected if I'm supposed to display_when_possible */
    public void project(Graphics g, ViewTransformer viewTransformer) 
	{
	/* override to only project if my synapse contains cellPoints that
	   are also projecting */
	
	if (display_when_possible) super.project(g,viewTransformer);
	}
    }
