package neurodata;

import neurodata.GL.*;
import java.awt.*;
import awwwesome.visualworld.*;
import java.util.*;


/** CubesHelper is a small panel which contains the Time scrollbar and text field; it passes any events on up to Cubes. */ 
class CubesHelper extends Panel
    {
    Cubes c;
    public CubesHelper(Cubes _c) { c=_c; }
    public boolean handleEvent(Event evt) { return c.handleEvent(evt); }
    }


/** Cubes is the main class in this program.  It holds the awwwesome.visualworld Canvas3D, stores the Cells object, and maintains the GLDisplay object (if you're using Java3D).  It
also handles all interface issues, menu options, etc.
 */

public class Cubes extends Frame implements HitInterface, FindInterface, HighlightInterface, DisplaySynapseInterface
    {
    static int MINIMUM_HIT_DISTANCE = 15;// The minimum distance away to still select
    Cells cells;                        // The cells
    CellPoint selected_cell_point=null;	// The currently selected cell point
    Canvas3D canvas;			// The awwwesome.visualworld canvas
    GLDisplay gld;			// The Java3D canvas
    TextField t;			// The time field
    Scrollbar scroll_t;			// The time scrollbar
    Highlight highlight;		// The highlight dialog box
    FindPanel findpanel;		// The find dialog box
    DisplaySynapses displaysynapses;	// The display symapses dialog box
    Remarks remarks;			// The remarks dialog box
    int currentTimeStamp=-1;		// The current displayed time






    /** 
      Part of the HighlightInterface, and is called by the
      Highlight panel. This rather long method sets
      the display characteristics of all the cells in the system according
      to the highlight option determined in the interface.  There are a
      lot of options, but to sum up:
      
      <ul><li>
      cells may be highlighted (turned pink) if they are:
      <ul><li>neurons
      <li>descendents of some particular cell,
      <li>members of some expression pattern, cell fate, or cell group.
      </ul>

      <li>nonhighlighted cells may either be shown (in blue) or not shown at all.
      
      <li>highlighted cells may be displayed with their labels (by default, this
      doesn't happen)
      
      <li>the cells Z2 and Z3 may or may not be shown (they have volume problems
      due to a lack of germ-line cell information, so it's often nice to not
      display them).
      </ul>

      */
	
    public void setDisplayInfo(String result,int highlight_option,
			       boolean display_others,boolean display_label, 
			       boolean display_zcells)
	{
	Hashtable descendents=null;
	Fate fate=null;
	if (highlight_option==Highlight.HIGHLIGHT_DESCENDENTS)
	    {
	    descendents = new Hashtable(1000);
	    /* gather descendents */
	    Cell root = cells.fetchCell(result);  /* result better be valid!  
						  otherwise we're adding new items by accident */
	    root.descendentsOf(root,descendents);  // add those descendants!
	    }
	else if (highlight_option==Highlight.HIGHLIGHT_FATES)
	    {
	    fate=cells.fetchFate(result);
	    }
	Enumeration e = cells.cell_dictionary.elements();
	while(e.hasMoreElements())
	    {
	    Cell cell = (Cell)e.nextElement();
	    if (highlight_option==Highlight.HIGHLIGHT_NEURONS)
		{
		if (cell.synapses.size()>0)  // we assume this is a neuron
		    { 
		    cell.cp.setDisplayColor(Colors.alternateCellColor);
		    cell.cp.display_when_possible=true;
		    cell.cp.display_labels=display_label;
		    }
		else 
		    {
		    cell.cp.setDisplayColor(Colors.cellColor);
		    cell.cp.display_when_possible=display_others;
		    cell.cp.display_labels=false;
		    }		  
		}
	    else if (highlight_option==Highlight.HIGHLIGHT_DESCENDENTS)
		{
		if (descendents.containsKey(cell.official_name))  // I assume strings aren't considered address-unique!
		    {
		    cell.cp.setDisplayColor(Colors.alternateCellColor);
		    cell.cp.display_when_possible=true; 
		    cell.cp.display_labels=display_label;
		    }
		else 
		    {
		    cell.cp.setDisplayColor(Colors.cellColor);
		    cell.cp.display_when_possible=display_others;
		    cell.cp.display_labels=false;
		    }		  
		}
	    else if (highlight_option==Highlight.HIGHLIGHT_GROUPS)
		{
		if (cell.group==null)
		    {
		    cell.cp.setDisplayColor(Colors.cellColor);
		    cell.cp.display_when_possible=display_others;
		    cell.cp.display_labels=false;
		    }
		else if (cell.group.name.equals(result))  /* display that sucker */
		    {
		    cell.cp.setDisplayColor(Colors.alternateCellColor);
		    cell.cp.display_when_possible=true;  
		    cell.cp.display_labels=display_label;
		    }
		else 
		    {
		    cell.cp.setDisplayColor(Colors.cellColor);
		    cell.cp.display_when_possible=display_others;
		    cell.cp.display_labels=false;
		    }
		}
	    else if (highlight_option==Highlight.HIGHLIGHT_PATTERNS)
		{
		if (cell.pattern==null)
		    {
		    cell.cp.setDisplayColor(Colors.cellColor);
		    cell.cp.display_when_possible=display_others;
		    cell.cp.display_labels=false;
		    }
		else if (cell.pattern.name!=null && cell.pattern.name.equals(result))  // display that sucker
		    {
		    cell.cp.setDisplayColor(Colors.alternateCellColor); 
		    cell.cp.display_when_possible=true;  
		    cell.cp.display_labels=display_label;
		    }
		else
		    {
		    cell.cp.setDisplayColor(Colors.cellColor);
		    cell.cp.display_when_possible=display_others;
		    cell.cp.display_labels=false;
		    }
		}
	    else if (highlight_option==Highlight.HIGHLIGHT_FATES)
		{
		if (cell.fate.contains(fate))  /* display that sucker */
		    {
		    cell.cp.setDisplayColor(Colors.alternateCellColor);
		    cell.cp.display_when_possible=true;  
		    cell.cp.display_labels=display_label;
		    }
		else 
		    {
		    cell.cp.setDisplayColor(Colors.cellColor);
		    cell.cp.display_when_possible=display_others;
		    cell.cp.display_labels=false;
		    }
		}
	    else		// Highlight.HIGHLIGHT_NOTHING
		{
		/* Default */
		cell.cp.setDisplayColor(Colors.cellColor);
		cell.cp.display_when_possible=true; 
		cell.cp.display_labels=false;
		}

	    // am I Z2 or Z3?  Then I need to be handled specially
	    if (!display_zcells) 
		cell.cp.display_when_possible =
		    cell.cp.display_when_possible && 
			!cell.official_name.equals("Z2") && 
			    !cell.official_name.equals("Z3");
	    }
	
	/* We may need to deal with the neurons issue if we deselect a cell
	   or change the feature to display some other kind of synapse here. 
	   But for the moment, this doesn't come up as an issue, so we can
	   just redraw(). 
	   */
	/* Reselect cell point if any -- this sets it to black again and re-loads synapses */
	
	if (selected_cell_point!=null) select(selected_cell_point);
	canvas.redraw();
        gld.formScene(currentTimeStamp,cells,this);
	}

    




    /** This method is part of the DisplaySynapse interface, and is called by the
      Synapse panel.  It sets the synapse display information.  You shouldn't
      call this yourself.  Basically, the options are:
      <ul><li>
      Display incoming synapses, outgoing synapses, or gap junctions.
      <li>
      Display the number of synapses on the synapse line.
      </ul>
     */

    public void setSynapseInfo(int synapse_option, boolean display_label)
	{
	Enumeration e = cells.cell_dictionary.elements();
	while(e.hasMoreElements())
	    {
	    Cell cell = (Cell)e.nextElement();
	    cell.cp.display_synapse_labels=display_label;
	    cell.cp.synapses_to_show=synapse_option;
	    }
	/* Reselect cell point if any -- this sets it to black again and re-loads synapses */
	if (selected_cell_point!=null) select(selected_cell_point);
	canvas.redraw();
	}


    /** This method is part of the FindInterface interface, and is called by
      the Find panel.  It searches for a particular cell, and selects it if
      it can find it.  If it`s not currently being displayed, this method
      returns a string indicating failure, telling the user where the cell
      is located in time (which the Find panel then displayes for the user). */

    public String doFind(String find_this)
	{
	Cell cell=cells.fetchCell(find_this);
	if (cell.cp.displayed)
	    {
	    /* We can display it */
	    select(cell.cp);
	    return null;
	    }
	else
	    {
	    return ("Can't find " + find_this + ". Try " + cell.birthday + 
		    " <= time < " + cell.death_day + ".");
	    }
	}



    /** The big enchalada.  The Cells object is told to add cells to the canvas, the
      coordinate system is added, and (if appropriate) a GL scene of the cells is formed.
      */

    public  void drawCells(int timestamp)
	{
	if (timestamp >= -1 && timestamp <= Cell.maximum_death_day)
	    {
	currentTimeStamp=timestamp;
	cells.addCellsToCanvas(timestamp,canvas);

	// Add coordinate system.  See Cells.java for an explanation of 60,40,40.
	Cube cube = new Cube(60,40,40);
	cube.setColor(Colors.coordinatesColor);
        cube.setOtherColor(Colors.coordinatesColor);
        canvas.addObject(cube);

	gld.formScene(timestamp,cells,this);
	}
	/* Reselect cell point if any -- this sets it to black again and re-loads synapses */
	if (selected_cell_point!=null) select(selected_cell_point);
	canvas.updateCamera();
	canvas.redraw();
      	}
    


    /** Quits the program. */

    public boolean doWindowClose()
	{
	System.exit(0);
	return true;		// Will this ever be reached?
	}


    /** Deselects any currently selected cell. */
   
    public void deselect()
	{
	// deselect everyone
        Enumeration e =canvas.vw.objects.elements();
        while(e.hasMoreElements())
            {
            Object o=e.nextElement();
            if (o instanceof CellPoint)
                {
                CellPoint cp = (CellPoint) o;
                cp.setSelected(false,canvas);
                }
            }
	selected_cell_point=null;
	// and redraw.
        canvas.redraw();

        // load remarks
        remarks.setRemarks("");
	}
 


    /** Selects the cell owning cell_point, deselecting any others. */

    public void select(CellPoint cell_point)
	{
	// deselect everyone
	Enumeration e =canvas.vw.objects.elements();
	while(e.hasMoreElements())
	    {
	    Object o=e.nextElement();
	    if (o instanceof CellPoint)
		{
		CellPoint cp = (CellPoint) o;
		cp.setSelected(false,canvas);
		}
	    }
	
	// finally, select cell
	cell_point.setSelected(true,canvas);	// this apparently can be done even
					// the cell's currently not being
					// displayed.  Cool!
	selected_cell_point=cell_point;
	// and redraw.
	canvas.redraw();

	// load remarks
	String string="";
	for(int x=0;x<cell_point.cell.others.size();x++)
	    string = string + (String)cell_point.cell.others.elementAt(x) + "\n";
	remarks.setRemarks(string);
	}


    /** Handles mouse events, namely hit testing for selection of cells. */

    public void doMouseEvent(int x, int y)
	{
	CellPoint best = null;
	int best_distance=-1;
	int current_distance;

	Enumeration cells=canvas.vw.objects.elements();
	while(cells.hasMoreElements())
	    {
	    Object o=cells.nextElement();
	    if (o instanceof CellPoint)
		{
		CellPoint p = (CellPoint) o;
		current_distance=p.closenessToHit(x,y);
		if (current_distance != -1 && 
		    ((current_distance < best_distance) || best_distance==-1))
		    {  best_distance=current_distance ; best=p; }
		}
	    }
	if (best!=null && best_distance <= MINIMUM_HIT_DISTANCE * MINIMUM_HIT_DISTANCE) select(best);
	else deselect();
	}
	


    /** Starts up the program. */

    public static void main(String[] argv)
	{
	Cubes cubes=new Cubes("Java C. elegans");
	cubes.setup();
	}


    /** Handles menu events and return-key presses in the Time text field. */

    public boolean action(Event evt, Object what)
	{
	/* Handle menu events */
	if (evt.target instanceof MenuItem &&
	    "Find Cell".equals(what))
	    {
	    if (findpanel==null)
		findpanel=new FindPanel(cells.cell_dictionary.elements(),this);
	    findpanel.show();
	    return true;
	    }
	else if (evt.target instanceof MenuItem &&
	    "Highlight Options".equals(what))
	    {
	    if (highlight==null) 
		highlight=new Highlight(cells.cell_dictionary.elements(),cells.group_dictionary.elements(),
			   cells.pattern_dictionary.elements(),cells.fate_dictionary.elements(),this);
	    highlight.show();
	    return true;
	    }
	else if (evt.target instanceof MenuItem &&
	    "Synapse Options".equals(what))
	    {
	    if (displaysynapses==null) 
		displaysynapses=new DisplaySynapses(this);
	    displaysynapses.show();
	    return true;
	    }
	else if (evt.target instanceof MenuItem &&
	    "Remarks".equals(what))
	    {
	    remarks.show();
	    return true;
	    }
	else if (evt.target instanceof MenuItem && 
	    "Reset View".equals(what))
	    {
	    canvas.initCamera();
	    canvas.updateCamera();
	    canvas.redraw();
		return true;
	    }
	/* Handle RETURN in time area */
	else if (evt.target==t)
	    {
	    drawCells(Integer.parseInt(t.getText()));
	    scroll_t.setValue(Integer.parseInt(t.getText()));
	    return true;
	    }
	else return false;
	}

    

    /** Handles window-close and Time scrollbar events */

    public boolean handleEvent(Event evt)
	{
	/* Handle Window-Close Events*/
	 if (evt.id==Event.WINDOW_DESTROY)
	    {
	    /* User wants to close the window, possibly quit the program */
	    System.exit(0);
	    return true;  // probably never reaches here
	    }
	 else if (evt.target==scroll_t &&
		  (evt.id==Event.SCROLL_ABSOLUTE ||
		  evt.id==Event.SCROLL_LINE_DOWN ||
		  evt.id==Event.SCROLL_LINE_UP ||
		  evt.id==Event.SCROLL_PAGE_DOWN ||
		  evt.id==Event.SCROLL_PAGE_UP))
	     {
	     drawCells(scroll_t.getValue());
	     t.setText(String.valueOf(scroll_t.getValue()));
	     return true;
	     }
	else return super.handleEvent(evt);
	}


    public Cubes()
	{
	remarks = new Remarks();
	}

    public Cubes(String val)
	{
	super(val);
	remarks = new Remarks();
	}

    
    /** Sets up the Cubes object.  Much of this can't be done
      in the initializer because other Java objects haven't been
      initialized yet.  So in main(), Cubes is first initialized,
      then setup() is called. */

    public void setup()
	{
	cells=new Cells();
	cells.readCells(System.in);
	cells.postProcess();
	
	CubesHelper ch = new CubesHelper(this);
	ch.setLayout(new GridLayout(1,2));
	t = new TextField("-1");
	t.setEditable(true);
	ch.add(t);
	scroll_t = new Scrollbar(Scrollbar.HORIZONTAL,-1,1,-1,999);
	ch.add(scroll_t);

	canvas = new Canvas3D();
	

	gld=GLDisplay.newGLDisplay();
	if (gld.supportingGL)  // we're doing Java3D, not just a stub here
	    {
	    Frame frame = new Frame("Java3D C. elegans");
	    frame.setSize(700, 700);
	    frame.add("Center", gld);
	    frame.setVisible(true);
	    }

	canvas.vw.setBgColor(Colors.backgroundColor);
	canvas.setHitTarget(this);
	setLayout(new BorderLayout());
	add("Center",canvas);
	add("South",ch);
	MenuBar mb=new MenuBar();
	Menu visualizer = new Menu("Visualizer");
	mb.add(visualizer);
	MenuItem findCell=new MenuItem("Find Cell");
	visualizer.add(findCell);
	MenuItem highlightOptions = new MenuItem("Highlight Options");
	visualizer.add(highlightOptions);
	MenuItem synapsestuff = new MenuItem("Synapse Options");
	visualizer.add(synapsestuff);
	MenuItem remarkstuff = new MenuItem("Remarks");
	visualizer.add(remarkstuff);
	MenuItem resetstuff = new MenuItem("Reset View");
	visualizer.add(resetstuff);
	setMenuBar(mb);
	
	pack();
	drawCells(-1);   /* Initial setup */
	canvas.updateCamera();
	resize(500,500);
	show();
	}
    }
    
