package ec.es;
import ec.*;
import ec.simple.*;
import java.io.*;
import ec.util.*;

/* 
 * ESEvolutionState.java
 * 
 * Created: Thu Sep  7 17:27:47 2000
 * By: Sean Luke
 */

/**
 *
 * ESEvolutionState is an EvolutionState which accepts the (mu,lambda)
 * and (mu+lambda) breeding strategies found in the ec.es.MuPlusLambdaBreeder
 * and ec.es.MuCommaLambdaBreeder breeders, to do evolution-strategies style
 * evolution.
 *
 * <p>Evolution strategies breeders require a "mu" parameter and a "lambda"
 * parameter for each subpopulation.  "mu" refers to the number of parents
 * from which the new population will be built.  "lambda" refers to the
 * number of children generated by the mu parents.  Subpopulation sizes
 * will change as necessary to accommodate this fact in later generations.
 * The only rule for initial subpopulation sizes is that they must be
 * greater than or equal to the mu parameter for that subpopulation.
 * 
 * <p>Earlier versions of ECJ specified that the initial subpopulation size
 * would be lambda.  This is no longer the case, as it does not jibe with
 * the traditional (and IMHO, odd) ES approach of having the initial
 * subpopulation size be *mu*, of all things.  You can now set your initial subpopulation
 * size to whatever you like, totally independent of lambda and mu, as long as it is
 * &gt;= mu.
 *
 * <p>ESEvolutionState stores mu and lambda values for each subpopulation
 * in the population, as well as comparisons.  A comparison tells you
 * if &gt;1/5, &lt;1/5 or =1/5 of the new population was better than its
 * parents (the so-called evolution strategies "one-fifth rule".
 * Although the comparisons are gathered, no mutation objects are provided
 * which actually <i>use</i> them -- you're free to use them in any mutation
 * objects you care to devise which requires them.
 *
 * <p>To do evolution strategies evolution, you need to use an ESEvolutionState
 * object, either MuCommaLambdaBreeder or MuPlusLambdaBreeder, and the
 * breeding pipelines must contain <b>exactly</b> one ESSelection object called
 * each time an individual is generated.   For example, if you're just 
 * generating children by mutating a single selected individual into a child,
 * then you use the ESSelection object to pick that individual.  If you're
 * generating two children at a time by selecting two parents and crossing
 * them over, then each parent should be selected with ESSelection (and
 * in this case, you had better have a population size that's an even number!)
 * If you're generating one child at a time by selecting two parents and
 * crossing them over, then throwing away one of the children and mutating
 * the other, then you should have only <b>one</b> parent chosen through
 * ESSelection; the other might be chosen with Tournament Selection, say.
 *

 <p><b>Parameters</b><br>
 <table>
 <tr><td valign=top>breed.lambda.<i>subpop-num</i><br>
 <font size=-1>int >= 0</font></td><td>Specifies the 'lambda' parameter for the subpopulation.</td>
 </tr>
 <tr><td valign=top>breed.mu.<i>subpop-num</i><br>
 <font size=-1>int:  a multiple of "lambda"</font></td><td>Specifies the 'mu' parameter for the subpopulation.</td>
 </tr>
 </table>



 * @author Sean Luke
 * @version 1.0 
 */

public class ESEvolutionState extends SimpleEvolutionState
    {
    public static final String P_MU = "mu";
    public static final String P_LAMBDA = "lambda";

    public int[] mu;
    public int[] lambda;

    public byte[] comparison; 
    public static final byte C_OVER_ONE_FIFTH_BETTER = 1;
    public static final byte C_UNDER_ONE_FIFTH_BETTER = -1;
    public static final byte C_EXACTLY_ONE_FIFTH_BETTER = 0;
   
    /** Modified by multiple threads, don't fool with this */
    public int[] count;

    public void setup(final EvolutionState state, final Parameter base)
	{
	super.setup(state,base);

	// double check that we have valid breeders
	if (!(breeder instanceof ESBreederForm))
	    state.output.error("You've chosen to use Evolutionary Strategies-style Evolution, but your breeder is not in ESBreederForm.",base);
	Parameter p;
	
	// we're not using the base

	p = new Parameter(Initializer.P_POP).push(Population.P_SIZE);
	int size = state.parameters.getInt(p,null,1);  // if size is wrong, we'll let Population complain about it -- for us, we'll just make 0-sized arrays and drop out.
	
	mu = new int[size];
	lambda = new int[size];
	comparison = new byte[size];
	
	// load mu and lambda data
	for(int x=0;x<size;x++)
	    {
	    lambda[x] = state.parameters.getInt(ESDefaults.base().push(P_LAMBDA).push(""+x),null,1);            
	    if (lambda[x]==0) state.output.error("lambda must be an integer >= 1",ESDefaults.base().push(P_LAMBDA).push(""+x));
	    mu[x] = state.parameters.getInt(ESDefaults.base().push(P_MU).push(""+x),null,1);            
	    if (mu[x]==0) state.output.error("mu must be an integer >= 1",ESDefaults.base().push(P_MU).push(""+x));
            else if ((lambda[x] / mu[x]) * mu[x] != lambda[x]) // note integer division
                state.output.error("mu must be a multiple of lambda", ESDefaults.base().push(P_MU).push(""+x));
            }
	state.output.exitIfErrors();
	}


    public void run(int condition) throws IOException
	{

	if (condition == C_STARTED_FRESH)
	    {
	    output.message("Setting up");
	    setup(this,null);  // a garbage Parameter

	    output.message("Initializing Generation 0");
	    statistics.preInitializationStatistics(this);
	    population = initializer.initialPopulation(this);
            exchanger.initializeContacts(this);

	    // At this point we need to do load our population info
	    // and make sure it jibes with our mu info

	    // the first issue is: is the number of subpopulations
	    // equal to the number of mu's?

	    if (mu.length!=population.subpops.length) // uh oh
		output.fatal("For some reason the number of subpops is different than was specified in the file (conflicting with Mu and Lambda storage).",null);

	    // next, load our population, make sure there are no subpopulations smaller than the mu's
	    for(int x=0;x<population.subpops.length;x++)
		{
                if (population.subpops[0].individuals.length < mu[x])
                    output.error("Subpopulation " + x + " must be a multiple of the equivalent mu (that is, "+ mu[x]+").");
		}
	    output.exitIfErrors();

	    // okay, our mus and lambdas are all set.
	    statistics.postInitializationStatistics(this);
	    }
	else // condition == C_STARTED_FROM_CHECKPOINT
	    {
	    // the last thing we didn't do
	    generation++;
	    output.message("Generation" + generation);
	    }

	/* the big loop */
	
	int result = R_SUCCESS;
	while ( true )
	    {
	    //output.message("Evaluate");
	    statistics.preEvaluationStatistics(this);
	    evaluator.evaluatePopulation(this);
	    statistics.postEvaluationStatistics(this);
	    if (evaluator.runComplete(this) && quitOnRunComplete)
		{
		output.message("Found Ideal Individual");
		break;
		}

	    if (generation == numGenerations-1)
		{
		result = R_FAILURE;
		break;
		}

	    //output.message("Pre-Exchange");
	    statistics.prePreBreedingExchangeStatistics(this);
	    population = exchanger.preBreedingExchangePopulation(this);
	    statistics.postPreBreedingExchangeStatistics(this);

	    String exchangerWantsToShutdown = exchanger.runComplete(this);
	    if (exchangerWantsToShutdown!=null)
		{ output.message(exchangerWantsToShutdown); break; }

	    statistics.preBreedingStatistics(this);
	    // We may perform garbage collection immediately before breeding
	    // in the hopes that it makes nice large VM spaces for the new
	    // individuals to fit into.
	    if (gc && generation%gcModulo + 1 == gcModulo)
		{
		if (aggressivegc) aggressiveGC();
		else gc();
		}
	    
	    //output.message("Breed");
	    try { population = breeder.breedPopulation(this);}
	    catch (CloneNotSupportedException e) 
		{ throw new InternalError(); } // never happens
	    
	    //output.message("Post-Exchange");
	    statistics.postBreedingStatistics(this);
	    statistics.prePostBreedingExchangeStatistics(this);
	    population = exchanger.postBreedingExchangePopulation(this);
	    statistics.postPostBreedingExchangeStatistics(this);

	    if (checkpoint && generation%checkpointModulo +1 == checkpointModulo) 
		{
		output.message("Checkpointing");
		statistics.preCheckpointStatistics(this);
		Checkpoint.setCheckpoint(this);
		statistics.postCheckpointStatistics(this);
		}

	    generation++;
	    output.message("Generation " + generation);
	    }

	//Output.message("Finishing");
	/* finish up -- we completed. */
	statistics.finalStatistics(this,result);
	finisher.finishPopulation(this,result);
	exchanger.closeContacts(this,result);
	}



    
    public void go()
	{
	if (debugState==DEBUG_INIT)
	    {
	    output.message("" + (debugNum++) + ") DEBUG: INIT" + generation);
	    
	    output.message("Setting up");
	    setup(this,null);  // a garbage Parameter
	    
	    output.message("Initializing Generation 0");
	    statistics.preInitializationStatistics(this);
	    population = initializer.initialPopulation(this);
            exchanger.initializeContacts(this);

	    // At this point we need to do load our population info
	    // and make sure it jibes with our mu info

	    // the first issue is: is the number of subpopulations
	    // equal to the number of mu's?

	    if (mu.length!=population.subpops.length) // uh oh
		output.fatal("For some reason the number of subpops is different than was specified in the file (conflicting with Mu and Lambda storage).",null);

	    // next, load our population, make sure there are no subpopulations smaller than the mu's
	    for(int x=0;x<population.subpops.length;x++)
		{
                if (population.subpops[0].individuals.length < mu[x])
                    output.error("Subpopulation " + x + " must be a multiple of the equivalent mu (that is, "+ mu[x]+").");
		}
	    output.exitIfErrors();

	    // okay, our mus and lambdas are all set.

	    statistics.postInitializationStatistics(this);
	    debugState=DEBUG_EVAL;
	    }
	else if (debugState==DEBUG_EVAL)
	    {
	    output.message("" + (debugNum++) + ") DEBUG: EVAL, Generation " + generation);
	    statistics.preEvaluationStatistics(this);
	    evaluator.evaluatePopulation(this);
	    statistics.postEvaluationStatistics(this);
	    if (evaluator.runComplete(this) && quitOnRunComplete)
		{
		output.message("Found Ideal Individual");
		debugState=DEBUG_FINAL_SUCCESS;
		return;
		}
	    if (generation == numGenerations-1)
		{
		debugState=DEBUG_FINAL_FAILURE;
		return;
		}
	    debugState=DEBUG_BREED;
	    }
	else if (debugState==DEBUG_BREED)
	    {
	    output.message("" + (debugNum++) + ") DEBUG: BREED, Generation " + generation);
	    statistics.prePreBreedingExchangeStatistics(this);
	    population = exchanger.preBreedingExchangePopulation(this);
	    statistics.postPreBreedingExchangeStatistics(this);

	    String exchangerWantsToShutdown = exchanger.runComplete(this);
	    if (exchangerWantsToShutdown!=null)
		{ 
		output.message(exchangerWantsToShutdown);
		debugState=DEBUG_FINAL_FAILURE;
		return;
		}

	    statistics.preBreedingStatistics(this);
	    // We may perform garbage collection immediately before breeding
	    // in the hopes that it makes nice large VM spaces for the new
	    // individuals to fit into.
	    if (gc && generation%gcModulo + 1 == gcModulo)
		{
		if (aggressivegc) aggressiveGC();
		else gc();
		}
	    
	    try { population = breeder.breedPopulation(this);}
	    catch (CloneNotSupportedException e) 
		{ throw new InternalError(); } // never happens
	    
	    statistics.postBreedingStatistics(this);
	    statistics.prePostBreedingExchangeStatistics(this);
	    population = exchanger.postBreedingExchangePopulation(this);
	    statistics.postPostBreedingExchangeStatistics(this);

	    if (checkpoint && generation%checkpointModulo +1 == checkpointModulo) 
		{
		output.message("Checkpointing");
		statistics.preCheckpointStatistics(this);
		Checkpoint.setCheckpoint(this);
		statistics.postCheckpointStatistics(this);
		}

	    generation++;
	    output.message("Generation " + generation);
	    debugState=DEBUG_EVAL;
	    }
	else if (debugState==DEBUG_FINAL_SUCCESS || 
		 debugState==DEBUG_FINAL_FAILURE)
	    {
	    output.message("" + (debugNum++) + ") DEBUG: FINAL.  Cleaning up.");
	    int result = (debugState==DEBUG_FINAL_SUCCESS ?
			  R_SUCCESS : R_FAILURE);
	    statistics.finalStatistics(this,result);
	    finisher.finishPopulation(this,result);
	    exchanger.closeContacts(this,result);
	    output.flush();
	    debugState=DEBUG_END;
	    }
	else // debugState==DEBUG_END
	    {
	    output.message((debugNum++) + ") DEBUG: END.  Nothing more to do.");
	    }
	}
    }
