package ec;
import ec.util.*;
import java.io.IOException;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.OptionalDataException;
import java.io.PrintWriter;

/* 
 * Evolve.java
 * 
 * Created: Wed Aug 11 17:49:01 1999
 * By: Sean Luke
 */

/**
 * Evolve is the main entry class for an evolutionary computation run.
 *
 * <p> An EC run is done with one of two argument formats:
 *
 * <p><tt>java ec.Evolve -file </tt><i>parameter_file [</i><tt>-p </tt><i>parameter=value]*</i>
 *
 * <p>This starts a new evolutionary run, using the parameter file <i>parameter_file</i>.
 * The user can provide optional overriding parameters on the command-line with the <tt>-p</tt> option.
 *
 * <p><tt>java ec.Evolve -checkpoint </tt><i>checkpoint_file</i>
 * 
 * <p>This starts up an evolutionary run from a previous checkpoint file.
 *

 <p><b>Parameters</b><br>
 <table>

 <tr><td valign=top><tt>jobs</tt></br>
 <font size=-1> int >= 1 (default)</font></td>
 <td valign=top>(The number of jobs to iterate.  The current job number (0...jobs-1) will be added to each seed UNLESS the seed is loaded from the system time.  The job number also gets added as a prefix (if the number of jobs is more than 1)).</td></tr>

 <tr><td valign=top><tt>nostore</tt><br>
 <font size=-1> bool = <tt>true</tt> or <tt>false</tt> (default)</font></td>
 <td valign=top>(should the ec.util.Output facility <i>not</i> store announcements in memory?)</td></tr>

 <tr><td valign=top><tt>flush</tt><br>
 <font size=-1> bool = <tt>true</tt> or <tt>false</tt> (default)</font></td>
 <td valign=top>(should I flush all output as soon as it's printed (useful for debugging when an exception occurs))</td></tr>

 <tr><td valign=top><tt>verbosity</tt><br>
 <font size=-1>int &gt;= 0</font></td>
 <td valign=top>(the ec.util.Output object's verbosity)</td></tr>

 <tr><td valign=top><tt>evalthreads</tt><br>
 <font size=-1>int &gt;= 1</font></td>
 <td valign=top>(the number of threads to spawn for evaluation)</td></tr>

 <tr><td valign=top><tt>breedthreads</tt><br>
 <font size=-1>int &gt;= 1</font></td>
 <td valign=top>(the number of threads to spawn for breeding)</td></tr>

 <tr><td valign=top><tt>seed.</tt><i>n</i><br>
 <font size=-1>int != 0, or string  = <tt>time</tt></font></td>
 <td valign=top>(the seed for random number generator #<i>n</i>.  <i>n</i> should range from 0 to Max(evalthreads,breedthreads)-1.  If value is <tt>time</tt>, then the seed is based on the system clock plus <i>n</i>.)</td></tr>

 <tr><td valign=top><tt>state</tt><br>
 <font size=-1>classname, inherits and != ec.EvolutionState</font></td>
 <td valign=top>(the EvolutionState object class)</td></tr>

 <tr><td valign=top><tt>print-accessed-params</tt><br>
 <font size=-1>bool = <tt>true</tt> or <tt>false</tt> (default)</td>
 <td valign=top>(at the end of a run, do we print out a list of all the parameters requested during the run?)</td></tr>

 <tr><td valign=top><tt>print-used-params</tt><br>
 <font size=-1>bool = <tt>true</tt> or <tt>false</tt> (default)</td>
 <td valign=top>(at the end of a run, do we print out a list of all the parameters actually <i>used</i> during the run?)</td></tr>

 <tr><td valign=top><tt>print-unaccessed-params</tt><br>
 <font size=-1>bool = <tt>true</tt> or <tt>false</tt> (default)</td>
 <td valign=top>(at the end of a run, do we print out a list of all the parameters NOT requested during the run?)</td></tr>

 <tr><td valign=top><tt>print-unused-params</tt><br>
 <font size=-1>bool = <tt>true</tt> or <tt>false</tt> (default)</td>
 <td valign=top>(at the end of a run, do we print out a list of all the parameters NOT actually used during the run?)</td></tr>

 <tr><td valign=top><tt>print-all-params</tt><br>
 <font size=-1>bool = <tt>true</tt> or <tt>false</tt> (default)</td>
 <td valign=top>(at the end of a run, do we print out a list of all the parameters stored in the parameter database?)</td></tr>

 </table>
 * 
 *
 * @author Sean Luke
 * @version 1.0 
 */

public class Evolve 
    {
    public final static String P_PRINTACCESSEDPARAMETERS = "print-accessed-params";
    public final static String P_PRINTUSEDPARAMETERS = "print-used-params";
    public final static String P_PRINTALLPARAMETERS = "print-all-params";
    public final static String P_PRINTUNUSEDPARAMETERS = "print-unused-params";
    public final static String P_PRINTUNACCESSEDPARAMETERS = "print-unaccessed-params";

    /** The argument indicating that we're starting up from a checkpoint file. */
    public static final String A_CHECKPOINT = "-checkpoint";
    
    /** The argument indicating that we're starting fresh from a new parameter file. */
    public static final String A_FILE = "-file";

    /** flush announcements parameter */
    public static final String P_FLUSH = "flush";

    /** nostore parameter */
    public static final String P_STORE = "store";

    /** verbosity parameter */
    public static final String P_VERBOSITY = "verbosity";

    /** evalthreads parameter */
    public static final String P_EVALTHREADS = "evalthreads";

    /** breedthreads parameter */
    public static final String P_BREEDTHREADS = "breedthreads";

    /** seed parameter */
    public static final String P_SEED = "seed";

    /** 'time' seed parameter value */
    public static final String V_SEED_TIME = "time";

    /** state parameter */
    public static final String P_STATE = "state";


    /** Restores an EvolutionState from checkpoint if "-checkpoint FILENAME" is in the command-line arguments. */
    public static EvolutionState possiblyRestoreFromCheckpoint(String[] args)
        {
        for(int x=0;x<args.length-1;x++)
            if (args[x].equals(A_CHECKPOINT))
                {
                System.err.println("Restoring from Checkpoint " + args[x+1]);
                try
                    {
                    return Checkpoint.restoreFromCheckpoint(args[x+1]);
                    }
                catch(Exception e)
                    {
                    Output.initialError("An exception was generated upon starting up from a checkpoint.\nHere it is:\n" + e);
                    }
                }
        return null;  // should never happen
        }
                
    /** Loads a ParameterDatabase from checkpoint if "-params" is in the command-line arguments. */
    public static ParameterDatabase loadParameterDatabase(String[] args) 
        {
        ParameterDatabase parameters = null;
        for(int x=0;x<args.length-1;x++)
            if (args[x].equals(A_FILE))
                try
                    {
                    parameters = new ParameterDatabase(
                        new File(new File(args[x+1]).getAbsolutePath()),
                        args);
                    break;
                    }
                catch(Exception e)
                    {
                    Output.initialError(
                        "An exception was generated upon reading the parameter file \"" +
                        args[x+1] + "\".\nHere it is:\n" + e); 
                    }
        if (parameters==null)
            Output.initialError("No parameter file was specified." );
        return parameters;
        }

    /** Initializes an evolutionary run given the parameters and a random seed adjustment (added to each random seed).
        The adjustment offers a convenient way to change the seeds of the random number generators each time you
        do a new evolutionary run.  You are of course welcome to replace the random number generators after initialize(...)
        but before startFresh(...) */
                
    public static EvolutionState initialize(ParameterDatabase parameters, int randomSeedAdjustment)
        {
        EvolutionState state=null;
        Output output;
        MersenneTwisterFast[] random;
        int[] seeds;
        int breedthreads = 1;
        int evalthreads = 1;
        int verbosity;
        boolean store;
        int x;

        // 1. create the output
        store = parameters.getBoolean(new Parameter(P_STORE),null,false);

        verbosity = parameters.getInt(new Parameter(P_VERBOSITY),null,0);
        if (verbosity<0)
            Output.initialError("Verbosity should be an integer >= 0.\n",
                                new Parameter(P_VERBOSITY)); 

        output = new Output(store,verbosity);
        output.setFlush(
            parameters.getBoolean(new Parameter(P_FLUSH),null,false));


        // stdout is always log #0.  stderr is always log #1.
        // stderr accepts announcements, and both are fully verbose 
        // by default.
        output.addLog(ec.util.Log.D_STDOUT,Output.V_VERBOSE,false);
        output.addLog(ec.util.Log.D_STDERR,Output.V_VERBOSE,true);

                
        // 2. set up thread values

        breedthreads = parameters.getInt(
            new Parameter(P_BREEDTHREADS),null,1);

        if (breedthreads < 1)
            Output.initialError("Number of breeding threads should be an integer >0.",
                                new Parameter(P_BREEDTHREADS));


        evalthreads = parameters.getInt(
            new Parameter(P_EVALTHREADS),null,1);

        if (evalthreads < 1)
            Output.initialError("Number of eval threads should be an integer >0.",
                                new Parameter(P_EVALTHREADS));



        // 3. create the Mersenne Twister random number generators,
        // one per thread

        random = new MersenneTwisterFast[breedthreads > evalthreads ? 
                                         breedthreads : evalthreads];
        seeds = new int[breedthreads > evalthreads ? 
                        breedthreads : evalthreads];
                                                
//  another way to do it?
//            long ltime = System.currentTimeMillis();
//            int time = (int)(ltime >>> 32) ^ (int)(ltime & 0x00000000FFFFFFFFL);  // fold on top of one another
        int time = (int)System.currentTimeMillis();  // safe because we're getting low-order bits 
        String seed_message = "Seed: ";
        for (x=0;x<random.length;x++)
            {
            int seed = 1;  // have to initialize to make the compiler happy
            String tmp_s = parameters.getString(
                new Parameter(P_SEED).push(""+x),null);
            if (tmp_s==null) // uh oh
                {
                Output.initialError("Seed should be an integer.",
                                    new Parameter(P_SEED).push(""+x));
                                
                }
            else if (tmp_s.equalsIgnoreCase(V_SEED_TIME))
                {
                seed = time++;  // no adjustment
                if (seed==0)
                    Output.initialError("Whoa! This Java version is returning 0 for System.currentTimeMillis(), which ain't right.  This means you can't use '"+V_SEED_TIME+"' as a seed ",new Parameter(P_SEED).push(""+x));
                else seed_message = seed_message + seed + " ";
                }
            else
                {
                try
                    {
                    seed = parameters.getInt(new Parameter(P_SEED).push(""+x),null) + randomSeedAdjustment;
                    }
                catch (NumberFormatException e)
                    {
                    Output.initialError("Invalid Seed Value (must be an integer):\n" + e);
                    }
                seed_message = seed_message + seed + " ";
                }
                                
            seeds[x] = seed ;
            }

        for (x=0;x<random.length;x++)
            {
            for (int y=x+1;y<random.length;y++)
                if (seeds[x]==seeds[y])
                    {
                    Output.initialError(P_SEED+"."+x+" ("+seeds[x]+") and "+P_SEED+"."+y+" ("+seeds[y]+") ought not be the same seed."); 
                    }
            random[x] = new MersenneTwisterFast(seeds[x]);
            }

        // 4.  Start up the evolution
                
        // what evolution state to use?
        state = (EvolutionState)
            parameters.getInstanceForParameter(new Parameter(P_STATE),null,
                                               EvolutionState.class);
        state.parameters = parameters;
        state.random = random;
        state.output = output;
        state.evalthreads = evalthreads;
        state.breedthreads = breedthreads;

        output.systemMessage(Version.message());
        output.systemMessage("Threads:  breed/" + breedthreads + " eval/" + evalthreads);
        output.systemMessage(seed_message);
                
        return state;
        }
                
                
    /** Begins a fresh evolutionary run with a given state.  The state should have been
        provided by initialize(...).  The jobPrefix is added to the front of output and
        checkpoint filenames.  If it's null, nothing is added to the front.  */
        
    public static void startFresh(EvolutionState state, String jobFilePrefix)
        {
        ParameterDatabase parameters = state.parameters;
                
        if (jobFilePrefix != null) 
            state.output.setFilePrefix(jobFilePrefix);
                
        // setup and call the initializer
        state.startFresh();
                
        // at this point the checkpointPrefix has been specified.  We tack onto it:
        if (jobFilePrefix!=null)
            state.checkpointPrefix = jobFilePrefix + state.checkpointPrefix;
                
        int result = EvolutionState.R_NOTDONE;
        while ( result == EvolutionState.R_NOTDONE )
            {
            result = state.evolve();
            }
        state.finish(result);
                   
        // flush the output
        state.output.flush();

        // Possibly print out the run parameters
        PrintWriter pw = new PrintWriter(System.err);
                
        // before we print out access information, we need to still "get" these
        // parameters, so that they show up as accessed and gotten.
        state.parameters.getBoolean(new Parameter(P_PRINTUSEDPARAMETERS),null,false);
        state.parameters.getBoolean(new Parameter(P_PRINTACCESSEDPARAMETERS),null,false);
        state.parameters.getBoolean(new Parameter(P_PRINTUNUSEDPARAMETERS),null,false);
        state.parameters.getBoolean(new Parameter(P_PRINTUNACCESSEDPARAMETERS),null,false);
        state.parameters.getBoolean(new Parameter(P_PRINTALLPARAMETERS),null,false);
                
        //...okay, here we go...
                
        if (state.parameters.getBoolean(new Parameter(P_PRINTUSEDPARAMETERS),null,false))
            {
            pw.println("\n\nUsed Parameters\n===============\n");
            state.parameters.listGotten(pw);
            }

        if (state.parameters.getBoolean(new Parameter(P_PRINTACCESSEDPARAMETERS),null,false))
            {
            pw.println("\n\nAccessed Parameters\n===================\n");
            state.parameters.listAccessed(pw);
            }

        if (state.parameters.getBoolean(new Parameter(P_PRINTUNUSEDPARAMETERS),null,false))
            {
            pw.println("\n\nUnused Parameters\n"+
                       "================= (Ignore parent.x references) \n");
            state.parameters.listNotGotten(pw);
            }

        if (state.parameters.getBoolean(new Parameter(P_PRINTUNACCESSEDPARAMETERS),null,false))
            {
            pw.println("\n\nUnaccessed Parameters\n"+
                       "===================== (Ignore parent.x references) \n");
            state.parameters.listNotAccessed(pw);
            }

        if (state.parameters.getBoolean(new Parameter(P_PRINTALLPARAMETERS),null,false))
            {
            pw.println("\n\nAll Parameters\n==============\n");
            // list only the parameters visible.  Shadowed parameters not shown
            parameters.list(pw,false);
            }

        pw.flush();

        System.err.flush();
        System.out.flush();
                
        // finish by closing down Output.  This is because gzipped and other buffered
        // streams just don't shut write themselves out, and finalize isn't called
        // on them because Java's being obnoxious.  Pretty stupid.
        state.output.close();
        }





    /** Top-level evolutionary loop.  Feel free to modify this as you like in a subclass. */
    public static void main(String[] args)
        {
        EvolutionState state;
        ParameterDatabase parameters;
                
        // if we're loading from checkpoint, let's finish out the most recent job
        state = possiblyRestoreFromCheckpoint(args);
        if (state!=null)
            state.run(EvolutionState.C_STARTED_FROM_CHECKPOINT);
                        
        // this simple job iterator just uses the 'jobs' parameter, iterating from 0 to 'jobs' - 1
        // inclusive.  The current job number is stored in state.jobs[0], so we'll begin there if
        // we had loaded from checkpoint.
                
        int jobCurrent = 0;                             // what's the next job number?
        if (state != null)  // loaded from checkpoint
            {
            try
                {
                if (state.runtimeArguments == null)
                    Output.initialError("Checkpoint completed from job started by foreign program (probably GUI).  Exiting...");
                args = state.runtimeArguments;                                      // restore runtime arguments from checkpoint
                jobCurrent = ((Integer)(state.job[0])).intValue() + 1;  // extract next job number
                }
            catch (Exception e)
                {
                Output.initialError("EvolutionState's jobs variable is not set up properly.  Exiting...");
                }
            }


        // at this point we have: a current job number (jobCurrent) and the runtime
        // arguments passed in or -- if we restored from checkpoint -- the original ones.
        // We first make a sacrificial ParameterDatabase to load the number of jobs to run for.
        parameters = loadParameterDatabase(args);
        int numJobs = parameters.getIntWithDefault(new Parameter("jobs"), null, 1);
        if (numJobs < 1)
            Output.initialError("The 'jobs' parameter must be >= 1 (or not exist, which defaults to 1)");
                
                
        // Now we know how many jobs remain.  You can of course use the jobs[] array in other
        // ways if you like.  This is just the default main().
        for(int job = jobCurrent ; job < numJobs; job++)
            {
                        
            // load the parameter database (reusing the sacrificial one if we can)
            if (parameters == null)
                parameters = loadParameterDatabase(args);
                        
            // Initialize the EvolutionState, then set its job variables
            state = initialize(parameters, job);  // pass in job# as the seed increment
            state.output.systemMessage("Job: " + job);
            state.job = new Object[1];                // make the job argument storage
            state.job[0] = new Integer(job);  // stick the current job in our job storage
            state.runtimeArguments = args;    // stick the runtime arguments in our storage

            // define the file prefix if you like
            String jobFilePrefix = null;
            if (numJobs > 1)                                  // only if iterating (so we can be backwards-compatible),
                jobFilePrefix = "job." + job + ".";     // add a prefix for checkpoint/output files 
                                
            // Here you can set up the EvolutionState's parameters further before it's setup(...).
            // This includes replacing the random number generators, changing values in state.parameters,
            // changing instance variables (except for job and runtimeArguments, please), etc.





            // now we let it go
            startFresh(state, jobFilePrefix);
            parameters = null;  // so we load a fresh database next time around
            }

        System.exit(0);
        }







    /** The old entry method for an evolutionary run.  Loads either from a checkpoint file, 
        or loads a parameter file and sets up from that.  */
        
    public static void main2(String[] args)
        {
        // First we determine if we're loading from a checkpoint file
        
        EvolutionState state=null;
        ParameterDatabase parameters=null;
        Output output;
        MersenneTwisterFast[] random;
        int[] seeds;
        int breedthreads = 1;
        int evalthreads = 1;
        int verbosity;
        boolean store;
        int x;


        // first we load from a checkpoint file if the user requested
        // that and then we start up from there.

        for(x=0;x<args.length-1;x++)
            if (args[x].equals(A_CHECKPOINT))
                {
                try
                    {
                    System.err.println("Restoring from Checkpoint " + args[x+1]);
                    state=Checkpoint.restoreFromCheckpoint(args[x+1]);
                    state.run(EvolutionState.C_STARTED_FROM_CHECKPOINT);
                    }
                catch(OptionalDataException e)
                    {
                    Output.initialError(
                        "A ClassNotFoundException was generated upon" +
                        "starting up from a checkpoint." +
                        "\nHere it is:\n" + e); 
                    }
                catch(ClassNotFoundException e)
                    {
                    Output.initialError(
                        "A ClassNotFoundException was generated upon" +
                        "starting up from a checkpoint." +
                        "\nHere it is:\n" + e); 
                    }
                catch (IOException e)
                    { 
                    Output.initialError(
                        "An IO Exception was generated upon" +
                        "starting up, probably in setting up a log" +
                        "\nHere it is:\n" + e); 
                    }
                System.exit(0);
                }

        // at this point, we don't have a checkpoint file so we try
        // reading instead from a parameter file and starting up fresh
        
        if (state==null) // couldn't find a checkpoint argument
            {

            // 0. find the parameter database
            for(x=0;x<args.length-1;x++)
                if (args[x].equals(A_FILE))
                    {
                    try
                        {
                        parameters=new ParameterDatabase(
                            // not available in jdk1.1: new File(args[x+1]).getAbsoluteFile(),
                            new File(new File(args[x+1]).getAbsolutePath()),
                            args);
                        break;
                        }
                    catch(FileNotFoundException e)
                        { Output.initialError(
                            "A File Not Found Exception was generated upon" +
                            "reading the parameter file \"" + args[x+1] + 
                            "\".\nHere it is:\n" + e); }
                    catch(IOException e)
                        { Output.initialError(
                            "An IO Exception was generated upon reading the" +
                            "parameter file \"" + args[x+1] +
                            "\".\nHere it is:\n" + e); } 
                    }
            if (parameters==null)
                Output.initialError(
                    "No parameter file was specified." ); 


            // 1. create the output
            store = parameters.getBoolean(new Parameter(P_STORE),null,false);

            verbosity = parameters.getInt(new Parameter(P_VERBOSITY),null,0);
            if (verbosity<0)
                Output.initialError("Verbosity should be an integer >= 0.\n",
                                    new Parameter(P_VERBOSITY)); 

            output = new Output(store,verbosity);
            output.setFlush(
                parameters.getBoolean(new Parameter(P_FLUSH),null,false));


            // stdout is always log #0.  stderr is always log #1.
            // stderr accepts announcements, and both are fully verbose 
            // by default.
            output.addLog(ec.util.Log.D_STDOUT,Output.V_VERBOSE,false);
            output.addLog(ec.util.Log.D_STDERR,Output.V_VERBOSE,true);

            
            // 2. set up thread values

            breedthreads = parameters.getInt(
                new Parameter(P_BREEDTHREADS),null,1);

            if (breedthreads < 1)
                Output.initialError("Number of breeding threads should be an integer >0.",
                                    new Parameter(P_BREEDTHREADS));


            evalthreads = parameters.getInt(
                new Parameter(P_EVALTHREADS),null,1);

            if (evalthreads < 1)
                Output.initialError("Number of eval threads should be an integer >0.",
                                    new Parameter(P_EVALTHREADS));



            // 3. create the Mersenne Twister random number generators,
            // one per thread

            random = new MersenneTwisterFast[breedthreads > evalthreads ? 
                                             breedthreads : evalthreads];
            seeds = new int[breedthreads > evalthreads ? 
                            breedthreads : evalthreads];
                            
//  another way to do it?
//            long ltime = System.currentTimeMillis();
//            int time = (int)(ltime >>> 32) ^ (int)(ltime & 0x00000000FFFFFFFFL);  // fold on top of one another
            int time = (int)System.currentTimeMillis();  // safe because we're getting low-order bits 
            String seed_message = "Seed: ";
            for (x=0;x<random.length;x++)
                {
                int seed = 1;
                String tmp_s = parameters.getString(
                    new Parameter(P_SEED).push(""+x),null);
                if (tmp_s==null) // uh oh
                    {
                    Output.initialError("Seed should be an integer.",
                                        new Parameter(P_SEED).push(""+x));
                    
                    }
                else if (tmp_s.equalsIgnoreCase(V_SEED_TIME))
                    {
                    seed = time++;
                    if (seed==0)
                        Output.initialError("Whoa! This Java version is returning 0 for System.currentTimeMillis(), which ain't right.  This means you can't use '"+V_SEED_TIME+"' as a seed ",new Parameter(P_SEED).push(""+x));
                    else seed_message = seed_message + seed + " ";
                    }
                else
                    {
                    try
                        {
                        seed = parameters.getInt(new Parameter(P_SEED).push(""+x),null);
                        }
                    catch (NumberFormatException e)
                        {
                        Output.initialError("Invalid Seed Value (must be an integer):\n" + e);
                        }
                    seed_message = seed_message + seed + " ";
                    }
                    
                seeds[x] = seed;
                }

            for (x=0;x<random.length;x++)
                {
                for (int y=x+1;y<random.length;y++)
                    if (seeds[x]==seeds[y])
                        {
                        Output.initialError(P_SEED+"."+x+" ("+seeds[x]+") and "+P_SEED+"."+y+" ("+seeds[y]+") ought not be the same seed."); 
                        }
                random[x] = new MersenneTwisterFast(seeds[x]);
                }

            // 4.  Start up the evolution
            
            // what evolution state to use?
            state = (EvolutionState)
                parameters.getInstanceForParameter(new Parameter(P_STATE),null,
                                                   EvolutionState.class);
            state.parameters = parameters;
            state.random = random;
            state.output = output;
            state.evalthreads = evalthreads;
            state.breedthreads = breedthreads;

            output.systemMessage(Version.message());
            output.systemMessage("Threads:  breed/" + breedthreads + " eval/" + evalthreads);
            output.systemMessage(seed_message);
            
//            try 
//                {
            state.run(EvolutionState.C_STARTED_FRESH);
//                }
/*
  catch (IOException e)
  { 
  Output.initialError(
  "An IO Exception was generated upon" +
  "starting up, probably in setting up a log" +
  "\nHere it is:\n" + e); 
  }
*/
            
            // Possibly print out the run parameters

               
            // flush the output
            output.flush();

            PrintWriter pw = new PrintWriter(System.err);
            
            // before we print out access information, we need to still "get" these
            // parameters, so that they show up as accessed and gotten.
            parameters.getBoolean(new Parameter(P_PRINTUSEDPARAMETERS),null,false);
            parameters.getBoolean(new Parameter(P_PRINTACCESSEDPARAMETERS),null,false);
            parameters.getBoolean(new Parameter(P_PRINTUNUSEDPARAMETERS),null,false);
            parameters.getBoolean(new Parameter(P_PRINTUNACCESSEDPARAMETERS),null,false);
            parameters.getBoolean(new Parameter(P_PRINTALLPARAMETERS),null,false);
            
            //...okay, here we go...
            
            if (parameters.getBoolean(new Parameter(P_PRINTUSEDPARAMETERS),null,false))
                {
                pw.println("\n\nUsed Parameters\n===============\n");
                parameters.listGotten(pw);
                }

            if (parameters.getBoolean(new Parameter(P_PRINTACCESSEDPARAMETERS),null,false))
                {
                pw.println("\n\nAccessed Parameters\n===================\n");
                parameters.listAccessed(pw);
                }

            if (parameters.getBoolean(new Parameter(P_PRINTUNUSEDPARAMETERS),null,false))
                {
                pw.println("\n\nUnused Parameters\n"+
                           "================= (Ignore parent.x references) \n");
                parameters.listNotGotten(pw);
                }

            if (parameters.getBoolean(new Parameter(P_PRINTUNACCESSEDPARAMETERS),null,false))
                {
                pw.println("\n\nUnaccessed Parameters\n"+
                           "===================== (Ignore parent.x references) \n");
                parameters.listNotAccessed(pw);
                }

            if (parameters.getBoolean(new Parameter(P_PRINTALLPARAMETERS),null,false))
                {
                pw.println("\n\nAll Parameters\n==============\n");
                // list only the parameters visible.  Shadowed parameters not shown
                parameters.list(pw,false);
                }


            pw.flush();

            System.err.flush();
            System.out.flush();
            
            // finish by closing down Output.  This is because gzipped and other buffered
            // streams just don't shut write themselves out, and finalize isn't called
            // on them because Java's being obnoxious.  Pretty stupid.
            output.close();
            }
        }
    }

