/*
 * Created on Oct 7, 2004
 */
package ec.eval;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.io.Serializable;
import java.net.ServerSocket;
import java.net.Socket;

import ec.EvolutionState;
import ec.Individual;
import ec.Population;
import ec.Problem;
import ec.coevolve.GroupedProblemForm;
import ec.simple.SimpleProblemForm;
import ec.util.CompressingInputStream;
import ec.util.CompressingOutputStream;
import ec.util.MersenneTwisterFast;
import ec.util.Parameter;

import java.util.ArrayList;

/**
 * A MasterProblem object is a singleton object that uses remote slave processes
 * to evaluate individuals in a distributed fashion. Evaluation of individuals
 * is the only distributed behavior of the MasterProblem. All other behavior is
 * delegated to a local instance of the actual problem (specified for the
 * experiment via the <tt>eval.problem</tt> parameter). It is not intended for
 * users to extend MasterProblem.
 * 
 * <p>
 * To take advantage of the distributed funtionality offered by MasterProblem,
 * specify the <tt>eval.masterproblem</tt> parameter which must have a value
 * of <tt>ec.eval.MasterProblem</tt>.
 * 
 * <p>
 * MasterProblem expects a number of slave processes to connect equal to the
 * number of evaluation threads specified by the <tt>evalthreads
 * </tt>
 * parameter. The MasterProblem will not evaluate any individuals until the
 * expected number of slave processes have registered. Any additional slave
 * processes that register will be ignored.
 * 
 * <p>
 * There is no fault tolerance built in to the MasterProblem. If a slave process
 * dies in the middle of evaluating an individual, the MasterProblem will become
 * aware of the problem and abort the job. If a slave process hangs while
 * processing an individual (if, for example, the slave is enters an infinite
 * loop) the master process will hang waiting for a response.
 * 
 * <p>
 * Future versions of MasterProblem will take advantage of available Slave
 * processes and adjust it's utilization of them as they come and go. This means
 * that the MasterProblem will not abort unless all existing Slave processes
 * die. The MasterProblem will also be equiped with a (user configurable)
 * timeout mechanism. If the timeout expires while waiting for a response from a
 * given slave, the evaluation will be aborted and the Slave will be considered
 * dead. At this point, the MasterProblem will adjust it's utilization of
 * available Slaves and restart the evaluation process.
 * 
 * <p>
 * <b>Parameters </b> <br>
 * <table>
 * <tr>
 * <td valign=top><tt>eval.compression</tt><br>
 * <font size=-1>bool =<tt>true</tt> or <tt>false</tt> (default) </font>
 * </td>
 * <td valign=top>(whether to use compression when transmitting individuals to
 * slave processes)</td>
 * </tr>
 * 
 * <tr>
 * <td valign=top><tt>eval.master.port</tt><br>
 * <font size=-1>int </font></td>
 * <td valign=top>(port to which slave process will connect and register
 * themselves with the master process)</td>
 * </tr>
 * 
 * <tr>
 * <td valign=top><tt>eval.masterproblem</tt><br>
 * <font size=-1>classname, == ec.eval.MasterProblem </font></td>
 * <td valign=top>(the master problem class)</td>
 * </tr>
 * </table>
 * 
 * @author spaus
 */
public class MasterProblem extends Problem 
    implements SimpleProblemForm, GroupedProblemForm {

    public Problem problem;

    public MasterProblemServer server;

    public Thread serverThread;

    public boolean batchMode;

    public void setup(final EvolutionState state, final Parameter base) {
        super.setup(state, base);
        server = new MasterProblemServer();
        batchMode = false;
    }

    Sync[] sync;
    // a place to store individuals to be evaluated
    ArrayList[] evaluatedIndividuals;
    // the worker thread;
    WorkerThread[] wt;

    public void prepareToEvaluate(final EvolutionState state, final int threadnum)
        {
        int N = server.slaves.length;
        if( evaluatedIndividuals == null || evaluatedIndividuals.length != N )
            {
            evaluatedIndividuals = new ArrayList[N];
            sync = new Sync[N];
            for( int i = 0 ; i < N ; i++ )
                {
                evaluatedIndividuals[i] = new ArrayList();
                sync[i] = new Sync();
                }
            wt = new WorkerThread[N];
            }
        for( int i = 0 ; i < N ; i++ )
            {
            evaluatedIndividuals[i].clear();
            sync[i].reset(evaluatedIndividuals[i],0,0,Integer.MIN_VALUE);
            wt[i] = new WorkerThread(this,state,i);
            wt[i].start();
            }
        batchMode = true;
        }

    public void finishEvaluating(final EvolutionState state, final int threadnum)
        {
        // flush out individuals to make them available
        SlaveInfo slaveInfo = server.slaves[threadnum];
        DataOutputStream dataOut = slaveInfo.dataOut;
        try { dataOut.flush(); } catch (IOException e) { }
        for( int i = 0 ; i < server.slaves.length ; i++ )
            sync[i].finishEvaluating();
        batchMode = false;
        }

    public void evaluate(EvolutionState state, Individual ind, int threadnum) {
        /*
         * Since we're going to make a distributed call here, this check might
         * not make sense.
         */
        if (!(problem instanceof SimpleProblemForm)) {
            state.output.fatal("StarProblem.evaluate(...) invoked, but the Problem is not of type SimpleProblemForm");
            }

        synchronized(server)
            {
            while (! server.allSlavesConnected())
                {
                try
                    {
                    this.server.wait();
                    }
                catch (InterruptedException e)
                    {
                    // Server notified here
                    }
                }
            }
        
        // Acquire a slave socket
        SlaveInfo slaveInfo = server.slaves[threadnum];

        DataInputStream dataIn = slaveInfo.dataIn;
        DataOutputStream dataOut = slaveInfo.dataOut;

        try
            {
            // Tell the server we're evaluating a SimpleProblemForm
            dataOut.writeInt(1);
            }
        catch( IOException e ) { state.output.fatal("Caught fatal IOException\n"+e); }
        
        // Determine the subpopulation number associated with this individual
        int subPopNum = 0;
        boolean found = false;
        for (int x=0;x<state.population.subpops.length && !found;x++)
            {
            if (state.population.subpops[x].species == ind.species)
                {
                subPopNum = x;
                found = true;
                }
            }

        if (!found)
            {
            // Is it possible that there isn't a matching species?
            state.output.fatal("Whoa!  Couldn't find a matching species for the individual!");
            }
        
        try
            {
            // Transmit the subpopulation number to the slave 
            dataOut.writeInt(subPopNum);

            // Transmit the individual to the server for evaluation...
            ind.writeIndividual(state, dataOut);

        try { dataOut.flush(); } catch (IOException e) { }
            if( batchMode )
                {
                sync[threadnum].put(ind);
                }
            else
                {
                // only flush if not in batch mode
                dataOut.flush();
                ind.readIndividual(state, dataIn);
                }
            }
        catch (Exception e)
            {
            e.printStackTrace();
            state.output.fatal("Exception encountered reading result from slave "+
                               server.slaves[threadnum].slaveName+":\n"+e);
            synchronized(server.slaves)
                {
                server.slaves[threadnum].shutdown();
                server.slaves[threadnum] = null;
                }
            }
    }

    /* (non-Javadoc)
     * @see ec.simple.SimpleProblemForm#describe(ec.Individual, ec.EvolutionState, int, int, int)
     */
    public void describe(Individual ind, EvolutionState state, int threadnum,
                         int log, int verbosity) {
        if (!(problem instanceof SimpleProblemForm)) {
            state.output.fatal("StarProblem.describe(...) invoked, but the Problem is not of SimpleProblemForm");
            }

        ((SimpleProblemForm)problem).describe( ind, state, threadnum, log, verbosity);
    }

    /* (non-Javadoc)
     * @see ec.coevolve.GroupedProblemForm#preprocessPopulation(ec.EvolutionState, ec.Population)
     */
    public void preprocessPopulation(EvolutionState state, Population pop) {
        if (!(problem instanceof GroupedProblemForm)) {
            state.output.fatal("StarProblem.preprocessPopulation(...) invoked, but the Problem is not of GroupedProblemForm");
            }

        ((GroupedProblemForm) problem).preprocessPopulation(state, pop);
    }

    /* (non-Javadoc)
     * @see ec.coevolve.GroupedProblemForm#postprocessPopulation(ec.EvolutionState, ec.Population)
     */
    public void postprocessPopulation(EvolutionState state, Population pop) {
        if (!(problem instanceof GroupedProblemForm)) {
            state.output.fatal("StarProblem.postprocessPopulation(...) invoked, but the Problem is not of GroupedProblemForm");
            }

        ((GroupedProblemForm) problem).postprocessPopulation(state, pop);
    }

    /* (non-Javadoc)
     * @see ec.coevolve.GroupedProblemForm#evaluate(ec.EvolutionState, ec.Individual[], boolean[], boolean, int)
     */
    public void evaluate(EvolutionState state, Individual[] inds,
                         boolean[] updateFitness, boolean countVictoriesOnly, int threadnum) {
        /*
         * Since we're going to make a distributed call here, this check might
         * not make sense.
         */
        if (!(problem instanceof GroupedProblemForm)) {
            state.output.fatal("StarProblem.evaluate(...) invoked, but the Problem is not of GroupedProblemForm");
            }

        if (! server.allSlavesConnected())
            {
            synchronized( this.server )
                {
                try
                    {
                    this.server.wait();
                    }
                catch (InterruptedException e)
                    {
                    // Server notified here
                    }
                }
            }

        // Acquire a slave socket
        SlaveInfo slaveInfo = server.slaves[threadnum];

        DataInputStream dataIn = slaveInfo.dataIn;
        DataOutputStream dataOut = slaveInfo.dataOut;

        try
            {
            // Tell the server we're evaluating a GroupedProblemForm
            dataOut.writeInt(2);
            }
        catch( IOException e ) { state.output.fatal("Caught fatal IOException\n"+e); }

        // Determine the subpopulation number associated with the individuals
        int subPopNum = 0;
        boolean subPopNumFound = false;
        for (int x=0;x<state.population.subpops.length && !subPopNumFound;x++)
            {
            if (state.population.subpops[x].species == inds[0].species)
                {
                subPopNum = x;
                subPopNumFound = true;
                }
            }

        if (!subPopNumFound)
            {
            // Is it possible that there isn't a matching species?
            state.output.fatal("Whoa!  Couldn't find a matching species for the individual!");
            }
        
        // ... then read the results back
        try
            {
            // Transmit the subpopulation number to the slave 
            dataOut.writeInt(subPopNum);
                        
            // Tell the server how many individuals are involved in this evaluation
            dataOut.writeInt(inds.length);

            // Tell the server whether to count victories only or not.
            dataOut.writeBoolean(countVictoriesOnly);
                        
            // Transmit the individuals to the server for evaluation...
            for(int i=0;i<inds.length;i++)
                {
                inds[i].writeIndividual(state, dataOut);
                dataOut.writeBoolean(updateFitness[i]);
                }

            if( batchMode )
                {
                for(int i=0;i<inds.length;i++)
                    sync[threadnum].put(inds[i]);
                }
            else
                {
                // only flush if not in batch mode
                dataOut.flush();
                for(int i=0;i<inds.length;i++)
                    inds[i].readIndividual(state, dataIn);
                }
            }
        catch (Exception e)
            {
            e.printStackTrace();
            state.output.fatal("Exception encountered reading result from slave "+
                               server.slaves[threadnum].slaveName+":\n"+e);
            synchronized(server.slaves)
                {
                server.slaves[threadnum].shutdown();
                server.slaves[threadnum] = null;
                }
            }
    }

    /** Custom serialization */
    private void writeObject(ObjectOutputStream out) throws IOException
        {
        out.writeObject(problem);
        out.writeObject(server);
        }

    private void readObject(ObjectInputStream in) throws IOException, ClassNotFoundException
        {
        problem = (Problem) in.readObject();
        server = (MasterProblemServer) in.readObject();
        }

    public void initializeContacts( final EvolutionState state )
        {
        server.setupServerFromDatabase(state);
        serverThread = server.spawnThread();
        }

    public void reinitializeContacts( final EvolutionState state )
        {
        initializeContacts(state);
        }

    public void shutdown()
        {
        this.server.shutdown();
        try
            {
            this.serverThread.join();
            }
        catch (InterruptedException e)
            {
            }
        }
    }

// Just a structure to hold information about each server process.

/**
 * A structure to hold information about each slave process.
 * 
 * @author spaus
 */

class SlaveInfo {
    public SlaveInfo( 
        String slaveName,
        Socket evalSocket,
        DataOutputStream dataOut,
        DataInputStream dataIn )
        {
        this.slaveName = slaveName;
        this.evalSocket = evalSocket;
        this.dataOut = dataOut;
        this.dataIn = dataIn;
        //this.connected = evalSocket.isConnected();
        }
    
    public void shutdown()
        {
        try
            {
            // -1 means shutdown
            dataOut.writeInt(-1);
            dataOut.close();
            dataIn.close();
            evalSocket.close();
            //connected = false;
            }
        catch (IOException e)
            {
            // Just ignore the exception since we're closing the socket and
            // streams.
            }
        }

    /**
     * Name of the slave process
     */
    public String slaveName;

    /**
     * Socket for communication with the slave process
     */
    public Socket evalSocket;

    /**
     * Used to transmit data to the slave.
     */
    public DataOutputStream dataOut;

    /**
     * Used to read results and randoms state from slave.
     */
    public DataInputStream dataIn;

    /*
     * Indicates whether the socket has connected to the slave (not used)
     */
    //public boolean connected;
    }

/**
 * A background thread that listens for slave processes to register. Assigns a
 * slave to an evaluation thread on registration. Listens on the port specified
 * by the <tt>eval.master.port</tt> parameter. Configures the socket IO
 * streams to use compression if <tt>eval.compression</tt> is set to <tt>
 * true</tt>.
 * 
 * @author spaus
 */

class MasterProblemServer
    implements Runnable, Serializable
    {
    public static final String P_EVALMASTERPORT = "eval.master.port";

    public static final String P_EVALCOMPRESSION = "eval.compression";

    public SlaveInfo[] slaves;

    /**
     * Random states for the slave processes.
     */
    public MersenneTwisterFast[] randomStates;

    /**
     *  
     */
    public ServerSocket servSock;

    /**
     * Indicates whether compression is used over the socket IO streams.
     */
    public boolean useCompression;

    public EvolutionState state;

    /**
     * Indicates to the background thread that a shutdown is in progress and to
     * stop processing.
     */
    private boolean shutdownInProgress = false;

    /**
     * After the MasterProblemServer is created, it needs to be told to
     * initialize itself from information in the parameter database. It
     * distinguishes between starting fresh and restoring from a checkpoint by
     * the state of the randomStates and slaves arrays.
     * 
     * @param state the evolution state
     */
    public void setupServerFromDatabase( final EvolutionState state )
        {
        this.state = state;

        int port = state.parameters.getInt(
            new Parameter( P_EVALMASTERPORT ));

        useCompression = state.parameters.getBoolean(new Parameter(P_EVALCOMPRESSION),null,false);

        try
            {
            servSock = new ServerSocket(port);
            state.output.systemMessage("Star problem server listening on "+port);
            }
        catch( IOException e )
            {
            state.output.fatal("Unable to bind to port " + port + ": " + e);
            }

        if (slaves == null)
            slaves = new SlaveInfo[state.evalthreads];

        if (randomStates == null)
            randomStates = new MersenneTwisterFast[state.evalthreads];

        for (int i = 0; i < randomStates.length; ++i)
            {
            randomStates[i] = state.random[i];
            }
        }

    /**
     * Indicates that the background thread is to shut down and closes the
     * server socket. (should probably be synchronized).
     */
    public void shutdown()
        {
        state.output.systemMessage("Shutting down server thread.");

        shutdownInProgress = true;

        try
            {
            servSock.close();
            }
        catch (IOException e)
            {
            }
        }

    /**
     * @return <tt>true</tt> if <tt>evalthreads</tt> slaves have connected,
     *         <tt>false</tt> otherwise.
     */
    public synchronized boolean allSlavesConnected()
        {
        boolean allSlavesConnected = true;

        for ( int i = 0; i < slaves.length; ++i )
            {
            if (slaves[i] == null ) //  || ! slaves[i].connected )
                {
                allSlavesConnected = false;
                break;
                }
            }

        return allSlavesConnected;
        }

    /**
     * Writes the slaves' random states to the checkpoint file.
     * 
     * @param s checkpoint file output stream
     * @throws IOException
     */
    private void writeObject(ObjectOutputStream out) throws IOException
        {
        out.writeInt(randomStates.length);

        for ( int i = 0; i < randomStates.length; ++i )
            {
            randomStates[i] = new MersenneTwisterFast();

            // Tell the slave we want it's random state
            slaves[i].dataOut.writeInt(3);
            
            randomStates[i].readState(slaves[i].dataIn);
            out.writeObject(randomStates[i]);
            }
        }

    /**
     * Restores the slaves random states from the checkpoint file.
     * 
     * @param s checkpoint file input stream.
     * @throws IOException
     * @throws ClassNotFoundException
     */
    private void readObject(ObjectInputStream in) throws IOException, ClassNotFoundException
        {
        int numSlaves = in.readInt();

        slaves = new SlaveInfo[numSlaves];
        randomStates = new MersenneTwisterFast[numSlaves];

        for ( int i = 0; i < randomStates.length; ++i )
            {
            randomStates[i] = (MersenneTwisterFast) in.readObject();
            }
        }

    /**
     * @return an index in the slaves array of an unconnected slave or
     *         <tt>-1</tt> if all slaves are connected.
     */
    private int findUnconnectedSlave()
        {
        int unconnectedSlave = -1;

        for ( int i = 0; i < slaves.length; ++i )
            {
            if (slaves[i] == null ) // || ! slaves[i].connected )
                {
                unconnectedSlave = i;
                break;
                }
            }

        return unconnectedSlave;
        }

    public void run()
        {
        Socket slaveSock;

        try
            {
            while (!shutdownInProgress)
                {
                slaveSock = servSock.accept();
//                              slaveSock.setTcpNoDelay(true);
//                              System.out.println(slaveSock.getSendBufferSize());//XX
//                              slaveSock.setSendBufferSize(8000);
                state.output.systemMessage("Connection accepted from "+slaveSock);
                
                synchronized (slaves)
                    {
                    if (allSlavesConnected())
                        {
                        /**
                         * TODO Need a better message here.
                         */
                        state.output.systemMessage("Slave not needed.  We're all full.");
                        slaveSock.close();
                        }
                    else
                        {
                        int newSlaveIndex = findUnconnectedSlave();
                        if (newSlaveIndex >= 0)
                            {
                            DataInputStream dataIn = null;
                            DataOutputStream dataOut = null;
                            try
                                {
                                InputStream tmpIn = slaveSock.getInputStream();
                                if (this.useCompression)
                                    tmpIn = new CompressingInputStream(tmpIn);

                                dataIn = new DataInputStream(tmpIn);
                                
                                OutputStream tmpOut = slaveSock.getOutputStream();
                                if (this.useCompression)
                                    tmpOut = new CompressingOutputStream(tmpOut);

                                dataOut = new DataOutputStream(tmpOut);
                                System.out.println("Waiting for the name of the island....");//XX
                                String slaveName = dataIn.readUTF();
                                System.out.println("The name of the island is "+slaveName);//XX
                                
                                SlaveInfo newSlave = new SlaveInfo( slaveName, slaveSock, dataOut, dataIn );
                                    
                                // Write random state for eval thread to slave
                                randomStates[newSlaveIndex].writeState(dataOut);
                                dataOut.flush();

                                slaves[newSlaveIndex] = newSlave;

                                synchronized(this)
                                    {
                                    if (allSlavesConnected())
                                        this.notifyAll();
                                    }
                                }
                            catch (IOException e)
                                {
                                state.output.fatal("Unable to open stream from slave socket:\n"+e);
                                if ( slaves[newSlaveIndex] != null)
                                    {
                                    slaves[newSlaveIndex].shutdown();
                                    slaves[newSlaveIndex] = null;
                                    }
                                }
                            }
                        else
                            {
                            state.output.warning("Some slave is not connected, but unable to find it's index.");
                            }
                        }
                    }
                }
            }
        catch (IOException e)
            {
            if (!this.shutdownInProgress)
                state.output.fatal("An exception occured while waiting for a connection:\n"+e);  
            }

        for (int i = 0; i < slaves.length; ++i)
            if (slaves[i] != null)
                slaves[i].shutdown();
        }

    /**
     * Creates and starts a background thread for this server. 
     * 
     * @return the background thread
     */
    public Thread spawnThread()
        {
        Thread thread = new Thread(this);
        thread.start();
        return thread;
        }

    }

class WorkerThread extends Thread
    {
    MasterProblem problem;
    EvolutionState state;
    int threadNumber;

    boolean shouldStop = false;

    WorkerThread( MasterProblem p, EvolutionState s, int t )
        {
        problem = p;
        state = s;
        threadNumber = t;
        shouldStop = false;
        }

    public void run()
        {
        while( true )
            {
            Individual ind = problem.sync[threadNumber].get();
            if( ind == null )
                {
                problem.sync[threadNumber].workerExits();
                break;
                }

            try
                {
                // ... then read the results back
                DataInputStream dataIn = problem.server.slaves[threadNumber].dataIn;
                ind.readIndividual(state, dataIn);
                }
            catch (Exception e)
                {
                synchronized(problem.server.slaves)
                    {
                    problem.server.slaves[threadNumber].shutdown();
                    problem.server.slaves[threadNumber] = null;
                    }
                e.printStackTrace();
                state.output.fatal("Exception encountered reading result from slave "+
                                   problem.server.slaves[threadNumber].slaveName+":\n"+e);
                }
            }
        }
    }

class Sync
    {
    private ArrayList individuals;
    private int numRead;
    private int numEval;
    private int total2Read;
    private boolean workerExited;

    public synchronized void reset( ArrayList inds, int nR, int nE, int t2R )
        {
        individuals = inds;
        numRead = nR;
        numEval = nE;
        total2Read = t2R;
        workerExited = false;
        }

    public synchronized Individual get()
        {
        while( (numEval >= numRead) && (numEval > total2Read) )
            {
            try
                {
                wait();
                }
            catch (InterruptedException e) {}
            }
        if( numEval == total2Read )
            {
            notifyAll();
            return null;
            }
        else
            {
            Individual result = (Individual)(individuals.get(numEval));
            numEval++;
            notifyAll();
            return result;
            }
        }

    public synchronized void put(Individual ind)
        {
        individuals.add(ind);
        numRead++;
        notifyAll();
        }

    public synchronized void finishEvaluating()
        {
        total2Read = numRead;
        notifyAll();
        while( !workerExited )
            {
            try
                {
                wait();
                }
            catch (InterruptedException e) {}
            }
        }

    public synchronized void workerExits()
        {
        workerExited = true;
        notifyAll();
        }

    }
