/*
 * Created on Oct 8, 2004
 */
package ec.eval;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.ConnectException;
import java.net.Socket;
import java.net.UnknownHostException;

import ec.EvolutionState;
import ec.Fitness;
import ec.Individual;
import ec.Subpopulation;
import ec.coevolve.GroupedProblemForm;
import ec.simple.SimpleProblemForm;
import ec.util.CompressingInputStream;
import ec.util.CompressingOutputStream;
import ec.util.MersenneTwisterFast;
import ec.util.Output;
import ec.util.Parameter;
import ec.util.ParameterDatabase;
import ec.util.Version;

/**
 * Slave is the main entry point for a slave evaluation process.
 * 
 * <p>
 * The slave is run with one of the following argument formats:
 * 
 * @author spaus
 */
public class Slave {
    public final static String P_PRINTACCESSEDPARAMETERS = "print-accessed-params";

    public final static String P_PRINTUSEDPARAMETERS = "print-used-params";

    public final static String P_PRINTALLPARAMETERS = "print-all-params";

    public final static String P_PRINTUNUSEDPARAMETERS = "print-unused-params";

    public final static String P_PRINTUNACCESSEDPARAMETERS = "print-unaccessed-params";

    public final static String P_EVALSLAVENAME = "eval.slave-name";

    public final static String P_EVALMASTERHOST = "eval.master.host";

    public final static String P_EVALMASTERPORT = "eval.master.port";

    public final static String P_EVALCOMPRESSION = "eval.compression";

    public static final String P_SUBPOP = "pop.subpop";

    /** The argument indicating that we're starting up from a checkpoint file. */
    public static final String A_CHECKPOINT = "-checkpoint";

    /** The argument indicating that we're starting fresh from a new parameter file. */
    public static final String A_FILE = "-file";

    /** flush announcements parameter */
    public static final String P_FLUSH = "flush";

    /** nostore parameter */
    public static final String P_STORE = "store";

    /** verbosity parameter */
    public static final String P_VERBOSITY = "verbosity";

    /** seed parameter */
    public static final String P_SEED = "seed";

    /** 'time' seed parameter value */
    public static final String V_SEED_TIME = "time";

    /** state parameter */
    public static final String P_STATE = "state";

    /** How long we sleep in between attempts to connect to the master (in milliseconds). */
    public static final int SLEEP_TIME = 100;

    public static void main(String[] args) {
        EvolutionState state = null;
        ParameterDatabase parameters = null;
        Output output;
        MersenneTwisterFast[] random = new MersenneTwisterFast[1];
        random[0] = new MersenneTwisterFast();
        int breedthreads = 1;
        int evalthreads = 1;
        int verbosity;
        boolean store;
        int x;

        // 0. find the parameter database
        for (x = 0; x < args.length - 1; x++)
            if (args[x].equals(A_FILE))
                {
                try
                    {
                    parameters = new ParameterDatabase(
                        // not available in jdk1.1: new File(args[x+1]).getAbsoluteFile(),
                        new File(new File(args[x + 1]).getAbsolutePath()),
                        args);
                    break;
                    }
                catch(FileNotFoundException e)
                    { Output.initialError(
                        "A File Not Found Exception was generated upon" +
                        "reading the parameter file \"" + args[x+1] + 
                        "\".\nHere it is:\n" + e); }
                catch(IOException e)
                    { Output.initialError(
                        "An IO Exception was generated upon reading the" +
                        "parameter file \"" + args[x+1] +
                        "\".\nHere it is:\n" + e); } 
                }
        if (parameters == null)
            Output.initialError(
                "No parameter file was specified." ); 

        // 1. create the output
        store = parameters.getBoolean(new Parameter(P_STORE), null, false);

        verbosity = parameters.getInt(new Parameter(P_VERBOSITY), null, 0);
        if (verbosity < 0)
            Output.initialError("Verbosity should be an integer >= 0.\n",
                                new Parameter(P_VERBOSITY));

        output = new Output(store, verbosity);
        output.setFlush(
            parameters.getBoolean(new Parameter(P_FLUSH),null,false));

        // stdout is always log #0. stderr is always log #1.
        // stderr accepts announcements, and both are fully verbose
        // by default.
        output.addLog(ec.util.Log.D_STDOUT, Output.V_VERBOSE, false);
        output.addLog(ec.util.Log.D_STDERR, Output.V_VERBOSE, true);

        // 4. Set up the evolution state

        // what evolution state to use?
        state = (EvolutionState)
            parameters.getInstanceForParameter(new Parameter(P_STATE),null,
                                               EvolutionState.class);
        state.parameters = parameters;
        state.output = output;
        state.evalthreads = 1;
        state.breedthreads = 1;

        state.setup(null, null);

        output.systemMessage(Version.message());

        // 5. Open a server socket and listen for requests
        String slaveName = state.parameters.getString(
            new Parameter(P_EVALSLAVENAME),null);

        /*
        // Search for the node info for the associated server name
        if (slaveName == null)
        {
        slaveName = 
        Output.initialError("No slave name specified, using ", new Parameter(P_EVALSLAVENAME));
        }
        */
        
        String masterHost = state.parameters.getString(
            new Parameter(P_EVALMASTERHOST),null );
        int masterPort = state.parameters.getInt(
            new Parameter(P_EVALMASTERPORT));
        boolean useCompression = state.parameters.getBoolean(new Parameter(P_EVALCOMPRESSION),null,false);

        // Continue to serve new masters until killed.
        while (true)
            {
            try
                {
                Socket socket;
                long connectAttemptCount = 0;
                state.output.message("Connecting to master at "+masterHost+":"+masterPort);
                while (true)
                    {
                    try
                        {
                        socket = new Socket(masterHost, masterPort);
//                                              socket.setTcpNoDelay(true);
//                                              socket.setSendBufferSize(8000);
                        break;
                        }
                    catch (ConnectException e)   // it's not up yet...
                        {
                        connectAttemptCount++;
                        try
                            {
                            Thread.sleep(SLEEP_TIME);
                            }
                        catch( InterruptedException f )
                            {
                            }
                        }
                    }
                state.output.message("Connected to master after " + (connectAttemptCount * SLEEP_TIME) + " ms");
                
                DataInputStream dataIn = null;
                DataOutputStream dataOut = null;
                try
                    {
                    InputStream tmpIn = socket.getInputStream();
                    if (useCompression)
                        tmpIn = new CompressingInputStream(tmpIn);

                    dataIn = new DataInputStream(tmpIn);
                    OutputStream tmpOut = socket.getOutputStream();
                    if (useCompression)
                        tmpOut = new CompressingOutputStream(tmpOut);

                    dataOut = new DataOutputStream(tmpOut);
                    }
                catch (IOException e)
                    {
                    state.output.fatal("Unable to open input stream from socket:\n"+e);
                    }
    
                // specify the slaveName
                if (slaveName==null)
                    {
                    slaveName = socket.getLocalAddress().toString() + "/" + System.currentTimeMillis();
                    state.output.message("No slave name specified.  Using: " + slaveName);
                    }
                    
                dataOut.writeUTF(slaveName);
                System.out.println("The name of the island is "+slaveName);//XX
                dataOut.flush();
                
                // Read random state from Master
                random[0].readState(dataIn);

                state.random = random;
                // Is this a Simple or Grouped ProblemForm?
                int problemType;
                boolean done = false;
                try
                    {
                    while (! done)
                        {
                        // -1 means to shut down
                        problemType = dataIn.readInt();
                        switch (problemType)
                            {
                            case 1:
                                evaluateSimpleProblemForm(state, dataIn, dataOut);
                                break;
                            
                            case 2:
                                evaluateGroupedProblemForm(state, dataIn, dataOut);
                                break;
                                
                            case 3:
                                checkpointRandomState(state, dataOut);
                                break;
                                
                            case -1:
                                done = true;
                                socket.close();
                                break;
                                
                            default:
                                state.output.fatal("Unknown problem form specified: "+problemType);
                            }
                        }
                    }
                catch (IOException e)
                    {
                    // Since an IOException can happen here if the peer closes the socket
                    // on it's end, we don't necessarily have to exit.  Maybe we don't
                    // even need to print a warning, but we'll do so just to indicate
                    // something happened.
                    state.output.warning("Unable to read type of evaluation from master.  Maybe the master closed it's socket and exited?:\n"+e);
                    }
                } 
            catch (UnknownHostException e)
                {
                state.output.fatal(e.getMessage());
                }
            catch (IOException e)
                {
                state.output.fatal("Unable to connect to master:\n" + e);
                }
            }
    }
    
    public static void evaluateSimpleProblemForm( EvolutionState state, 
                                                  DataInputStream dataIn, DataOutputStream dataOut )
        {
        // Read the subpopulation number
        int subPopNum = -1;
        try
            {
            subPopNum = dataIn.readInt();
            }
        catch (IOException e)
            {
            state.output.fatal("Unable to read the subpopulation number from the master:\n"+e);
            }
        
        // Here we need to know the subpopulation number so as to create the
        // correct type of subpopulation in order to create the correct type
        // of individual.
        Parameter param = new Parameter(P_SUBPOP).push("" + subPopNum);
        Subpopulation subPop = 
            (Subpopulation)(state.parameters.getInstanceForParameterEq(
                                param,null,
                                Subpopulation.class));
        // Setup the subpopulation so that it is in a valid state.
        subPop.setup(state, param);
        
        // Read the individual from the stream
        Individual ind = null;
        try
            {
            ind = subPop.species.newIndividual(
                state, subPop, (Fitness)(subPop.f_prototype.protoClone()));
            ind.readIndividual(state,dataIn);
            }
        catch (IOException e)
            {
            state.output.fatal("Unable to read individual from master.");
            }
        catch (CloneNotSupportedException e)
            {
            state.output.fatal("Unable to clone a new fitness.");
            }

        // Evaluate the individual
        // TODO Check to make sure the real problem is an instance of SimpleProblemForm
        ((SimpleProblemForm)(state.evaluator.p_problem)).evaluate( state, ind, 0 );
        
        // Return the evaluated individual to the master
        try
            {
            ind.writeIndividual(state, dataOut);
            dataOut.flush();//XX
            }
        catch( IOException e ) { state.output.fatal("Caught fatal IOException\n"+e ); }
        }
    
    public static void evaluateGroupedProblemForm( EvolutionState state, 
                                                   DataInputStream dataIn, DataOutputStream dataOut )
        {
        // Read the subpopulation number and the number of individuals
        // from the master.
        int subPopNum = -1;
        int numInds = -1;
        boolean countVictoriesOnly = false;
        try
            {
            subPopNum = dataIn.readInt();
            numInds = dataIn.readInt();
            countVictoriesOnly = dataIn.readBoolean();
            }
        catch (IOException e)
            {
            state.output.fatal("Unable to read the subpopulation number from the master:\n"+e);
            }

        // Here we need to know the subpopulation number so as to create the
        // correct type of subpopulation in order to create the correct type
        // of individual.
        Parameter param = new Parameter(P_SUBPOP).push("" + subPopNum);
        Subpopulation subPop = 
            (Subpopulation)(state.parameters.getInstanceForParameterEq(
                                param,null,
                                Subpopulation.class));

        // Setup the subpopulation so that it is in a valid state.
        subPop.setup(state, param);

        // Read the individuals from the stream
        Individual inds[] = new Individual[numInds];
        boolean updateFitness[] = new boolean[numInds];
        try
            {
            for(int i=0;i<inds.length;++i)
                {
                inds[i] = subPop.species.newIndividual(
                    state, subPop, (Fitness)(subPop.f_prototype.protoClone()));
                inds[i].readIndividual(state,dataIn);
                updateFitness[i] = dataIn.readBoolean();
                }
            }
        catch (IOException e)
            {
            state.output.fatal("Unable to read individual from master.");
            }
        catch (CloneNotSupportedException e)
            {
            state.output.fatal("Unable to clone a new fitness.");
            }

        // Evaluate the individual
        // TODO Check to make sure the real problem is an instance of GroupedProblemForm
        ((GroupedProblemForm)(state.evaluator.p_problem)).evaluate( state, inds, updateFitness, countVictoriesOnly, 0 );

        try
            {
            // Return the evaluated individual to the master
            for(int i=0;i<inds.length;i++)
                inds[i].writeIndividual(state, dataOut);
            dataOut.flush();
            }
        catch( IOException e ) { state.output.fatal("Caught fatal IOException\n"+e ); }
        }

    private static void checkpointRandomState(final EvolutionState state,
                                              DataOutputStream dataOut )
        {
        state.output.systemMessage("Checkpointing");

        try
            {
            state.random[0].writeState(dataOut);
            }
        catch (IOException e)
            {
            state.output.fatal("Exception while checkpointing random state:\n"+e);
            }
        }
    }
