package ec.app.regression;
import ec.util.*;
import ec.*;
import ec.gp.*;
import ec.gp.koza.*;

/* 
 * Regression.java
 * 
 * Created: Mon Nov  1 15:46:19 1999
 * By: Sean Luke
 */

/**
 * Regression implements the Symbolic Regression problem.
 *
 <p><b>Parameters</b><br>
 <table>
 <tr><td valign=top><i>base</i>.<tt>data</tt><br>
 <font size=-1>classname, inherits or == ec.app.regression.RegressionData</font></td>
 <td valign=top>(the class for the prototypical GPData object for the Regression problem)</td></tr>
  </table>

 <p><b>Parameter bases</b><br>
 <table>
 <tr><td valign=top><i>base</i>.<tt>data</tt></td>
 <td>species (the GPData object)</td></tr>
 </table>
 *
 * @author Sean Luke
 * @version 1.0 
 */

public class Regression extends GPProblem 
    {
    public static final int NUMINPUTS = 20;
    public static final String P_DATA = "data";

    public double currentValue;
    
    // these are read-only during evaluation-time, so
    // they can be just light-cloned and not deep cloned.
    // cool, huh?
    
    public double inputs[];
    public double outputs[];

    // we'll need to deep clone this one though.
    public RegressionData input;

    public final double func(double x)
	{ return x*x*x*x + x*x*x + x*x + x; }

    public Object protoClone() throws CloneNotSupportedException
	{
	// don't bother copying the inputs and outputs; they're read-only :-)
	// don't bother copying the currentValue; it's transitory
	// but we need to copy our regression data
	Regression myobj = (Regression) (super.protoClone());

	myobj.input = (RegressionData)(input.protoClone());
	return myobj;
	}

    public void setup(final EvolutionState state,
		      final Parameter base)
	{
	// very important, remember this
	super.setup(state,base);

	// we're not really doing any set up here (oh, we might
	// load NUMINPUTS I guess, but I'm lazy).  But this
	// is the right place to compute our inputs so they
	// can be copied with protoClone later
	
	inputs = new double[NUMINPUTS];
	outputs = new double[NUMINPUTS];
	
	for(int x=0;x<NUMINPUTS;x++)
	    {
	    inputs[x] = state.random[0].nextDouble() * 2.0 - 1.0;
	    outputs[x] = func(inputs[x]);
	    state.output.println("{" + inputs[x] + "," + outputs[x] + "},",3000,0);
	    }

	// set up our input -- don't want to use the default base, it's unsafe
	input = (RegressionData) state.parameters.getInstanceForParameterEq(
	    base.push(P_DATA), null, RegressionData.class);
	input.setup(state,base.push(P_DATA));
	}


    public void evaluate(final EvolutionState state, 
			 final Individual[] ind, 
			 final int threadnum)
	{
	for(int x=0;x<ind.length;x++)
	    {
	    if (!ind[x].evaluated)  // don't bother reevaluating
		{
		int hits = 0;
		double sum = 0.0;
		double result;
		for (int y=0;y<NUMINPUTS;y++)
		    {
		    currentValue = inputs[y];
		    ((GPIndividual)ind[x]).trees[0].child.eval(
			state,threadnum,input,stack,((GPIndividual)ind[x]),this);

		    // It's possible to get NaN because cos(infinity) and
		    // sin(infinity) are undefined (hence cos(exp(3000)) zings ya!)
		    // So since NaN is NOT =,<,>,etc. any other number, including
		    // NaN, we're CAREFULLY wording our cutoff to include NaN.
		    // Interesting that this has never been reported before to
		    // my knowledge.

		    final double HIT_LEVEL = 0.01;
		    final double PROBABLY_ZERO = 1.11E-15;
		    final double BIG_NUMBER = 1.0e15;  // the same as lilgp uses

		    result = Math.abs(outputs[y] - input.x);

		    if (! (result < BIG_NUMBER ) )   // *NOT* (input.x >= BIG_NUMBER)
			result = BIG_NUMBER;

		    // very slight math errors can creep in when evaluating
		    // two equivalent by differently-ordered functions, like
		    // x * (x*x*x + x*x)  vs. x*x*x*x + x*x

		    else if (result<PROBABLY_ZERO)  // slightly off
			result = 0.0;
		    
		    if (result <= HIT_LEVEL) hits++;  // whatever!

		    sum += result;		    }
		
		// the fitness better be KozaFitness!
		KozaFitness f = ((KozaFitness)ind[x].fitness);
		f.setFitness(state,(float)sum);
		f.hits = hits;
		ind[x].evaluated = true;
		}
	    }
	}
    }
