package ec.gp.koza;
import ec.*;
import ec.gp.*;
import ec.util.*;

/* 
 * GrowBuilder.java
 * 
 * Created: Thu Oct  7 18:03:49 1999
 * By: Sean Luke
 */


/** GrowBuilder is a GPNodeBuilder which
    implements the GROW tree building method described in Koza I/II.  
    
    <p>GROW works by choosing a random integer <i>d</i> between minDepth and maxDepth, inclusive.  It then grows a tree of depth 1 to <i>d</i> inclusive.

    <p>Actually, claiming to implement the Koza I/II approach is a bit of a fib -- Koza's original code is somewhat ad-hoc.  In Koza, <i>d</i> is always set to maxDepth, and trees may not consist of a single terminal.  This makes strongly-typed implementations unecessarily restrictive in function options.

    <p>This implementation instead follows lil-gp's approach, which is to choose <i>d</i> at random from between minDepth and maxDepth, inclusive, and to allow trees consisting of single terminals. 

    <p>Note that the Koza I/II implementations solely use GROW for subtree mutations.
    
   <p> This algorithm ignores <tt>requestedSize</tt>, so no pipelines can ask it to grow a tree of a specific fixed size.  The algorithm also ignores any user-provided size distributions.

   <p>Common maxDepth values:
   <ul>
   <li>Koza II specifies 6 as its maximum depth for initialization.
   <li>Koza I instead uses 5 as its maximum depth for initialization.
   <li>Koza I and II specify a maximum depth of 5 for subtree mutation.
   <li>lil-gp follows Koza II.
   <li>Suggested default value: 6
   </ul>
   
   <p>Common minDepth values:
   <ul>
   <li>	Because they disallow single terminals, in essence Koza I/II have this
   set at 2, but it's not really.  
   <li> lil-gp uses 2.
   <li> Suggested default value: 2, though you might change it for subtree mutations.
   </ul>

    <p><b>Parameters</b><br>
    <table>
    <tr><td valign=top><i>base</i>.<tt>min-depth</tt><br>
    <font size=-1>int &gt;= 1</font></td>
    <td valign=top>(smallest "maximum" depth the builder may use for building a tree. 2 is the default.)</td></tr>
    
    <tr><td valign=top><i>base</i>.<tt>max-depth</tt><br>
    <font size=-1>int &gt;= <i>base</i>.<tt>min-depth</tt></font></td>
    <td valign=top>(largest "maximum" depth thie builder may use for building a tree. 6 is the default.)</td></tr>
    </table>
    
 <p><b>Default Base</b><br>
 gp.koza.grow

    * @author Sean Luke
    * @version 1.0 
    */



public class GrowBuilder extends GPNodeBuilder
    {
    public static final String P_GROWBUILDER = "grow";
    public static final String P_MAXDEPTH = "max-depth";
    public static final String P_MINDEPTH = "min-depth";

    /** The largest maximum tree depth GROW can specify. */
    public int maxDepth;

    /** The smallest maximum tree depth GROW can specify. */
    public int minDepth;

    public Parameter defaultBase()
	{
	return GPKozaDefaults.base().push(P_GROWBUILDER); 
	}

    public void setup(final EvolutionState state, final Parameter base)
	{
	super.setup(state,base);

	Parameter def = defaultBase();

	// load maxdepth and mindepth, check that maxdepth>0, mindepth>0, maxdepth>=mindepth
	maxDepth = state.parameters.getInt(base.push(P_MAXDEPTH),def.push(P_MAXDEPTH),1);
	if (maxDepth<=0)
	    state.output.fatal("The Max Depth for GrowBuilder must be at least 1.",
			       base.push(P_MAXDEPTH),def.push(P_MAXDEPTH));
	
	minDepth = state.parameters.getInt(base.push(P_MINDEPTH),def.push(P_MINDEPTH),1);
	if (minDepth<=0)
	    state.output.fatal("The Max Depth for GrowBuilder must be at least 1.",
			       base.push(P_MINDEPTH),def.push(P_MINDEPTH));

	if (maxDepth<minDepth)
	    state.output.fatal("Max Depth must be >= Min Depth for GrowBuilder",
			       base.push(P_MAXDEPTH),def.push(P_MAXDEPTH));
	}    

    
    public GPNode newRootedTree(final EvolutionState state,
				final GPType type,
				final int thread,
				final GPNodeParent parent,
				final GPFunctionSet set,
				final int argposition,
				final int requestedSize) throws CloneNotSupportedException
	{
	GPNode n = growNode(state,0,state.random[thread].nextInt(maxDepth-minDepth+1) + minDepth,type,thread,parent,argposition,set);
	return n;
	}


    /** A private function which recursively returns a GROW tree to newRootedTree(...) */
    private GPNode growNode(final EvolutionState state,
			    final int current,
			    final int max,
			    final GPType type,
			    final int thread,
			    final GPNodeParent parent,
			    final int argposition,
			    final GPFunctionSet set) throws CloneNotSupportedException
	{
	if (current+1 >= maxDepth)  // we're at max depth, force a terminal
	    {
	    GPFuncInfo[] nn = set.terminals[type.type];
	    GPNode n = (GPNode)(nn[state.random[thread].nextInt(nn.length)].node.protoClone());
	    n.resetNode(state,thread);  // give ERCs a chance to randomize
	    n.argposition = (byte)argposition;
	    n.parent = parent;
	    return n;
	    }
	else // pick either a terminal or a nonterminal
	    {
	    GPFuncInfo[] nn = set.nodes[type.type];
	    GPNode n = (GPNode)(nn[state.random[thread].nextInt(nn.length)].node.protoClone());
	    n.resetNode(state,thread);  // give ERCs a chance to randomize
	    n.argposition = (byte)argposition;
	    n.parent = parent;

	    // Populate the node...
	    GPType[] childtypes = n.constraints().childtypes;
	    for(int x=0;x<childtypes.length;x++)
		n.children[x] = growNode(state,current+1,max,childtypes[x],thread,n,x,set);

	    return n;
	    }
	}
    }


