package ec.gp.koza;
import ec.*;
import ec.gp.*;
import ec.util.*;
import java.util.Hashtable;
import java.util.Enumeration;

/* 
 * HalfBuilder.java
 * 
 * Created: Thu Oct  7 18:03:49 1999
 * By: Sean Luke
 */

/** HalfBuilder is a GPNodeBuilder which 
    implements the RAMPED HALF-AND-HALF tree building method described in Koza I/II.  

    <p>RAMPED HALF-AND-HALF works by choosing a random integer <i>d</i> between minDepth and maxDepth, inclusive.  It then grows a tree of depth 1 to <i>d</i> inclusive.  (1-pickGrowProbability) of the time (by default, 0.5) it grows a tree using the FULL method, which generates full trees of exactly depth <i>d</i>.  (pickGrowProbability) of the time, it grows a tree using the GROW method, which may generate trees of any size between 1 and <i>d</i> inclusive.

    <p>Actually, claiming to implement the Koza I/II approach is a bit of a fib -- Koza's original code is somewhat ad-hoc.  It uses an odd mechanism for determining <i>d</i>; it hard-codes minDepth to 1, and goes round-robin from minDepth to maxDepth, inclusive, rather than choosing one at random.  Further, it's not legal to create trees consisting of a single terminal in the Koza approach; this means Koza's trees tend to be slightly bigger.    This also makes strongly-typed implementations unecessarily restrictive in function options.

    <p>This implementation instead follows lil-gp's approach, which is to choose <i>d</i> at random from between minDepth and maxDepth, inclusive, and to allow trees consisting of single terminals. 

   <p>Note that the Koza I/II implementations solely use GROW for subtree mutations.
   
   <p> This algorithm ignores <tt>requestedSize</tt>, so no pipelines can ask it to grow a tree of a specific fixed size.  The algorithm also ignores any user-provided size distributions.

   <p>The common value for pickGrowProbability is 0.5.

   <p>Common maxDepth values:
   <ul>
   <li>Koza II specifies 6 as its maximum depth for initialization.
   <li>Koza I instead uses 5 as its maximum depth for initialization.
   <li>Koza I and II specify a maximum depth of 5 for subtree mutation.
   <li>lil-gp follows Koza II.
   <li>Suggested default value: 6
   </ul>
   
   <p>Common minDepth values:
   <ul>
   <li>	Because they disallow single terminals, in essence Koza I/II have this
   set at 2, but it's not really.  
   <li> lil-gp uses 2.
   <li> Suggested default value: 2, though you might change it for subtree mutations.
   </ul>


   <p><b>Parameters</b><br>
   <table>
   <tr><td valign=top><i>base</i>.<tt>growp</tt><br>
   <font size=-1>0.0 &lt;= float &lt;= 1.0</font></td>
   <td valign=top>(the likelihood of choosing GROW (as opposed to FULL)></tr>

   <tr><td valign=top><i>base</i>.<tt>min-depth</tt><br>
   <font size=-1>int &gt;= 1</font></td>
   <td valign=top>(smallest "maximum" depth the builder may use for building a tree.  2 is the default.)</td></tr>
   
   <tr><td valign=top><i>base</i>.<tt>max-depth</tt><br>
   <font size=-1>int &gt;= <i>base</i>.<tt>min-depth</tt></font></td>
   <td valign=top>(largest "maximum" depth the builder may use for building a tree. 6 is the default.)</td></tr>
   </table>
   
   <p><b>Default Base</b><br>
   gp.koza.half

   * @author Sean Luke
   * @version 1.0 
   */


public class HalfBuilder extends GPNodeBuilder
    {
    public static final String P_HALFBUILDER = "half";
    public static final String P_MAXDEPTH = "max-depth";
    public static final String P_MINDEPTH = "min-depth";
    public static final String P_PICKGROWPROBABILITY = "growp";

    /** The largest maximum tree depth RAMPED HALF-AND-HALF can specify. */
    public int maxDepth;

    /** The smallest maximum tree depth RAMPED HALF-AND-HALF can specify. */
    public int minDepth;

    /** The likelihood of using GROW over FULL. */
    public float pickGrowProbability;
    
    public Parameter defaultBase()
	{
	return GPKozaDefaults.base().push(P_HALFBUILDER); 
	}

    public void setup(final EvolutionState state, final Parameter base)
	{
	super.setup(state,base);

	Parameter def = defaultBase();

	// load maxdepth and mindepth, check that maxdepth>0, mindepth>0, maxdepth>=mindepth
	maxDepth = state.parameters.getInt(base.push(P_MAXDEPTH),def.push(P_MAXDEPTH),1);
	if (maxDepth<=0)
	    state.output.fatal("The Max Depth for HalfBuilder must be at least 1.",
			       base.push(P_MAXDEPTH),def.push(P_MAXDEPTH));
	
	minDepth = state.parameters.getInt(base.push(P_MINDEPTH),def.push(P_MINDEPTH),1);
	if (minDepth<=0)
	    state.output.fatal("The Max Depth for HalfBuilder must be at least 1.",
			       base.push(P_MINDEPTH),def.push(P_MINDEPTH));

	if (maxDepth<minDepth)
	    state.output.fatal("Max Depth must be >= Min Depth for HalfBuilder",
			       base.push(P_MAXDEPTH),def.push(P_MAXDEPTH));

	pickGrowProbability = state.parameters.getFloat(
	    base.push(P_PICKGROWPROBABILITY),
	    def.push(P_PICKGROWPROBABILITY),0.0f,1.0f);
	if (pickGrowProbability < 0.0f)
	    state.output.fatal("The Pick-Grow Probability for HalfBuilder must be a floating-point value between 0.0 and 1.0 inclusive.", base.push(P_MAXDEPTH),def.push(P_MAXDEPTH));
	}


    
    public GPNode newRootedTree(final EvolutionState state,
				final GPType type,
				final int thread,
				final GPNodeParent parent,
				final GPFunctionSet set,
				final int argposition,
				final int requestedSize) throws CloneNotSupportedException
	{
	if (state.random[thread].nextFloat() < pickGrowProbability)
	    return growNode(state,0,state.random[thread].nextInt(maxDepth-minDepth+1) + minDepth,type,thread,parent,argposition,set);
	else
	    return fullNode(state,0,state.random[thread].nextInt(maxDepth-minDepth+1) + minDepth,type,thread,parent,argposition,set);
	}

    /** A private method which recursively returns a FULL tree to newRootedTree(...) */
    private GPNode fullNode(final EvolutionState state,
			    final int current,
			    final int max,
			    final GPType type,
			    final int thread,
			    final GPNodeParent parent,
			    final int argposition,
			    final GPFunctionSet set) throws CloneNotSupportedException
	{
	// Pick a random node from Hashtable for a given type --
	// we assume it's been pre-checked for invalid type situations
	
	if (current+1 >= maxDepth)  // we're at max depth, force a terminal
	    {
	    GPFuncInfo[] nn = set.terminals[type.type];
	    GPNode n = (GPNode)(nn[state.random[thread].nextInt(nn.length)].node.protoClone());
	    n.resetNode(state,thread);  // give ERCs a chance to randomize
	    n.argposition = (byte)argposition;
	    n.parent = parent;
	    return n;
	    }
	else // we're not at max depth, force a nonterminal if you can
	    {
	    GPFuncInfo[] nn = set.nonterminals[type.type];
	    if (nn==null || nn.length ==0)  /* no nonterminals, hope the guy
					       knows what he's doing! */
		nn = set.terminals[type.type];
	    
	    GPNode n = (GPNode)(nn[state.random[thread].nextInt(nn.length)].node.protoClone());
	    n.resetNode(state,thread);  // give ERCs a chance to randomize
	    n.argposition = (byte)argposition;
	    n.parent = parent;

	    // Populate the node...
	    GPType[] childtypes = n.constraints().childtypes;
	    for(int x=0;x<childtypes.length;x++)
		n.children[x] = fullNode(state,current+1,max,childtypes[x],thread,n,x,set);

	    return n;
	    }
	}

    /** A private method which recursively returns a GROW tree to newRootedTree(...) */
    private GPNode growNode(final EvolutionState state,
			    final int current,
			    final int max,
			    final GPType type,
			    final int thread,
			    final GPNodeParent parent,
			    final int argposition,
			    final GPFunctionSet set) throws CloneNotSupportedException
	{
	// Pick a random node from Hashtable for a given type --
	// we assume it's been pre-checked for invalid type situations
	
	if (current+1 >= maxDepth)  // we're at max depth, force a terminal
	    {
	    GPFuncInfo[] nn = set.terminals[type.type];
	    GPNode n = (GPNode)(nn[state.random[thread].nextInt(nn.length)].node.protoClone());
	    n.resetNode(state,thread);  // give ERCs a chance to randomize
	    n.argposition = (byte)argposition;
	    n.parent = parent;
	    return n;
	    }
	else // pick either a terminal or a nonterminal
	    {
	    GPFuncInfo[] nn = set.nodes[type.type];
	    GPNode n = (GPNode)(nn[state.random[thread].nextInt(nn.length)].node.protoClone());
	    n.resetNode(state,thread);  // give ERCs a chance to randomize
	    n.argposition = (byte)argposition;
	    n.parent = parent;

	    // Populate the node...
	    GPType[] childtypes = n.constraints().childtypes;
	    for(int x=0;x<childtypes.length;x++)
		n.children[x] = growNode(state,current+1,max,childtypes[x],thread,n,x,set);

	    return n;
	    }
	}
    }


