package ec.steadystate;
import ec.simple.*;
import ec.*;
import ec.util.*;

/* 
 * SteadyStateBreeder.java
 * 
 * Created: Tue Aug 10 21:00:11 1999
 * By: Sean Luke
 */

/**
 * A SteadyStateBreeder is an extension of SimpleBreeder which works in conjunction
 * with SteadyStateEvolutionState to breed individuals using a steady-state breeding
 * method.
 *
 * SteadyStateBreeder marks <i>numnewinds</i> individuals for death in each
 * subpopulation.  It then replaces those individuals in a subpopulation
 * with new individuals bred from the rest of the subpopulation.
 *
 * @author Sean Luke
 * @version 1.0 
 */

public class SteadyStateBreeder extends SimpleBreeder
    {
    /** If state.generation is 0, this acts exactly like SimpleBreeder.
	Else, it only breeds one new individual per subpopulation, to 
	place in position 0 of the subpopulation.  
    */
    BreedingPipeline[][] bp;

    public SteadyStateBreeder() { bp = null; }


    /** Called to check to see if the breeding sources are correct */
    public void sourcesAreProperForm(final SteadyStateEvolutionState state,
				     final BreedingPipeline[][] bp)
	{
	for(int x=0;x<bp.length;x++)
	    for(int y=0;y < bp[x].length; y++)
		{
		if (!(bp[x][y] instanceof SteadyStateBreedingSourceForm))
		    state.output.error("Breeding Pipeline " + y + " of subpopulation " + x + " is not of SteadyStateBreedingSourceForm");
		((SteadyStateBreedingSourceForm)(bp[x][y])).sourcesAreProperForm(state);
		}
	state.output.exitIfErrors();
	}

    /** Called whenever individuals have been replaced by new
	individuals in the population. */
    public void individualReplaced(final SteadyStateEvolutionState state,
				   final int subpopulation,
				   final int thread,
				   final int individual)
	{
	for(int x=0;x<bp.length;x++)
	    for(int y=0;y < bp[x].length; y++)
		((SteadyStateBreedingSourceForm)bp[x][y]).
		    individualReplaced(state,subpopulation,thread,individual);
	}


    public Population breedPopulation(EvolutionState state) throws CloneNotSupportedException
	{
	final SteadyStateEvolutionState st = (SteadyStateEvolutionState) state;


	if (st.generation==0) // first time
	    {
	    
	    super.breedPopulation(st);
	    
	    // Load my steady-state breeding pipelines
	    
	    if (bp == null)
		{
		// set up the breeding pipelines
		bp = new BreedingPipeline[st.population.subpops.length][];
		for(int pop=0;pop<bp.length;pop++)
		    {
		    bp[pop] = st.population.subpops[pop].species.pipelines(st);
		    for(int x=0;x<bp[pop].length;x++)
			if (!bp[pop][x].produces(st,st.population,pop,0))
			    st.output.error("Breeding pipeline " + x + " of subpopulation " + pop + " does not produce individuals of the expected species " + st.population.subpops[pop].species.getClass().getName() + " and with the expected Fitness class " + st.population.subpops[pop].f_prototype.getClass().getName());
		    }
		// are they of the proper form?
		sourcesAreProperForm(st,bp);
		
		// warm them up
		for(int pop=0;pop<bp.length;pop++)
		    for(int x=0;x<bp[pop].length;x++)
			bp[pop][x].prepareToProduce(state,pop,0);
		}
	    }

	// yes, yes, this is after creating bp, so it's less efficient,
	// but safer because the sourcesAreProperForm() check is done before this


	// mark individuals for death

	for (int pop = 0; pop<st.population.subpops.length;pop++)
	    {
	    SelectionMethod deselector = 
		((SteadyStateSpeciesForm)(st.population.subpops[pop].species)).deselector();
	    st.newIndividuals[pop] = deselector.produce(pop,st,0);
	    }

	// create new individuals

	for(int pop=0;pop<st.population.subpops.length;pop++)
	    st.newIndividuals[pop]= bp[pop][
		BreedingSource.pickRandom(
		    bp[pop],state.random[0].nextFloat())].
		produce(
		    1,1,st.newIndividuals[pop],pop,
		    st.population.subpops[pop].individuals,st,0);

	return st.population;
	}
    }
