package ec.util;

/* 
 * RandomChoice.java
 * 
 * Created: Tue Feb  1 22:41:36 2000
 * By: Sean Luke
 */

/**
 * @author Sean Luke
 * @version 1.0 
 */

public class RandomChoice  
    {

    /** Normalizes probabilities, then converts them into continuing
	sums.  This prepares them for being usable in pickFromDistribution.
	If the probabilities are all 0, or any of them are negative,
	then an ArithmeticException is thrown.
	For example, 
	{0.6, 0.4, 0.2, 0.8} -> {0.3, 0.2, 0.1, 0.4} -> {0.3, 0.5, 0.6, 1.0} */

    public static void organizeDistribution(final float[] probabilities)
	{
	// first normalize
	double sum=0.0;
	for(int x=0;x<probabilities.length;x++)
	    {
	    if (probabilities[x]<0.0)
		 throw new ArithmeticException("Distribution has negative probabilities");
	    sum += probabilities[x];
	    }
	if (sum==0.0) throw new ArithmeticException("Distribution has all 0 probabilities");
	for(int x=0;x<probabilities.length;x++)
	    probabilities[x] /= sum;

	// now sum
	sum=0.0;
	for(int x=0;x<probabilities.length;x++)
	    {
	    sum += probabilities[x];
	    probabilities[x] = (float)sum;
	    }
	// just in case...
	probabilities[probabilities.length-1] = 1.0f;
	}

    /** Normalizes probabilities, then converts them into continuing
	sums.  This prepares them for being usable in pickFromDistribution.
	If the probabilities are all 0, or any of them are negative,
	then an ArithmeticException is thrown.
	For example, 
	{0.6, 0.4, 0.2, 0.8} -> {0.3, 0.2, 0.1, 0.4} -> {0.3, 0.5, 0.6, 1.0} */

    public static void organizeDistribution(final double[] probabilities)
	{
	// first normalize
	double sum=0.0;
	for(int x=0;x<probabilities.length;x++)
	    {
	    if (probabilities[x]<0.0)
		 throw new ArithmeticException("Distribution has negative probabilities");
	    sum += probabilities[x];
	    }
	if (sum==0.0) throw new ArithmeticException("Distribution has all 0 probabilities");
	for(int x=0;x<probabilities.length;x++)
	    probabilities[x] /= sum;

	// now sum
	sum=0.0;
	for(int x=0;x<probabilities.length;x++)
	    {
	    sum += probabilities[x];
	    probabilities[x] = sum;
	    }
	// just in case...
	probabilities[probabilities.length-1] = 1.0f;
	}

    /** Normalizes the probabilities associated
	with an array of objects, then converts them into continuing
	sums.  This prepares them for being usable in pickFromDistribution.
	If the probabilities are all 0, or any of them are negative,
	then an ArithmeticException is thrown.
	For example, 
	{0.6, 0.4, 0.2, 0.8} -> {0.3, 0.2, 0.1, 0.4} -> {0.3, 0.5, 0.6, 1.0} 
	The probabilities are retrieved and set using chooser.*/

    public static void organizeDistribution(final Object[] objs, 
					    final RandomChoiceChooser chooser)
	{
	// first normalize
	double sum=0.0;
	for(int x=0;x<objs.length;x++)
	    {
	    if (chooser.getProbability(objs[x])<0.0)
		 throw new ArithmeticException("Distribution has negative probabilities");
	    sum += chooser.getProbability(objs[x]);
	    }
	if (sum==0.0) throw new ArithmeticException("Distribution has all 0 probabilities");
	for(int x=0;x<objs.length;x++)
	    chooser.setProbability(objs[x],
			   (float)(chooser.getProbability(objs[x]) / sum));

	// now sum
	sum=0.0;
	for(int x=0;x<objs.length;x++)
	    {
	    sum += chooser.getProbability(objs[x]);
	    chooser.setProbability(objs[x],(float)sum);
	    }
	// just in case...
	chooser.setProbability(objs[objs.length-1], 1.0f);
	}

    /** Picks a random item from an array of probabilities,
        normalized and summed as follows:  For example,
        if four probabilities are {0.3, 0.2, 0.1, 0.4}, then
        they should get normalized and summed by the outside owners
        as: {0.3, 0.5, 0.6, 1.0}.  If probabilities.length < checkboundary,
	then a linear search is used, else a binary search is used. */
    
    public static int pickFromDistribution(final float[] probabilities,
					   final float prob, final int checkboundary)
        {
        if (probabilities.length==1) // quick 
            return 0;
        else if (probabilities.length<checkboundary)
            {
            // simple linear scan
            for(int x=0;x<probabilities.length-1;x++)
                if (probabilities[x]>prob)
                    return x; 
            return probabilities.length-1;
            }
        else
            {
            // binary search
            int top = probabilities.length-1;
            int bottom = 0;
            int cur;

            while(top!=bottom)
                {
                cur = (top + bottom) / 2; // integer division

                if (probabilities[cur] > prob)
                    if (cur==0 || probabilities[cur-1] <= prob)
                        return cur;
                    else // step down
                        top = cur;
                else if (cur==probabilities.length-1) // oops
                    return cur;
                else if (bottom==cur) // step up
		    bottom++;  // (8 + 9)/2 = 8
		else
                    bottom = cur;  // (8 + 10) / 2 = 9
                }
            return bottom;  // oops
            }
        }



    /** Picks a random item from an array of probabilities,
        normalized and summed as follows:  For example,
        if four probabilities are {0.3, 0.2, 0.1, 0.4}, then
        they should get normalized and summed by the outside owners
        as: {0.3, 0.5, 0.6, 1.0}.  If probabilities.length < checkboundary,
	then a linear search is used, else a binary search is used. */
    
    public static int pickFromDistribution(final double[] probabilities,
					   final double prob, final int checkboundary)
        {
        if (probabilities.length==1) // quick 
            return 0;
        else if (probabilities.length<checkboundary)
            {
            // simple linear scan
            for(int x=0;x<probabilities.length-1;x++)
                if (probabilities[x]>prob)
                    return x; 
            return probabilities.length-1;
            }
        else
            {
            // binary search
            int top = probabilities.length-1;
            int bottom = 0;
            int cur;

            while(top!=bottom)
                {
                cur = (top + bottom) / 2; // integer division

                if (probabilities[cur] > prob)
                    if (cur==0 || probabilities[cur-1] <= prob)
                        return cur;
                    else // step down
                        top = cur;
                else if (cur==probabilities.length-1) // oops
                    return cur;
                else if (bottom==cur) // step up
		    bottom++;  // (8 + 9)/2 = 8
		else
                    bottom = cur;  // (8 + 10) / 2 = 9
                }
            return bottom;  // oops
            }
        }


    /** Picks a random item from an array of objects, each with an
	associated probability that is accessed by taking an object
	and passing it to chooser.getProbability(obj).  The objects'
	probabilities are 
        normalized and summed as follows:  For example,
        if four probabilities are {0.3, 0.2, 0.1, 0.4}, then
        they should get normalized and summed by the outside owners
        as: {0.3, 0.5, 0.6, 1.0}.  If probabilities.length < checkboundary,
	then a linear search is used, else a binary search is used. */
    
    public static int pickFromDistribution(final Object[] objs, 
					   final RandomChoiceChooser chooser,
					   final float prob, final int checkboundary)
        {
        if (objs.length==1) // quick 
            return 0;
        else if (objs.length<checkboundary)
            {
            // simple linear scan
            for(int x=0;x<objs.length-1;x++)
                if (chooser.getProbability(objs[x]) >prob)
                    return x; 
            return objs.length-1;
            }
        else
            {
            // binary search
            int top = objs.length-1;
            int bottom = 0;
            int cur;

            while(top!=bottom)
                {
                cur = (top + bottom) / 2; // integer division

                if (chooser.getProbability(objs[cur]) > prob)
                    if (cur==0 || chooser.getProbability(objs[cur-1]) <= prob)
                        return cur;
                    else // step down
                        top = cur;
                else if (cur==objs.length-1) // oops
                    return cur;
                else if (bottom==cur) // step up
		    bottom++;  // (8 + 9)/2 = 8
		else
                    bottom = cur;  // (8 + 10) / 2 = 9
                }
            return bottom;  // oops
            }
        }

    }




