package ec;
import ec.util.*;
import java.io.*;

/* 
 * Species.java
 * 
 * Created: Tue Aug 10 20:31:50 1999
 * By: Sean Luke
 */

/**
 * Species is a prototype which defines the features for a set of individuals
 * in the population.  Typically, individuals may breed if they belong to the
 * same species (but it's not a hard-and-fast rule).  Each Subpopulation has
 * one Species object which defines the species for individuals in that
 * Subpopulation.
 *
 * <p>Species are generally responsible for creating individuals, through
 * their newIndividual(...) method.  This method usually clones its prototypical
 * individual
 * and makes some additional modifications to the clone, then returns it.
 *
 * <p>Species also contain an array of <i>breeding pipelines</i>, which are
 * mechanisms by which individuals may breed to make new individuals in a
 * new population.  For example, selection followed by crossover followed by
 * a small amount of mutation might be a breeding pipeline.  Breeding Pipelines
 * are per-species.  The ones stored in a species are prototypes, and at breeding
 * time, they are generally cloned, one set of BreedingPipelines per thread,
 * to breed the new population.

 <p><b>Parameters</b><br>
 <table>
 <tr><td valign=top><i>base</i>.<tt>ind</tt><br>
 <font size=-1>classname, inherits and != ec.Individual</font></td>
 <td valign=top>(the class for the prototypical individual for the species)</td></tr>

 <tr><td valign=top><i>base</i>.<tt>numpipes</tt><br>
 <font size=-1>int &gt;= 1</font></td>
 <td valign=top>(total number of breeding pipelines for the species)</td></tr>

 <tr><td valign=top><i>base</i>.<tt>pipe</tt><i>.n</i><br>
 <font size=-1>classname, inherits and != ec.BreedingPipeline</font></td>
 <td valign=top>(the class for prototypical Breeding Pipeline #<i>n</i>)</td></tr>

 </table>


 <p><b>Parameter bases</b><br>
 <table>
 <tr><td valign=top><i>base</i>.<tt>ind</tt></td>
 <td>i_prototype (the prototypical individual)</td></tr>

 <tr><td valign=top><i>base</i>.<tt>pipe</tt><i>.n</i></td>
 <td>p_breedingpipelines[<i>n</i>] (breeding pipeline prototype <i>n</i>)</td></tr>

 </table>



 * @author Sean Luke
 * @version 1.0 
 */

public abstract class Species implements Prototype
    {
    public static final String P_INDIVIDUAL = "ind";
    public static final String P_PIPE = "pipe";
    public static final String P_NUMPIPES = "numpipes";

    /** The prototypical individual for this species.       
     */

    public Individual i_prototype;
    
    /** This can be an array of breeding pipelines which by default
	is the set of breeding pipelines for this Species. 
	
	<p> When you set this up, you must also manually 
	set the default probabilities for this breeding pipeline array.
	To do this, set the <i>probability</i> field of each pipeline
        in the array to some value between 0.0 and 1.0, representing its
        probability of being chosen over other pipelines.  These values
        don't have to be normalized -- that'll be done elsewhere for you
        anyway. 

	<p> The breeding pipelines here won't be directly used; instead,
	they'll be copied first (with protoClone()), and the copies will
	be used instead.  They may be copied more than once.*/
    public BreedingPipeline[] p_breedingpipelines;

    public Object protoClone() throws CloneNotSupportedException
	{
	Species myobj = (Species) (super.clone());
	myobj.i_prototype = (Individual) i_prototype.protoClone();
	myobj.p_breedingpipelines = new BreedingPipeline[p_breedingpipelines.length];
	for(int x=0;x<p_breedingpipelines.length;x++)
	    myobj.p_breedingpipelines[x] = (BreedingPipeline)(p_breedingpipelines[x].protoClone());
	return myobj;
	} 

    public final Object protoCloneSimple()
	{
	try { return protoClone(); }
	catch (CloneNotSupportedException e) 
	    { throw new InternalError(); } // never happens
	} 


    /** override this to provide a brand-new individual to fill in a population. The CloneNotSupportedException permits you to use protoClone() rather than protoCloneSimple(), for efficiency gains. */

    public abstract Individual newIndividual(final EvolutionState state,
					     final Subpopulation _population, 
					     final Fitness _fitness) 
	    throws CloneNotSupportedException;
    
    /** The default version of setup(...) loads requested pipelines and calls setup(...) on them and normalizes their probabilities.
    @see Prototype#setup(EvolutionState,Parameter)
    */

    /** override this to provide an individual read from a file; the individual will
	appear as it was written by printIndividual(...).  You should read and
	set up the fitness as well.  Don't close the file. */

    public abstract Individual newIndividual(final EvolutionState state,
					     final Subpopulation _population,
					     final Fitness _fitness,
					     final int thread,
					     final LineNumberReader reader)
	    throws IOException, CloneNotSupportedException;


    public void setup(final EvolutionState state, final Parameter base)
	{
	Parameter def = defaultBase();

	int numpipes = state.parameters.getInt(
	    base.push(P_NUMPIPES),def.push(P_NUMPIPES), 1);  // must have at least 1 pipe
	if (numpipes<=0)
	    state.output.fatal(
		"Cannot determine the number of breeding pipelines for the species.",
		base.push(P_NUMPIPES),def.push(P_NUMPIPES));

	p_breedingpipelines = new BreedingPipeline[numpipes];

	// load 'em and set 'em up
	float prob;
	float tot = 0.0f;
	for(int x=0;x<numpipes;x++)
	    {
	    Parameter pp = base.push(P_PIPE).push(""+x);
	    Parameter dd = def.push(P_PIPE).push(""+x);
	    p_breedingpipelines[x] = (BreedingPipeline)(
		state.parameters.getInstanceForParameter(
		    pp,dd,BreedingPipeline.class));

	    p_breedingpipelines[x].setup(state,pp);
	    if (p_breedingpipelines[x].probability<0.0) // null checked from state.output.error above
		state.output.error("Pipe #" + x + " must have a probability >= 0.0",base,def);
	    tot += p_breedingpipelines[x].probability;
	    }
	// I promised over in BreedingSource.java that this method would get called.
	state.output.exitIfErrors();

	if (tot<=0.0)
	    state.output.fatal("Pipes sum to <= 0.0",base,def); 

/*
	// normalize
	if (tot!=1.0)
	    {
	    state.output.message("Species " + getClass().getName()+ " has a probabilities which do not sum to 1.0.  Normalizing...");
	    for(int x=0;x<numpipes;x++)
		p_breedingpipelines[x].probability /= tot;
	    }
	
	// totalize
	float tmp = 0.0f;
	for(int x=0;x<numpipes-1;x++)
	    { 
	    tmp += p_breedingpipelines[x].probability; 
	    p_breedingpipelines[x].probability = tmp;
	    }
	p_breedingpipelines[numpipes-1].probability = 1.0f;
*/

	// normalize and totalize
	BreedingSource.setupProbabilities(p_breedingpipelines);


	// load our individual prototype
	i_prototype = (Individual)(state.parameters.getInstanceForParameter(base.push(P_INDIVIDUAL),def.push(P_INDIVIDUAL),Individual. class));
	i_prototype.setup(state,base.push(P_INDIVIDUAL));
	}

    
    /** Returns a freshly cloned set of pipelines for a thread to use. */

    public BreedingPipeline[] pipelines(final EvolutionState state)
	{
	if (p_breedingpipelines!=null && p_breedingpipelines.length > 0)
	    {
	    BreedingPipeline[] nbp = new BreedingPipeline[p_breedingpipelines.length];
	    for(int x=0;x<nbp.length;x++)
		nbp[x] = (BreedingPipeline)(p_breedingpipelines[x].protoCloneSimple());
	    return nbp;
	    }
	else state.output.fatal("Breeding pipelines have not been loaded yet for species " + getClass().getName() + ".");
	return null;
	}
    }


