
/* 
 * GPTree.java
 * 
 * Created: Sat Jan 22 19:20:13 2000
 * By: Sean Luke
 */

/**
 * @author Sean Luke
 * @version 1.0 
 */

package ec.gp;
import ec.*;
import ec.util.*;
import ec.EvolutionState;
import java.io.*;

/* 
 * GPTree.java
 * 
 * Created: Fri Aug 27 17:14:02 1999
 * By: Sean Luke
 */

/**
 * GPTree is a GPNodeParent which holds the root GPNode of a tree
 * of GPNodes.  GPTrees typically fill out an array held in a GPIndividual
 * (their "owner") and their roots are evaluated to evaluate a Genetic
 * programming tree.
 *
 * GPTrees also have <i>constraints</i>, which are shared, and define items
 * shared among several GPTrees.
 *

 <p><b>Parameters</b><br>
 <table>
 <tr><td valign=top><i>base</i>.<tt>tc</tt><br>
 <font size=-1>String</font></td>
 <td valign=top>(The tree's constraints)</td></tr>
 </table>

 <p><b>Default Base</b><br>
 gp.tree

 * @author Sean Luke
 * @version 1.0 
 */

public class GPTree implements GPNodeParent
    {
    public static final String P_TREE = "tree";
    public static final String P_TREECONSTRAINTS = "tc";
    public static final int NO_TREENUM = -1;

    /** the root GPNode in the GPTree */
    public GPNode child;

    /** the owner of the GPTree */
    public GPIndividual owner;

    /** constraints on the GPTree  -- don't access the constraints through
     this variable -- use the constraints() method instead, which will give
     the actual constraints object. */
    public byte constraints;

    public final GPTreeConstraints constraints() 
	{ return GPTreeConstraints.constraints[constraints]; }

    public Parameter defaultBase()
	{
	return GPDefaults.base().push(P_TREE);
	}

    /** Returns true if I am "genetically" the same as tree,
	though we may have different owners. */
    public boolean treeEquals(GPTree tree)
	{
	return child.rootedTreeEquals(tree.child);
	}

    /** Returns a hash code for comparing different GPTrees.  In
	general, two trees which are treeEquals(...) should have the
	same hash code. */
    public int treeHashCode()
	{
	return child.rootedTreeHashCode();
	}

    /** Proto-clones the tree but does NOT deep clone it.  The 
	tree's child is still the same, and not a copy. */
    public Object protoClone() throws CloneNotSupportedException
	{ 
	return super.clone();
	}

    public final Object protoCloneSimple()
	{
	try { return protoClone(); }
	catch (CloneNotSupportedException e) 
	    { throw new InternalError(); } // never happens
	} 

    /** An expensive function which determines my tree number -- only
	use for errors, etc. Returns ec.gp.GPTree.NO_TREENUM if the 
	tree number could not be
	determined (might happen if it's not been assigned yet). */
    public int treeNumber()
	{
	if (owner==null) return NO_TREENUM;
	if (owner.trees==null) return NO_TREENUM;
	for(int x=0;x<owner.trees.length;x++)
	    if (owner.trees[x]==this) return x;
	return NO_TREENUM;
	}


    /** Sets up a prototypical GPTree with those features it shares with
	other GPTrees in its position in its GPIndividual, and nothhing more.

	This must be called <i>after</i> the GPTypes and GPNodeConstraints 
	have been set up.  Presently they're set up in GPInitializer,
	which gets called before this does, so we're safe. */
    public void setup(final EvolutionState state, final Parameter base)
	{
	Parameter def = defaultBase();

	// determine my constraints -- at this point, the constraints should have been loaded.
	String s = state.parameters.getString(base.push(P_TREECONSTRAINTS),
					      def.push(P_TREECONSTRAINTS));
	if (s==null)
	    state.output.fatal("No tree constraints are defined for the GPTree " + base + ".");
	else constraints = GPTreeConstraints.constraintsFor(s,state).constraintNumber;
	
	state.output.exitIfErrors();  // because I promised
	// we're not loading the nodes at this point
	}



    /** Prints out the tree in single-line fashion suitable for reading
	in later by computer. O(n). 
	The default version of this method simply calls child's 
	printRootedTree(...) method. */

    public void printTree(final EvolutionState state, final int log,
			  final int verbosity)
	{
	child.printRootedTree(state,log,verbosity,0);
	}

    /** Prints out the tree in single-line fashion suitable for reading
	in later by computer. O(n). 
	The default version of this method simply calls child's 
	printRootedTree(...) method. */

    public void printTree(final EvolutionState state,
			  final int thread,
			  final PrintWriter writer)
	{
	child.printRootedTree(state,thread,writer,0);
	}

    /** Reads in the tree from a form printed by printTree. */
    public void readTree(final EvolutionState state, final int thread,
			 final LineNumberReader reader) throws IOException, CloneNotSupportedException
	{
	int linenumber = reader.getLineNumber();

	// the next line will be the child
	String s = reader.readLine();
	if (s==null)  // uh oh
	    state.output.fatal("Reading Line " + linenumber + ": " +
			       "No Tree found.");
	else
	    child = child.readRootedTree(linenumber,new DecodeReturn(s),
					 constraints().treetype,
					 constraints().functionset,this,0,state,thread);
	}

    /** Prints out the tree in a readable Lisp-like fashion. O(n). 
	The default version of this method simply calls child's 
	printRootedTreeForHumans(...) method. */
    
    public void printTreeForHumans(final EvolutionState state, final int log,
				   final int verbosity)
	{
	child.printRootedTreeForHumans(state,log,verbosity,0,0);
	}


    /** Builds a new randomly-generated rooted tree and attaches it to the GPTree. */

    public void buildTree(final EvolutionState state, final int thread) throws CloneNotSupportedException
	{
	child = constraints().init.newRootedTree(state,
						 constraints().treetype,
						 thread,
						 this,
						 constraints().functionset,
						 0,
						 GPNodeBuilder.NOSIZEGIVEN);
	}
    }
