package ec.gp.koza;
import ec.*;
import ec.util.*;
import ec.gp.*;
import java.io.Serializable;

/* 
 * ReproductionPipeline.java
 * 
 * Created: Thu Oct 14 20:55:48 1999
 * By: Sean Luke
 */

/**
 * ReproductionPipeline is a GPBreedingPipelien which
 * Implements the Reproduction operator as described in Koza I.  It's
 * very simple -- an individual is selected and a copy is added to the
 * new population.
 *
 * <p> This pipeline produces 1 new individual per produce(...) call.
 *
 <p><b>Typical Number of Individuals Produced Per <tt>produce(...)</tt> call</b><br>
 1

 <p><b>Number of Sources</b><br>
 1

 <p><b>Default Base</b><br>
 gp.koza.reproduce (not that it matters)

 * @author Sean Luke
 * @version 1.0 
 */

public class ReproductionPipeline extends GPBreedingPipeline
    {
    public static final String P_REPRODUCE = "reproduce";
    public static final int INDS_PRODUCED = 1;
    public static final int NUM_SOURCES = 1;
    
    public Parameter defaultBase() { return GPKozaDefaults.base().push(P_REPRODUCE); }

    public int numSources() { return NUM_SOURCES; }

    /** Returns 1 */
    public int typicalIndsProduced() { return INDS_PRODUCED; }


    public int produce(final int min, 
		       final int max, 
		       final int start,
		       final int subpopulation,
		       final Individual[] inds,
		       final EvolutionState state,
		       final int thread) throws CloneNotSupportedException
	{
	// how many individuals should we make?
	int n = INDS_PRODUCED;
	if (n < min) n = min;
	if (n > max) n = max;

	// grab n individuals from our source and stick 'em right into inds.
	// we'll modify them from there
	sources[0].produce(n,n,start,subpopulation,inds,state,thread);

	// now let's reproduce 'em
	for(int q=start; q < n+start; q++)
	    {
	
	    // couldn't be simpler.... though we're not protoing off of the 
	    // species prototye right now, but instead directly off the individual
	    // GPIndividual j = (GPIndividual)(s.population.subpops[subpopulation].species.i_prototype.protoClone());

	    GPIndividual i = (GPIndividual)inds[q];
	    GPIndividual j = (GPIndividual)(i.protoClone());

	    j.trees = new GPTree[((GPIndividual)i).trees.length];
	    // don't reset the evaluated flag -- let it get copied over, after all,
	    // we've already evaluated this individual.
	    
	    for(int x=0;x<j.trees.length;x++)
		{
		j.trees[x] = (GPTree)(i.trees[x].protoClone());
		j.trees[x].child = i.trees[x].child.cloneReplacing();
		j.trees[x].child.parent = j.trees[x];
		j.trees[x].child.argposition = 0;
		}
	    
	    // add the new individual, replacing its previous source
	    inds[q] = j;
	    }
	return n;
	}
    }
