package ec.simple;
import ec.BreedingPipeline;
import ec.Breeder;
import ec.EvolutionState;
import ec.Population;
import ec.util.Parameter;
import ec.BreedingSource;

/* 
 * SimpleBreeder.java
 * 
 * Created: Tue Aug 10 21:00:11 1999
 * By: Sean Luke
 */

/**
 * Breeds each subpopulation separately, with no inter-population exchange,
 * and using a generational approach.  A SimpleBreeder may have multiple
 * threads; it divvys up a subpopulation into chunks and hands one chunk
 * to each thread to populate.  One array of BreedingPipelines is obtained
 * from a population's Species for each operating breeding thread.
 *
 *
 * @author Sean Luke
 * @version 1.0 
 */

public class SimpleBreeder extends Breeder
    {

    public void setup(final EvolutionState state, final Parameter base) { }

    /** A simple breeder that doesn't attempt to do any cross-
	population breeding.  Basically it applies pipelines,
	one per thread, to various subchunks of a new population. */
    public Population breedPopulation(EvolutionState state) throws CloneNotSupportedException
	{
	int numinds[][] = 
	    new int[state.breedthreads][state.population.subpops.length];
	int from[][] = 
	    new int[state.breedthreads][state.population.subpops.length];
	    
	Population newpop = (Population) state.population.emptyClone();
	    
	for(int y=0;y<state.breedthreads;y++)
	    for(int x=0;x<state.population.subpops.length;x++)
		{
		// figure numinds
		if (y<state.breedthreads-1) // not last one
		    numinds[y][x]=
			state.population.subpops[x].individuals.length/
			state.breedthreads;
		else // in case we're slightly off in division
		    numinds[y][x]=
			state.population.subpops[x].individuals.length/
			state.breedthreads +
			    
			(state.population.subpops[x].individuals.length -
			 (state.population.subpops[x].individuals.length /
			  state.breedthreads)  // note integer division
			 *state.breedthreads);			 
		    
		// figure from
		from[y][x]=
		    (state.population.subpops[x].individuals.length/
		     state.breedthreads) * y;
		}
	    
	if (state.breedthreads==1)
	    {
	    breedPopChunk(newpop,state,numinds[0],from[0],0);
	    }
	else
	    {
	    Thread[] t = new Thread[state.breedthreads];
		
	    // start up the threads
	    for(int y=0;y<state.breedthreads;y++)
		{
		SimpleBreederThread r = new SimpleBreederThread();
		r.threadnum = y;
		r.newpop = newpop;
		r.numinds = numinds[y];
		r.from = from[y];
		r.me = this;
		r.state = state;
		t[y] = new Thread(r);
		t[y].start();
		}
		
	    // gather the threads
	    for(int y=0;y<state.breedthreads;y++) try
		{
		t[y].join();
		}
	    catch(InterruptedException e)
		{
		state.output.fatal("Whoa! The main breeding thread got interrupted!  Dying...");
		}
	    }
	return newpop;
	}

    /** A private helper function for breedPopulation which breeds a chunk
	of individuals in a subpopulation for a given thread.
	Although this method is declared
	public (for the benefit of a private helper class in this file),
	you should not call it. */

    public void breedPopChunk(Population newpop, EvolutionState state,
			      int[] numinds, int[] from, int threadnum) throws CloneNotSupportedException
	{
	for(int subpop=0;subpop<newpop.subpops.length;subpop++)
	    {
	    BreedingPipeline[] bp = newpop.subpops[subpop].species.pipelines(state);
		
	    // check to make sure that the breeding pipeline produces
	    // the right kind of individuals.  Don't want a mistake there! :-)
	    int x;
	    for(x=0;x<bp.length;x++)
		{
		if (!bp[x].produces(state,newpop,subpop,threadnum))
		    state.output.error("Breeding pipeline " + x + " of subpopulation " + subpop + " does not produce individuals of the expected species " + newpop.subpops[subpop].species.getClass().getName() + " or fitness " + newpop.subpops[subpop].f_prototype );
		bp[x].prepareToProduce(state,subpop,threadnum);
		}
	    state.output.exitIfErrors();
		
		
	    // start breedin'!
		
	    x=from[subpop];
	    int upperbound = from[subpop]+numinds[subpop];
	    while(x<upperbound)
		x += bp[bp.length==1 ? 0 :
		       BreedingSource.pickRandom(
			   bp,state.random[threadnum].nextFloat())].produce(
			       1,upperbound-x,x,subpop,
			       newpop.subpops[subpop].individuals,
			       state,threadnum);
	    if (x>upperbound) // uh oh!  Someone blew it!
		state.output.fatal("Whoa!  A breeding pipeline overwrote the space of another pipeline in subpopulation " + subpop + ".  You need to check your breeding pipeline code (in produce() ).");
	    }
	}
    }

/** A private helper class for implementing multithreaded breeding */
class SimpleBreederThread implements Runnable
    {
    Population newpop;
    public int[] numinds;
    public int[] from;
    public SimpleBreeder me;
    public EvolutionState state;
    public int threadnum;
    public void run()
	{
	try { me.breedPopChunk(newpop,state,numinds,from,threadnum); }
	catch (CloneNotSupportedException e) 
	    { throw new InternalError(); } // never happens
	}
    }
