package ec.steadystate;
import ec.*;
import ec.util.ParameterDatabase;
import ec.util.Parameter;
import ec.util.MersenneTwister;
import ec.util.Output;
import ec.util.Checkpoint;
import java.io.IOException;
import ec.util.Parameter;

/* 
 * SteadyStateEvolutionState.java
 * 
 * Created: Tue Aug 10 22:14:46 1999
 * By: Sean Luke
 */

/**
 * A SteadyStateEvolutionState is an EvolutionState which implements a simple
 * form of steady-state evolution.
 *
 * <p>First, all the individuals in the population are created and evaluated.
 * <b>(A)</b> Then 1 individual is selected by the breder for removal from the
 * population.  They are replaced by the result of breeding the other
 * individuals in the population.  Then just those newly-bred individuals are
 * evaluted.  Goto <b>(A)</b>.
 *
 * <p>Evolution stops when an ideal individual is found (if quitOnRunComplete
 * is set to true), or when the number of individuals evaluated exceeds the
 * parameter value numGenerations.
 * Every once in a while (once every <i>pseudogeneration</i> evaluations),
 * the system will garbage collect and write out a checkpoint file. 

 <p><b>Additional constraints:</b>
 <ul>
 <li> The breeder must be SteadyStateBreeder, or a subclass of it.
 <li> The evaluator must be a SteadyStateEvaluator, or a subclass of it.
 <li> All Species must implement the SteadyStateSpeciesForm interface.
 </ul>
 
 <p><b>Parameters</b><br>
 <table>
 <tr><td valign=top><tt>breed</tt><br>
 <font size=-1>classname, inherits or = ec.steadystate.SteadyStateBreeder</font></td>
 <td valign=top>(the class for breeder)</td></tr>
 <tr><td valign=top><tt>eval</tt><br>
 <font size=-1>classname, inherits or = ex.steadystate.SteadyStateEvaluator</font></td>
 <td valign=top>(the class for evaluator)</td></tr>


 * @author Sean Luke
 * @version 1.0 
 */

public class SteadyStateEvolutionState extends EvolutionState
    {
    /** base parameter for steady-state */
    public static final String P_STEADYSTATE = "steady";
    public static final String P_PSEUDOGENERATION = "pseudogeneration";

    public int newIndividuals[];

    public int pseudogeneration;
    
    public void setup(final EvolutionState state, final Parameter base)
	{
	super.setup(state,base);

	// double check that we have valid evaluators and breeders and exchangers
	if (!(breeder instanceof SteadyStateBreeder))
	    state.output.error("You've chosen to use Steady-State Evolution, but your breeder is not of the class SteadyStateBreeder.",base);
	if (!(evaluator instanceof SteadyStateEvaluator))
	    state.output.warning("You've chosen to use Steady-State Evolution, but your evaluator is not of the class SteadyStateEvaluator.",base);
	if (!(exchanger instanceof SteadyStateExchangerForm))
	    state.output.warning("You've chosen to use Steady-State Evolution, but your exchanger does not implement the SteadyStateExchangerForm.",base);

	pseudogeneration = parameters.getInt(new Parameter(P_PSEUDOGENERATION),null,1);
	if (pseudogeneration==0)
	    output.fatal("The pseudogeneration must be an integer >0.",base);
	
	// double-check to make sure that all the Species are of SteadyStateSpeciesForm.
	for (int pop=0; pop<state.population.subpops.length;pop++)
	    if (!(state.population.subpops[pop].species instanceof SteadyStateSpeciesForm))
		state.output.error("You've chosen to use Steady-State Evolution, but the species for subpopulation " + pop + " is is not a SteadyStateSpeciesForm Species",base);
	state.output.exitIfErrors();

	newIndividuals = new int[state.population.subpops.length];

	}
    
    
    /** Performs the evolutionary run.  Garbage collection and checkpointing are done only once every <i>pseudogeneration</i> evaluations.  The only Statistics calls made are preInitializationStatistics(), postInitializationStatistics(), occasional postEvaluationStatistics (done once every <i>pseudogeneration</i> evaluations), and finalStatistics(). */

    public void run(int condition) throws IOException
	{
	try
	    {
	    if (condition == C_STARTED_FRESH)
		{
		setup(this,null);  // a garbage Parameter
		exchanger.initializeContacts(this);
		statistics.preInitializationStatistics(this);
		population = initializer.initialPopulation(this);
		statistics.postInitializationStatistics(this);
		}
	    else // condition == C_STARTED_FROM_CHECKPOINT
		{
		output.restart();   // may throw an exception if there's a bad file
		resetFromCheckpoint();
		exchanger.reinitializeContacts(this);
		}
	    
	    
	    /* the big loop -- simplified to remove all the statistics
	       calls, since they'd be on a per-individual basis rather
	       than a per-population basis.*/
	    
	    int result = R_SUCCESS;
	    while ( true )
		{
		evaluator.evaluatePopulation(this);

		if (generation % pseudogeneration == (pseudogeneration-1))
		    statistics.postEvaluationStatistics(this);
		
		if (evaluator.runComplete(this) && quitOnRunComplete)
		    { output.message("Found Ideal Individual"); break; }
		
		if (generation == numGenerations-1)
		    {
		    result = R_FAILURE;
		    break;
		    }
		
		population = exchanger.preBreedingExchangePopulation(this);
		String exchangerWantsToShutdown = exchanger.runComplete(this);
		if (exchangerWantsToShutdown!=null)
		    { output.message(exchangerWantsToShutdown); break; }
		
		// gc
		
		if (gc && generation % pseudogeneration == (pseudogeneration-1))
		    {
		    if (aggressivegc) aggressiveGC();
		    else gc();
		    }
		
		// breed
		
		population = breeder.breedPopulation(this);
		population = exchanger.postBreedingExchangePopulation(this);
		
		if (checkpoint && generation % pseudogeneration == (pseudogeneration-1))
		    Checkpoint.setCheckpoint(this);
		
		generation++;
		}
	    
	    /* finish up -- we completed. */
	    statistics.finalStatistics(this,result);
	    finisher.finishPopulation(this,result);
	    exchanger.closeContacts(this,result);
	    }
	
	catch (CloneNotSupportedException e) 
	    { throw new InternalError(); } // never happens
	}
    }
