package ec.util;

import java.util.Properties;
import java.util.Vector;
import java.util.Enumeration;
import java.util.Hashtable;
import java.io.*;

/*
 * ParameterDatabase.java
 * Created: Sat Aug  7 12:09:19 1999
 */


 /**
 *
 * <p>This extension of the Properties class allows you to set, get, and
 * delete Parameters in a hierarchical tree-like database.  The database consists
 * of a list of Parameters, plus an array of "parent databases"
 * which it falls back on when it can't find the Parameter you're looking
 * for.  Parents may also have arrays of parents, and so on..
 *
 * <p>The parameters are loaded from a Java property-list file, which is
 * basically a collection of parameter=value pairs, one per line.  Empty lines
 * and lines beginning with # are ignored.  These parameters and their values
 * are <b>case-sensitive</b>, and whitespace is trimmed I believe. 
 * 
 * <p>An optional set of parameters, "parent.<i>n</i>", where <i>n</i> are
 * consecutive integers starting at 0, define the filenames of the database's
 * parents. 
 *
 * <p>When you create a ParameterDatabase using new ParameterDatabase(),
 * it is created thus:
 *
<p>
<table border=0 cellpadding=0 cellspacing=0>
<tr><td><tt>DATABASE:</tt></td><td><tt>&nbsp;database</tt></td></tr>
<tr><td><tt>FROM:</tt></td><td><tt>&nbsp;(empty)</tt></td></tr>
</table>

 *
 * <p> When you create a ParameterDatabase using new ParameterDatabase(<i>file</i>),
 * it is created by loading the database file, and its parent file tree, thus:

<p>
<table border=0 cellpadding=0 cellspacing=0>
<tr><td><tt>DATABASE:</tt></td><td><tt>&nbsp;database</tt></td><td><tt>&nbsp;-&gt;</tt></td><td><tt>&nbsp;parent0</tt></td><td><tt>&nbsp;+-&gt;</tt></td><td><tt>&nbsp;parent0</tt></td><td><tt>&nbsp;+-&gt;</tt></td><td><tt>&nbsp;parent0</tt></td><td><tt>&nbsp;+-&gt;</tt></td><td><tt>&nbsp;....</tt></td></tr>
<tr><td><tt>FROM:</tt></td><td><tt>&nbsp;(empty)</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;(file)</tt></td><td><tt>&nbsp;|</tt></td><td><tt>&nbsp;(parent.0)</tt></td><td><tt>&nbsp;|</tt></td><td><tt>&nbsp;(parent.0)</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;....</tt></td></tr>
<tr><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;|</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;+-&gt;</tt></td><td><tt>&nbsp;parent1</tt></td><td><tt>&nbsp;+-&gt;</tt></td><td><tt>&nbsp;....</tt></td></tr>
<tr><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;|</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;|</tt></td><td><tt>&nbsp;(parent.1)</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td></tr>
<tr><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;|</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;....</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td></tr>
<tr><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;|</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td></tr>
<tr><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;+-&gt;</tt></td><td><tt>&nbsp;parent1</tt></td><td><tt>&nbsp;+-&gt;</tt></td><td><tt>&nbsp;....</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td></tr>
<tr><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;|</tt></td><td><tt>&nbsp;(parent.1)</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td></tr>
<tr><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;....</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td></tr>
</table>

 *
 * <p> When you create a ParameterDatabase using new ParameterDatabase(<i>file,argv</i>),
 * the preferred way, it is created thus:
 *

<p>
<table border=0 cellpadding=0 cellspacing=0>
<tr><td><tt>DATABASE:</tt></td><td><tt>&nbsp;database</tt></td><td><tt>&nbsp;-&gt;</tt></td><td><tt>&nbsp;parent0</tt></td><td><tt>&nbsp;+-&gt;</tt></td><td><tt>&nbsp;parent0</tt></td><td><tt>&nbsp;+-&gt;</tt></td><td><tt>&nbsp;parent0</tt></td><td><tt>&nbsp;+-&gt;</tt></td><td><tt>&nbsp;parent0</tt></td><td><tt>&nbsp;+-&gt;</tt></td><td><tt>&nbsp;....</tt></td></tr>
<tr><td><tt>FROM:</tt></td><td><tt>&nbsp;(empty)</tt></td><td><tt>&nbsp;</tt></td><td><tt>(argv)</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;(file)</tt></td><td><tt>&nbsp;|</tt></td><td><tt>&nbsp;(parent.0)</tt></td><td><tt>&nbsp;|</tt></td><td><tt>&nbsp;(parent.0)</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;....</tt></td></tr>
<tr><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;|</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;+-&gt;</tt></td><td><tt>&nbsp;parent1</tt></td><td><tt>&nbsp;+-&gt;</tt></td><td><tt>&nbsp;....</tt></td></tr>
<tr><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;|</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;|</tt></td><td><tt>&nbsp;(parent.1)</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td></tr>
<tr><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;|</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;....</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td></tr>
<tr><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;|</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td></tr>
<tr><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;+-&gt;</tt></td><td><tt>&nbsp;parent1</tt></td><td><tt>&nbsp;+-&gt;</tt></td><td><tt>&nbsp;....</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td></tr>
<tr><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;|</tt></td><td><tt>&nbsp;(parent.1)</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td></tr>
<tr><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;....</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td><td><tt>&nbsp;</tt></td></tr>
</table>


 * <p>...that is, the actual top database is empty, and stores parameters
 * added programmatically; its parent is a database formed from arguments
 * passed in on the command line; <i>its</i> parent is the parameter database
 * which actually loads from foo.  This allows you to
 * programmatically add parameters which override those in foo, then delete
 * them, thus bringing foo's parameters back in view.
 *
 * <p>Once a parameter database is loaded, you query it with the
 * <tt>get</tt> methods.  The database, then its parents, are searched
 * until a match is found for your parameter.  The search rules are thus:
 * (1) the root database is searched first.  (2) If a database being searched
 * doesn't contain the data, it searches its parents recursively, starting with
 * parent 0, then moving up, until all searches are exhausted or something was
 * found. (3) No database is searched twice.
 * 
 * <p>You can set a parameter
 * (in the topmost database <i>only</i> with the <tt>set</tt> command.
 * The <tt>remove</tt> command removes a parameter from the topmost database
 * only.  The <tt>removeDeeply</tt> command removes that parameter from
 * every database.
 *
 * <p>The values stored in a parameter database must not contain "#", "=", 
 * non-ascii values, or whitespace.
 *
 * <p><b>Note for JDK 1.1</b>.  Finally recovering from stupendous idiocy,
 * JDK 1.2 included parseDouble() and parseFloat() commands; now you can
 * READ A FLOAT FROM A STRING without having to create a Float object first!
 * Anyway, you will need to modify the getFloat() method below if you're
 * running on JDK 1.1, but understand that large numbers of calls to the
 * method may be inefficient.  Sample JDK 1.1 code is given with those methods,
 * but is commented out.
 *
 *
 * @author Sean Luke
 * @version 1.0
 */



public final class ParameterDatabase extends Properties implements Serializable
    {
    public static final String C_HERE = "$";
    public static final String UNKNOWN_VALUE = "";
    private Vector parents;
    private File directory;
    private boolean checked;
    private Hashtable gotten;
    private Hashtable accessed;


    /** Searches down through databases to find a given parameter, whose value
	must be a full Class name, and the class must be a descendent of but not equal 
	to <i>mustCastTosuperclass</i>.  Loads the class and returns an instance 
	(constructed with the default constructor), or throws a ParamClassLoadException 
	if there is no such Class.  If the parameter is not found, the defaultParameter is used. 
	The parameter chosen is marked "used". */ 
    public final Object getInstanceForParameter(Parameter parameter, Parameter defaultParameter, Class mustCastTosuperclass)
	    throws ParamClassLoadException
	{
	Parameter p;
	if (exists(parameter)) p = parameter;
	else if (exists(defaultParameter)) p = defaultParameter;
	else throw new ParamClassLoadException("No class name provided.\nPARAMETER: " + 
						       parameter + (defaultParameter==null ? "" : "\n     ALSO: " + defaultParameter));
	try
	    {
	    Class c = Class.forName(get(p));
	    if (!mustCastTosuperclass.isAssignableFrom(c))
		throw new ParamClassLoadException(
		    "The class " + c.getName() + "\ndoes not cast into the superclass " +
		    mustCastTosuperclass.getName() + "\nPARAMETER: " + parameter + (defaultParameter==null ? "" : "\n     ALSO: " + defaultParameter));
	    if (mustCastTosuperclass==c)
		throw new ParamClassLoadException(
		    "The class " + c.getName() + 
		    "\nmust not be the same as the required superclass " + 
		    mustCastTosuperclass.getName() + "\nPARAMETER: " + parameter + (defaultParameter==null ? "" : "\n     ALSO: " + defaultParameter));
	    return c.newInstance();
	    }
	catch(ClassNotFoundException e)
	    {
	    throw new ParamClassLoadException(
		"Class not found: " + get(p) + "\nPARAMETER: " + parameter + 
		 (defaultParameter==null ? "" : "\n     ALSO: " + defaultParameter) + "\nEXCEPTION: \n\n" + e);
	    }
	catch(IllegalArgumentException e)
	    {
	    throw new ParamClassLoadException(
		"Could not load class: " + get(p) + "\nPARAMETER: " + parameter + (defaultParameter==null ? "" : "\n     ALSO: " + defaultParameter) + "\nEXCEPTION: \n\n" + e);
	    }
	catch(InstantiationException e)
	    {
	    throw new ParamClassLoadException(
		"The requested class is an interface or an abstract class: " + 
		get(p) + "\nPARAMETER: " + parameter 
		 + (defaultParameter==null ? "" : "\n     ALSO: " + defaultParameter) +  "\nEXCEPTION: \n\n" + e);
	    }
	catch(IllegalAccessException e)
	    {
	    throw new ParamClassLoadException(
		"The requested class cannot be initialized with the default initializer: " +
		get(p) + "\nPARAMETER: " + parameter + (defaultParameter==null ? "" : "\n     ALSO: " + defaultParameter) + "\nEXCEPTION: \n\n" + e);
	    }
	}


    /** Searches down through databases to find a given parameter, whose value must be a full Class name, and the class must be a descendent, or equal to, <i>mustCastTosuperclass</i>.  Loads the class and returns an instance (constructed with the default constructor), or throws a ParamClassLoadException if there is no such Class.  The parameter chosen is marked "used". */ 
    public final Object getInstanceForParameterEq(Parameter parameter, Parameter defaultParameter, Class mustCastTosuperclass)
	    throws ParamClassLoadException
	{
	Parameter p;
	if (exists(parameter)) p = parameter;
	else if (exists(defaultParameter)) p = defaultParameter;
	else throw new ParamClassLoadException("No class name provided.\nPARAMETER: " + 
						       parameter + "\n     ALSO: " + (defaultParameter==null ? "" : "\n     ALSO: " + defaultParameter));
	try
	    {
	    Class c = Class.forName(get(p));
	    if (!mustCastTosuperclass.isAssignableFrom(c))
		throw new ParamClassLoadException(
		    "The class " + c.getName() + "\ndoes not cast into the superclass " +
		    mustCastTosuperclass.getName() + "\nPARAMETER: " + parameter + 
		    "\n     ALSO: " + (defaultParameter==null ? "" : "\n     ALSO: " + defaultParameter));
	    return c.newInstance();
	    }
	catch(ClassNotFoundException e)
	    {
	    throw new ParamClassLoadException(
		"Class not found: " + get(p) + "\nPARAMETER: " + parameter + 
		"\n     ALSO: " + (defaultParameter==null ? "" : "\n     ALSO: " + defaultParameter) + "\nEXCEPTION: \n\n" + e);
	    }
	catch(IllegalArgumentException e)
	    {
	    throw new ParamClassLoadException(
		"Could not load class: " + get(p) + "\nPARAMETER: " + parameter + 
		"\n     ALSO: " + (defaultParameter==null ? "" : "\n     ALSO: " + defaultParameter) + "\nEXCEPTION: \n\n" + e);
	    }
	catch(InstantiationException e)
	    {
	    throw new ParamClassLoadException(
		"The requested class is an interface or an abstract class: " + 
		get(p) + "\nPARAMETER: " + parameter + 
		"\n     ALSO: " + (defaultParameter==null ? "" : "\n     ALSO: " + defaultParameter) + "\nEXCEPTION: \n\n" + e);
	    }
	catch(IllegalAccessException e)
	    {
	    throw new ParamClassLoadException(
		"The requested class cannot be initialized with the default initializer: " +
		get(p) + "\nPARAMETER: " + parameter + 
		"\n     ALSO: " + (defaultParameter==null ? "" : "\n     ALSO: " + defaultParameter) + "\nEXCEPTION: \n\n" + e);
	    }
	}


    /** Searches down through databases to find a given parameter.  
	The value associated with this parameter must be a full Class name, 
	and the class must be a descendent of but not equal to <i>mustCastTosuperclass</i>.  
	Loads and returns the associated Class, or throws a ParamClassLoadException 
	if there is no such Class.  If the parameter is not found, the defaultParameter is used.
	The parameter chosen is marked "used". */ 
    public final Object getClassForParameter(Parameter parameter, Parameter defaultParameter, Class mustCastTosuperclass)
	    throws ParamClassLoadException
	{
	Parameter p;
	if (exists(parameter)) p = parameter;
	else if (exists(defaultParameter)) p = defaultParameter;
	else throw new ParamClassLoadException("No class name provided.\nPARAMETER: " + 
						       parameter + "\n     ALSO: " + (defaultParameter==null ? "" : "\n     ALSO: " + defaultParameter));
	try
	    {
	    Class c = Class.forName(get(p));
	    if (!mustCastTosuperclass.isAssignableFrom(c))
		throw new ParamClassLoadException(
		    "The class " + c.getName() + "\ndoes not cast into the superclass " +
		    mustCastTosuperclass.getName() + "\nPARAMETER: " + parameter + 
		    "\n     ALSO: " + (defaultParameter==null ? "" : "\n     ALSO: " + defaultParameter));
	    return c;
	    }
	catch(ClassNotFoundException e)
	    {
	    throw new ParamClassLoadException(
		"Class not found: " + get(p) + "\nPARAMETER: " + parameter + 
		"\n     ALSO: " + (defaultParameter==null ? "" : "\n     ALSO: " + defaultParameter) + "\nEXCEPTION: \n\n" + e);
	    }
	catch(IllegalArgumentException e)
	    {
	    throw new ParamClassLoadException(
		"Could not load class: " + get(p) + "\nPARAMETER: " + parameter + 
		"\n     ALSO: " + (defaultParameter==null ? "" : "\n     ALSO: " + defaultParameter) + "\nEXCEPTION: \n\n" + e);
	    }
	}
    


    /** Searches down through databases to find a given parameter;
	If the parameter does not exist, defaultValue is returned.
	If the parameter exists, and it is set to "false" (case insensitive), false is returned.
	Else true is returned. The parameter chosen is marked "used" if it exists. */
    public final boolean getBoolean(Parameter parameter, Parameter defaultParameter, boolean defaultValue)
	{
	if (exists(parameter)) return getBoolean(parameter, defaultValue);
	else return getBoolean(defaultParameter, defaultValue);
	}

    /** Searches down through databases to find a given parameter;
	If the parameter does not exist, defaultValue is returned.
	If the parameter exists, and it is set to "false" (case insensitive), false is returned.
	Else true is returned. The parameter chosen is marked "used" if it exists. */
    final boolean getBoolean(Parameter parameter, boolean defaultValue)
	{
	if (!exists(parameter)) return defaultValue;
	return (!get(parameter).equalsIgnoreCase("false"));
	}


    /** Searches down through databases to find a given parameter, which must be an integer.  If there is an error in parsing the parameter, then default is returned. The parameter chosen is marked "used" if it exists. */
    public final int getIntWithDefault(Parameter parameter, Parameter defaultParameter, int defaultValue)
	{
	if (exists(parameter)) return getIntWithDefault(parameter, defaultValue);
	else return getIntWithDefault(defaultParameter, defaultValue);
	}

    /** Searches down through databases to find a given parameter, which must be an integer.  If there is an error in parsing the parameter, then default is returned. The parameter chosen is marked "used" if it exists. */
    final int getIntWithDefault(Parameter parameter, int defaultValue)
	{
	if (exists(parameter))
	    {
	    try { return Integer.parseInt(get(parameter)); }
	    catch(NumberFormatException e) { return defaultValue; }
	    }
	else return defaultValue;
	}


    /** Searches down through databases to find a given parameter, whose value must be an integer >= minValue.  It returns the value, or minValue-1 if the value is out of range or if there is an error in parsing the parameter. The parameter chosen is marked "used" if it exists. */
    public final int getInt(Parameter parameter, Parameter defaultParameter, int minValue)
	{
	if (exists(parameter)) return getInt(parameter, minValue);
	else return getInt(defaultParameter, minValue);
	}

    /** Searches down through databases to find a given parameter, whose value must be an integer >= minValue.  It returns the value, or minValue-1 if the value is out of range or if there is an error in parsing the parameter. The parameter chosen is marked "used" if it exists. */
    final int getInt(Parameter parameter, int minValue)
	{
	if (exists(parameter))
	    {
	    try
		{
		int i = Integer.parseInt(get(parameter));
		if (i < minValue) return minValue-1;
		return i;
		}
	    catch(NumberFormatException e) { return minValue-1; }
	    }
	else return minValue-1;
	}

    /** Searches down through databases to find a given parameter, whose value must be an integer >= minValue and <= maxValue.  It returns the value, or minValue-1 if the value is out of range or if there is an error in parsing the parameter. The parameter chosen is marked "used" if it exists. */
    public final int getIntWithMax(Parameter parameter, Parameter defaultParameter, int minValue, int maxValue)
	{
	if (exists(parameter)) return getIntWithMax(parameter, minValue, maxValue);
	else return getIntWithMax(defaultParameter, minValue, maxValue);
	}

    /** Searches down through databases to find a given parameter, whose value must be an integer >= minValue and <= maxValue.  It returns the value, or minValue-1 if the value is out of range or if there is an error in parsing the parameter. The parameter chosen is marked "used" if it exists. */
    final int getIntWithMax(Parameter parameter, int minValue, int maxValue)
	{
	if (exists(parameter))
	    {
	    try
		{
		int i = Integer.parseInt(get(parameter));
		if (i < minValue) return minValue-1;
		if (i > maxValue) return minValue-1;
		return i;
		}
	    catch(NumberFormatException e) { return minValue-1; }
	    }
	else return minValue-1;
	}



    /** Searches down through databases to find a given parameter, whose value must be a float >= minValue.  If not, this method returns minvalue-1, else it returns the parameter value. The parameter chosen is marked "used" if it exists. */

    public final float getFloat(Parameter parameter, Parameter defaultParameter, double minValue)
	{
	if (exists(parameter)) return getFloat(parameter,minValue);
	else return getFloat(defaultParameter, minValue);
	}

    /** Searches down through databases to find a given parameter, whose value must be a float >= minValue.  If not, this method returns minvalue-1, else it returns the parameter value. The parameter chosen is marked "used" if it exists. */

    final float getFloat(Parameter parameter, double minValue)
	{
	if (exists(parameter))
	    {
	    try
		{
		float i = Float.valueOf(get(parameter)).floatValue();   // what stupidity...
		
		// For JDK 1.2 and later, this is more efficient...
		// float i = Float.parseFloat(get(parameter));
		// ...but we can't use it and still be compatible with JDK 1.1

		if (i < minValue) return (float)(minValue-1);
		return i;
		}
	    catch(NumberFormatException e) { return (float)(minValue-1); }
	    }
	else return (float)(minValue-1);
	}





    /** Searches down through databases to find a given parameter, whose value must be a float >= minValue and <= maxValue.  If not, this method returns minvalue-1, else it returns the parameter value. The parameter chosen is marked "used" if it exists. */

    public final float getFloat(Parameter parameter, Parameter defaultParameter, double minValue, double maxValue)
	{
	if (exists(parameter)) return getFloat(parameter,minValue,maxValue);
	else return getFloat(defaultParameter,minValue,maxValue);
	}


    /** Searches down through databases to find a given parameter, whose value must be a float >= minValue and <= maxValue.  If not, this method returns minvalue-1, else it returns the parameter value. The parameter chosen is marked "used" if it exists. */

    final float getFloat(Parameter parameter, double minValue, double maxValue)
	{
	if (exists(parameter))
	    {
	    try
		{
		float i = Float.valueOf(get(parameter)).floatValue();   // what stupidity...
		
		// For JDK 1.2 and later, this is more efficient...
		// float i = Float.parseFloat(get(parameter));
		// ...but we can't use it and still be compatible with JDK 1.1

		if (i < minValue) return (float)(minValue-1);
		if (i > maxValue) return (float)(minValue-1);
		return i;
		}
	    catch(NumberFormatException e) { return (float)(minValue-1); }
	    }
	else return (float)(minValue-1);
	}


    /** Searches down through databases to find a given parameter, whose value must be a long >= minValue.  
	If not, this method returns errValue, else it returns the parameter value. The parameter chosen is marked "used" if it exists. */

    public final long getLong(Parameter parameter, Parameter defaultParameter, long minValue)
	{
	if (exists(parameter)) return getLong(parameter, minValue);
	else return getLong(defaultParameter, minValue);
	}

    /** Searches down through databases to find a given parameter, whose value must be a long >= minValue.  
	If not, this method returns errValue, else it returns the parameter value. The parameter chosen is marked "used" if it exists. */
    final long getLong(Parameter parameter, long minValue)
	{
	if (exists(parameter))
	    {
	    try
		{
		long i = Long.parseLong(get(parameter));
		if (i < minValue) return minValue-1;
		return i;
		}
	    catch(NumberFormatException e) { return minValue-1; }
	    }
	else return (minValue-1);
	}



    /** Searches down through databases to find a given parameter, whose value must be a long >= minValue and =< maxValue.  If not, this method returns errValue, else it returns the parameter value. The parameter chosen is marked "used" if it exists. */
    public final long getLong(Parameter parameter, Parameter defaultParameter, long minValue, long maxValue)
	{
	if (exists(parameter)) return getLong(parameter,minValue,maxValue);
	else return getLong(defaultParameter,minValue,maxValue);
	}

    /** Searches down through databases to find a given parameter, whose value must be a long >= minValue and =< maxValue.  If not, this method returns errValue, else it returns the parameter value. The parameter chosen is marked "used" if it exists. */
    final long getLong(Parameter parameter, long minValue, long maxValue)
	{
	if (exists(parameter))
	    {
	    try
		{
		long i = Long.parseLong(get(parameter));
		if (i < minValue) return minValue-1;
		if (i > maxValue) return minValue-1;
		return i;
		}
	    catch(NumberFormatException e) { return minValue-1; }
	    }
	else return (minValue-1);
	}





    /** Searches down through the databases to find a given parameter, whose value must be 
	an absolute or relative path name.  If it is absolute, a File is made based on
	the path name.  If it is relative, a file is made by resolving the path name
	with respect to the directory in which the file was which defined this
	ParameterDatabase in the ParameterDatabase hierarchy. If the parameter
	is not found, this returns null.  The File is not checked for validity. 
	The parameter chosen is marked "used" if it exists. */
	
    public final File getFile(Parameter parameter, Parameter defaultParameter)
	{
	if (exists(parameter)) return getFile(parameter);
	else return getFile(defaultParameter);
	}
    
    
    /** Searches down through the databases to find a given parameter, whose value must be 
	an absolute or relative path name.  If the parameter begins with a "$",
	a file is made based on the relative path name and returned directly.
	Otherwise, if it is absolute, a File is made based on
	the path name, or if it is relative, a file is made by resolving the path name
	with respect to the directory in which the file was which defined this
	ParameterDatabase in the ParameterDatabase hierarchy. If the parameter
	is not found, this returns null.  The File is not checked for validity. 
	The parameter chosen is marked "used" if it exists. */
    
    final File getFile(Parameter parameter)
	{
	if (exists(parameter))
	    {
	    String p = get(parameter);
	    if (p==null) return null;
	    if (p.startsWith(C_HERE))
		return new File(p.substring(C_HERE.length()));
	    else
		{
		File f = new File(p);
		if (f.isAbsolute()) return f;
		else return new File(directoryFor(parameter),p);
		}
	    }
	else return null;
	}
    
    
    
    
    /** Searches down through databases to find a given parameter.  Returns the parameter's value (trimmed) or null if not found or if the trimmed result is empty. The parameter chosen is marked "used" if it exists. */
    
    public final synchronized String getString(Parameter parameter, Parameter defaultParameter)
	{
	if (exists(parameter)) return getString(parameter);
	else return getString(defaultParameter);
	}
    
    /** Searches down through databases to find a given parameter.  Returns the parameter's value (trimmed) or null if not found or if the trimmed result is empty. The parameter chosen is marked "used" if it exists. */
    
    final synchronized String getString(Parameter parameter)
	{
	if (exists(parameter)) return get(parameter);
	else return null;
	}


    /** Searches down through databases to find a given parameter.  
	Returns the parameter's value trimmed of whitespace, or defaultValue.trim() if the
	result is not found or the trimmed result is empty. */
    public final String getStringWithDefault(Parameter parameter, Parameter defaultParameter, String defaultValue)
	{
	if (exists(parameter)) return getStringWithDefault(parameter, defaultValue);
	else return getStringWithDefault(defaultParameter, defaultValue);
	}

    /** Searches down through databases to find a given parameter.  
	Returns the parameter's value trimmed of whitespace, or defaultValue.trim() if the
	result is not found or the trimmed result is empty. */
    public final String getStringWithDefault(Parameter parameter, String defaultValue)
	{
	if (exists(parameter))
	    {
	    String result = get(parameter);
	    if (result==null) result = defaultValue.trim();
	    else 
		{
		result=result.trim();
		if (result.length()==0) result = defaultValue.trim();
		}
	    return result;
	    }
	else return null;
	}















    /** Clears the checked flag */
    public final synchronized void uncheck()
	{
	if (!checked)  return; // we already unchecked this path -- this is dangerous if parents are used without children
	checked = false;
	int size = parents.size();
	for(int x=0;x<size;x++) ((ParameterDatabase)(parents.elementAt(x))).uncheck();
	}


    /** Sets a parameter in the topmost database to a given value, trimmed of whitespace.  */
    public final synchronized void set(Parameter parameter, String value)
	{
	put(parameter.param,value.trim());
	}
    
    /** Prints out all the parameters marked as used, plus their values.  If a
	parameter was listed as "used" but not's actually in the database,
	the value printed is UNKNOWN_VALUE (set to "?????") */

    public final synchronized void listGotten(PrintWriter p)
	{
	Vector vec = new Vector();
	Enumeration e = gotten.keys();
	while(e.hasMoreElements())
	    vec.addElement(e.nextElement());

	// sort the keys
	Object[] array = new Object[vec.size()];
	vec.copyInto(array);
	QuickSort.qsort(array,new SortComparator() 
	    { 
	    public boolean lt(Object a, Object b) 
		{ return ((String)a).compareTo((String)b) < 0; }
	    public boolean gt(Object a, Object b) 
		{ return ((String)a).compareTo((String)b) > 0; }
	    });
	
	// Uncheck and print each item
	for(int x=0;x<array.length;x++)
	    {
	    String s = (String)(array[x]);
	    String v = null;
	    if (s!=null) 
		{
		v = (String)(_get(s));
		uncheck();
		}
	    if (v==null) v = UNKNOWN_VALUE;
	    p.println(s + " = " + v);
	    }
	}


    /** Prints out all the parameters marked as accessed ("gotten" by some
	getFoo(...) method), plus their values.  
	If this method ever prints UNKNOWN_VALUE ("?????"), that's a bug. */

    public final synchronized void listAccessed(PrintWriter p)
	{
	Vector vec = new Vector();
	Enumeration e = accessed.keys();
	while(e.hasMoreElements())
	    vec.addElement(e.nextElement());

	// sort the keys
	Object[] array = new Object[vec.size()];
	vec.copyInto(array);
	QuickSort.qsort(array,new SortComparator() 
	    { 
	    public boolean lt(Object a, Object b) 
		{ return ((String)a).compareTo((String)b) < 0; }
	    public boolean gt(Object a, Object b) 
		{ return ((String)a).compareTo((String)b) > 0; }
	    });
	
	// Uncheck and print each item
	for(int x=0;x<array.length;x++)
	    {
	    String s = (String)(array[x]);
	    String v = null;
	    if (s!=null)
		{
		v = (String)(_get(s));
		uncheck();
		}
	    if (v==null) v = UNKNOWN_VALUE;
	    p.println(s + " = " + v);
	    }
	}


    /** Returns true if parameter exist in the database */
    public final synchronized boolean exists(Parameter parameter)
	{
	if (parameter==null) return false;
	String result = _get(parameter.param);
	uncheck();
	accessed.put(parameter.param,Boolean.TRUE);
	return (result!=null);
	}

    /** Returns true if either parameter or defaultParameter exists in the database */
    public final synchronized boolean exists(Parameter parameter, Parameter defaultParameter)
	{
	if (exists(parameter)) return true;
	if (exists(defaultParameter)) return true;
	return false;
	}


    protected final synchronized String get(Parameter parameter)
	{
	String result =  _get(parameter.param);
	uncheck();

	// set hashtable appropriately
	if (parameter!=null) accessed.put(parameter.param,Boolean.TRUE);
	if (parameter!=null) gotten.put(parameter.param,Boolean.TRUE);
	return result;
	}

    /** Private helper function */
    private final synchronized String _get(String parameter)
	{
	if (parameter==null) return null;
	if (checked) return null;  // we already searched this path
	checked = true;
	String result =getProperty(parameter);
	if (result==null)
	    {
	    int size = parents.size();
	    for(int x=0;x<size;x++) 
		{
		result = ((ParameterDatabase)(parents.elementAt(x)))._get(parameter);
		if (result!=null) return result;
		}
	    }
	else
	    {  // preprocess
	    result=result.trim();
	    if (result.length()==0) result = null;
	    }
	return result;
	}

    /** Searches down through databases to find the directory for the
	database which holds a given parameter.  Returns the
	directory name or null if not found. */

    public final File directoryFor(Parameter parameter)
	{
	File result =  _directoryFor(parameter);
	uncheck();
	return result;
	}

    /** Private helper function */
    private final synchronized File _directoryFor(Parameter parameter)
	{
	if (checked) return null;  // we already searched this path
	checked = true;
	File result = null;
	String p =getProperty(parameter.param);
	if (p==null)
	    {
	    int size = parents.size();
	    for(int x=0;x<size;x++) 
		{
		result = ((ParameterDatabase)(parents.elementAt(x)))._directoryFor(parameter);
		if (result!=null) return result;
		}
	    return result;
	    }
	else return directory;
	}


    /** Removes a parameter from the topmost database. */
    public final synchronized void remove(Parameter parameter)
	{
	remove(parameter.param);
	}

    /** Removes a parameter from the database and all its parent databases. */
    public final synchronized void removeDeeply(Parameter parameter)
	{
	_removeDeeply(parameter);
	uncheck();
	}

    /** Private helper function */
    private synchronized void _removeDeeply(Parameter parameter)
	{
	if (checked) return; // already removed from this path
	checked = true;
	remove(parameter);
	int size = parents.size();
	for(int x=0;x<size;x++) ((ParameterDatabase)(parents.elementAt(x))).removeDeeply(parameter);
	}


    /** Creates an empty parameter database. */
    public ParameterDatabase()
	{
	super();
	accessed = new Hashtable();
	gotten = new Hashtable();
	directory = new File(new File("").getAbsolutePath());  // uses the user path
	parents = new Vector();
	checked = false;  // unnecessary
	}

    /** Creates a new parameter database tree from a given database file and its parent files. */
    public ParameterDatabase(File filename) throws FileNotFoundException, IOException
	{
	this();
	directory = new File(filename.getParent());  // get the directory filename is in
	load(new FileInputStream(filename));

	// load parents

	for(int x=0;;x++)
	    {
	    String s = getProperty("parent."+x);
	    if (s==null) return;  // we're done

	    if (new File(s).isAbsolute())  // it's an absolute file definition
		parents.addElement(new ParameterDatabase(new File(s)));
	    else // it's relative to my path
		parents.addElement(new ParameterDatabase(new File(filename.getParent(),s)));
	    }
	}


    /** Creates a new parameter database from a given database file and argv list.
	The top-level database is completely empty, pointing to a second database
	which contains the parameter entries stored in args, which points to a
	tree of databases constructed using ParameterDatabase(filename).
    */
    
    public ParameterDatabase(File filename, String[] args) throws FileNotFoundException, IOException
	{
	this();
	directory = new File(filename.getParent());  // get the directory filename is in

	// Create the Parameter Database tree for the files
	ParameterDatabase files = new ParameterDatabase(filename);
	
	// Create the Parameter Database for the arguments
	ParameterDatabase a = new ParameterDatabase();
	a.parents.addElement(files);
	for(int x=0;x<args.length-1;x++)
	    {
	    if (args[x].equals("-p"))
		a.parseParameter(args[x+1]);
	    }
	
	// Set me up
	parents.addElement(a);
	}


    /** Parses and adds s to the database.  Returns true if there 
	was actually something to parse. */
    private final boolean parseParameter(String s)
	{
	s = s.trim();
	if (s.length()==0) return false;
	if (s.charAt(0)=='#') return false;
	int eq = s.indexOf('=');
	if (eq<0) return false;
	put(s.substring(0,eq),s.substring(eq+1));
	return true;
	}

    /** Prints out all the parameters in the database.
	Useful for debugging.  If listShadowed is true, each
	parameter is printed with the parameter database it's located in.
	If listShadowed is false, only active parameters are listed,
	and they're all given in one big chunk. */
    public final void list(PrintWriter p, boolean listShadowed)
	{
	if (listShadowed)
	    _list(p,listShadowed,"root",null);
	else
	    {
	    Hashtable gather = new Hashtable();
	    _list(p,listShadowed,"root",gather);


	    Vector vec = new Vector();
	    Enumeration e = gather.keys();
	    while(e.hasMoreElements())
		vec.addElement(e.nextElement());
	    
	    // sort the keys
	    Object[] array = new Object[vec.size()];
	    vec.copyInto(array);
	    QuickSort.qsort(array,new SortComparator() 
		{ 
		public boolean lt(Object a, Object b) 
		    { return ((String)a).compareTo((String)b) < 0; }
		public boolean gt(Object a, Object b) 
		    { return ((String)a).compareTo((String)b) > 0; }
		});
	    
	    // Uncheck and print each item
	    for(int x=0;x<array.length;x++)
		{
		String s = (String)(array[x]);
		String v = null;
		if (s!=null)
		    v = (String)gather.get(s);
		if (v==null) v= UNKNOWN_VALUE;
		p.println(s + " = " + v); 
		}
	    }
	}

    /** Private helper function. */
    private final void _list(PrintWriter p, boolean listShadowed, String prefix, Hashtable gather)
	{
	if (listShadowed)
	    {
	    // Print out my header
	    p.println("\n########" + prefix);
	    list(p);
	    int size = parents.size();
	    for(int x=0;x<size;x++) ((ParameterDatabase)(parents.elementAt(x)))._list(p,listShadowed,prefix + "." + x,gather);
	    }
	else
	    {
	    // load in reverse order so things get properly overwritten
	    int size = parents.size();
	    for(int x=size-1;x>=0;x--) ((ParameterDatabase)(parents.elementAt(x)))._list(p,listShadowed,prefix,gather);
	    Enumeration e = keys();
	    while(e.hasMoreElements())
		{
		String key = (String)(e.nextElement());
		gather.put(key,get(key));
		}
	    }
	}




    /** Test the ParameterDatabase */
    public static void main(String[] args) throws FileNotFoundException,IOException
	{

	ParameterDatabase pd = new ParameterDatabase(new File(args[0]),args);
	pd.set(new Parameter("Hi there"), "Whatever");
	pd.set(new Parameter(new String[] {"1", "2", "3"}), " Whatever ");
	pd.set(new Parameter(new String[] {"a", "b", "c"}).pop().push("d"),
	       "Whatever");

	System.out.println("\n\n PRINTING ALL PARAMETERS \n\n");
	pd.list(new PrintWriter(System.out,true),true);
	System.out.println("\n\n PRINTING ONLY VALID PARAMETERS \n\n");
	pd.list(new PrintWriter(System.out,true),false);
	}

    }
