package ec.vector;

import ec.*;
import ec.util.*;
import java.io.Serializable;
import java.util.*;
import java.io.*;

/*
 * ByteVectorIndividual.java
 * Created: Tue Mar 13 15:03:12 EST 2001
 */

/**
 * ByteVectorIndividual is a VectorIndividual whose genome is an array of bytes.
 * Gene values may range from mingene to maxgene, inclusive.
 * The default mutation method randomizes genes to new values in this range,
 * with <tt>mutationProbability</tt>.
 *
 <p><b>Parameters</b><br>
 <table>
 <tr><td valign=top><tt>min-gene</tt><br>
 <font size=-1>int (default=0)</font></td>
 <td valign=top>(the minimum gene value)</td></tr>

 <tr><td valign=top><tt>max-gene</tt><br>
 <font size=-1>int &gt;= min-gene</font></td>
 <td valign=top>(the maximum gene value)</td></tr>

 </table>

 <p><b>Default Base</b><br>
 vector.byte-vect-ind

 * @author Liviu Panait
 * @author Sean Luke
 * @version 1.0
 */

public class ByteVectorIndividual extends VectorIndividual
    {
    public static final String P_BYTEVECTORINDIVIDUAL = "byte-vect-ind";
    public final static String P_MINGENE = "min-gene";
    public final static String P_MAXGENE = "max-gene";
    public byte[] genome;
    public byte mingene;
    public byte maxgene;
    
    public Parameter defaultBase()
        {
        return VectorDefaults.base().push(P_BYTEVECTORINDIVIDUAL);
        }

    public Object protoClone() throws CloneNotSupportedException
        {
        ByteVectorIndividual myobj = (ByteVectorIndividual) (super.protoClone());

        // must clone the genome
        myobj.genome = (byte[])(genome.clone());
        
        return myobj;
        } 

    public void setup(final EvolutionState state, final Parameter base)
        {
        super.setup(state,base);
        genome = new byte[genomeSize];

	Parameter def = defaultBase();
    
        int mingene_t = state.parameters.getIntWithDefault(base.push(P_MINGENE),def.push(P_MINGENE),0);
        int maxgene_t = state.parameters.getInt(base.push(P_MAXGENE),def.push(P_MAXGENE),mingene);
        if (maxgene < mingene)
            state.output.fatal("ByteVectorIndividual must have a min-gene which is <= the max-gene",
                base.push(P_MAXGENE),def.push(P_MAXGENE));
        mingene = (byte)mingene_t;
        maxgene = (byte)maxgene_t;
        if (mingene != mingene_t)
            state.output.fatal("ByteVectorIndividual must have a min-gene within the byte range",
                base.push(P_MINGENE),def.push(P_MINGENE));
        if (maxgene != maxgene_t)
            state.output.fatal("ByteVectorIndividual must have a max-gene within the byte range",
                base.push(P_MAXGENE),def.push(P_MAXGENE));
        }
        
    public void defaultCrossover(EvolutionState state, int thread, VectorIndividual ind)
        {
        ByteVectorIndividual i = (ByteVectorIndividual) ind;
        byte tmp;
        int point;

        if (genome.length != i.genome.length)
            state.output.fatal("Genome lengths are not the same for fixed-length vector crossover");
        switch(crossoverType)
            {
            case C_ONE_POINT:
                point = state.random[thread].nextInt((genome.length / chunksize)+1);
                for(int x=0;x<point*chunksize;x++)
                    { 
                    tmp = i.genome[x];
                    i.genome[x] = genome[x]; 
                    genome[x] = tmp; 
                    }
                break;
            case C_TWO_POINT: 
                int point0 = state.random[thread].nextInt((genome.length / chunksize)+1);
                point = state.random[thread].nextInt((genome.length / chunksize)+1);
                if (point0 > point) { int p = point0; point0 = point; point = p; }
                for(int x=point0*chunksize;x<point*chunksize;x++)
                    {
                    tmp = i.genome[x];
                    i.genome[x] = genome[x];
                    genome[x] = tmp;
                    }
                break;
            case C_ANY_POINT:
                for(int x=0;x<genome.length/chunksize;x++) 
                    if (state.random[thread].nextBoolean(crossoverProbability))
                        for(int y=x*chunksize;y<(x+1)*chunksize;y++)
                            {
                            tmp = i.genome[y];
                            i.genome[y] = genome[y];
                            genome[y] = tmp;
                            }
                break;
            }
        }

    /** Splits the genome into n pieces, according to points, which *must* be sorted. 
        pieces.length must be 1 + points.length */
    public void split(int[] points, Object[] pieces)
        {
        int point0, point1;
        point0 = 0; point1 = points[0];
        for(int x=0;x<pieces.length;x++)
            {
            pieces[x] = new byte[point1-point0];
            System.arraycopy(genome,point0,pieces[x],point0,point1-point0);
            point0 = point1;
            if (x==pieces.length-2)
                point1 = genome.length;
            else point1 = points[x+1];
            }
        }
    
    /** Joins the n pieces and sets the genome to their concatenation.*/
    public void join(Object[] pieces)
        {
        int sum=0;
        for(int x=0;x<pieces.length;x++)
            sum += ((byte[])(pieces[x])).length;
        
        int runningsum = 0;
        byte[] newgenome = new byte[sum];
        for(int x=0;x<pieces.length;x++)
            {
            System.arraycopy(pieces[x], 0, newgenome, runningsum, ((byte[])(pieces[x])).length);
            runningsum += ((byte[])(pieces[x])).length;
            }
        // set genome
        genome = newgenome;
        }

    /** Destructively mutates the individual in some default manner.  The default form
        simply randomizes genes to a uniform distribution from the min and max of the gene values. */
    public void defaultMutate(EvolutionState state, int thread)
        {
        if (mutationProbability>0.0)
            for(int x=0;x<genome.length;x++)
                if (state.random[thread].nextBoolean(mutationProbability))
                    genome[x] = (byte)(mingene + state.random[thread].nextInt(maxgene-mingene+1));
        }
        
    

    /** Initializes the individual by randomly choosing Bytes uniformly from mingene to maxgene. */
    public void reset(EvolutionState state, int thread)
        {
        for(int x=0;x<genome.length;x++)
            genome[x] = (byte)(mingene + state.random[thread].nextInt(maxgene-mingene+1));
        }

    public int hashCode()
	{
	// stolen from GPIndividual.  It's a decent algorithm.
	int hash = this.getClass().hashCode();

	hash = ( hash << 1 | hash >>> 31 );
        for(int x=0;x<genome.length;x++)
            hash = ( hash << 1 | hash >>> 31 ) ^ genome[x];

	return hash;
	}

    public void readIndividual(final EvolutionState state, 
			       final LineNumberReader reader)
	throws IOException, CloneNotSupportedException
	{
	// First, was I evaluated?
	int linenumber = reader.getLineNumber();
	String s = reader.readLine();
	if (s==null || s.length() < EVALUATED_PREAMBLE.length()) // uh oh
	    state.output.fatal("Reading Line " + linenumber + ": " +
			       "Bad 'Evaluated?' line.");
	DecodeReturn d = new DecodeReturn(s, EVALUATED_PREAMBLE.length());
	Code.decode(d);
	if (d.type!=DecodeReturn.T_BOOLEAN)
	    state.output.fatal("Reading Line " + linenumber + ": " +
			       "Bad 'Evaluated?' line.");
	evaluated = (d.l!=0);

	// Next, what's my fitness?
	fitness.readFitness(state,reader);

	// get the size of the genome
	int lll = Integer.parseInt( reader.readLine() );

	genome = new byte[ lll ];

	for( int i = 0 ; i < genome.length ; i++ )
	    genome[i] = Byte.parseByte( reader.readLine() );

	}

    public void printIndividualForHumans(final EvolutionState state,
					 final int log, 
					 final int verbosity)
	{
	state.output.println(EVALUATED_PREAMBLE + (evaluated ? "true" : "false"), 
	  verbosity, log);
	fitness.printFitnessForHumans(state,log,verbosity);
	String s = "";
	for( int i = 0 ; i < genome.length ; i++ )
	    s = s + " " + genome[i];
	state.output.println( s, verbosity, log );
	}

    public void printIndividual(final EvolutionState state,
				final int log, 
				final int verbosity)
	{
	state.output.println(EVALUATED_PREAMBLE + Code.encode(evaluated), 
			     verbosity, log);
	fitness.printFitness(state,log,verbosity);
	state.output.println( ""+genome.length, verbosity, log );
	for( int i = 0 ; i < genome.length ; i++ )
	    state.output.println( "" + genome[i], verbosity, log );
	}

    public void printIndividual(final EvolutionState state,
				final PrintWriter writer)
	{
	writer.println(EVALUATED_PREAMBLE + Code.encode(evaluated));
	fitness.printFitness(state,writer);
	writer.println( genome.length );
	for( int i = 0 ; i < genome.length ; i++ )
	    writer.println( genome[i] );
	}

    public boolean equals(Object ind)
	{
	if (!(this.getClass().equals(ind.getClass()))) return false; // SimpleRuleIndividuals are special.
	ByteVectorIndividual i = (ByteVectorIndividual)ind;
	if( genome.length != i.genome.length )
	    return false;
	for( int j = 0 ; j < genome.length ; j++ )
	    if( genome[j] != i.genome[j] )
		return false;
	return true;
	}

    public Object getGenome()
        { return genome; }
    public void setGenome(Object gen)
        { genome = (byte[]) gen; }
    public long genomeLength()
        { return genome.length; }
    }
