package ec.vector;

import ec.*;
import ec.util.*;
import java.io.Serializable;
import java.util.*;
import java.io.*;

/*
 * GeneVectorIndividual.java
 * Created: Tue Mar 13 15:03:12 EST 2001
 */

/**
 * GeneVectorIndividual is a VectorIndividual whose genome is an array of 
 * objects which subclass from VectorGene.  You will need to implement these
 * gene objects as you see fit.
 * The default mutation method calls the <tt>mutate</tt> method on genes
 * with <tt>mutationProbability</tt>.
 *
 * To keep things simple, GeneVectorIndividuals don't have constraints and
 * all the other stuff you see elsewhere in ECJ.  The downside of this is that
 * each GeneVectorIndividual must retain a pointer to the prototypical VectorGene
 * instance.
 
 <p><b>Parameter bases</b><br>
 <table>

 <tr><td valign=top><tt>gene</tt></td>
 <td>vector-gene class for this individual</td></tr>
 </table>

 <p><b>Default Base</b><br>
 vector.gene-vect-ind


 * @author Liviu Panait
 * @author Sean Luke
 * @version 1.0
 */

public class GeneVectorIndividual extends VectorIndividual
    {
    public static final String P_GENEVECTORINDIVIDUAL = "gene-vect-ind";
    public final static String P_GENE = "gene";

    // GeneVectorIndividual doesn't have Constraints.  It's designed to be simple
    // and easy to use.  But the downside of that is that we must hold onto a
    // prototype of our VectorGene.  

    public VectorGene gene_prototype;
    public VectorGene[] genome;
    
    public Parameter defaultBase()
        {
        return VectorDefaults.base().push(P_GENEVECTORINDIVIDUAL);
        }

    public Object protoClone() throws CloneNotSupportedException
        {
        GeneVectorIndividual myobj = (GeneVectorIndividual) (super.protoClone());

        // must clone the genome
        myobj.genome = (VectorGene[])(genome.clone());
        
        // clone the genes
        for(int x=0;x<myobj.genome.length;x++)
            myobj.genome[x] = (VectorGene)(myobj.genome[x].protoClone());
            
        // don't clone the prototype
        
        return myobj;
        } 

    public void setup(final EvolutionState state, final Parameter base)
        {
        super.setup(state,base);
        genome = new VectorGene[genomeSize];

	Parameter def = defaultBase();
    
        gene_prototype = (VectorGene)(state.parameters.getInstanceForParameter(base.push(P_GENE), def.push(P_GENE),
                                VectorGene.class));
        gene_prototype.setup(state,base.push(P_GENE));
        }
        
    public void defaultCrossover(EvolutionState state, int thread, VectorIndividual ind)
        {
        GeneVectorIndividual i = (GeneVectorIndividual) ind;
        VectorGene tmp;
        int point;

        if (genome.length != i.genome.length)
            state.output.fatal("Genome lengths are not the same for fixed-length vector crossover");
        switch(crossoverType)
            {
            case C_ONE_POINT:
                point = state.random[thread].nextInt((genome.length / chunksize)+1);
                for(int x=0;x<point*chunksize;x++)
                    { 
                    tmp = i.genome[x];
                    i.genome[x] = genome[x]; 
                    genome[x] = tmp; 
                    }
                break;
            case C_TWO_POINT: 
                int point0 = state.random[thread].nextInt((genome.length / chunksize)+1);
                point = state.random[thread].nextInt((genome.length / chunksize)+1);
                if (point0 > point) { int p = point0; point0 = point; point = p; }
                for(int x=point0*chunksize;x<point*chunksize;x++)
                    {
                    tmp = i.genome[x];
                    i.genome[x] = genome[x];
                    genome[x] = tmp;
                    }
                break;
            case C_ANY_POINT:
                for(int x=0;x<genome.length/chunksize;x++) 
                    if (state.random[thread].nextBoolean(crossoverProbability))
                        for(int y=x*chunksize;y<(x+1)*chunksize;y++)
                            {
                            tmp = i.genome[y];
                            i.genome[y] = genome[y];
                            genome[y] = tmp;
                            }
                break;
            }
        }

    /** Splits the genome into n pieces, according to points, which *must* be sorted. 
        pieces.length must be 1 + points.length */
    public void split(int[] points, Object[] pieces)
        {
        int point0, point1;
        point0 = 0; point1 = points[0];
        for(int x=0;x<pieces.length;x++)
            {
            pieces[x] = new VectorGene[point1-point0];
            System.arraycopy(genome,point0,pieces[x],point0,point1-point0);
            point0 = point1;
            if (x==pieces.length-2)
                point1 = genome.length;
            else point1 = points[x+1];
            }
        }
    
    /** Joins the n pieces and sets the genome to their concatenation.*/
    public void join(Object[] pieces)
        {
        int sum=0;
        for(int x=0;x<pieces.length;x++)
            sum += ((VectorGene[])(pieces[x])).length;
        
        int runningsum = 0;
        VectorGene[] newgenome = new VectorGene[sum];
        for(int x=0;x<pieces.length;x++)
            {
            System.arraycopy(pieces[x], 0, newgenome, runningsum, ((VectorGene[])(pieces[x])).length);
            runningsum += ((VectorGene[])(pieces[x])).length;
            }
        // set genome
        genome = newgenome;
        }

    /** Destructively mutates the individual in some default manner.  The default form
        simply randomizes genes to a uniform distribution from the min and max of the gene values. */
    public void defaultMutate(EvolutionState state, int thread)
        {
        if (mutationProbability>0.0)
            for(int x=0;x<genome.length;x++)
                if (state.random[thread].nextBoolean(mutationProbability))
                    genome[x].mutate(state,thread);
        }
        
    

    /** Initializes the individual by randomly choosing Genes uniformly from mingene to maxgene. */
    public void reset(EvolutionState state, int thread)
        {
        for(int x=0;x<genome.length;x++)
            genome[x].reset(state,thread);
        }

    public int hashCode()
	{
	// stolen from GPIndividual.  It's a decent algorithm.
	int hash = this.getClass().hashCode();

	hash = ( hash << 1 | hash >>> 31 );
        for(int x=0;x<genome.length;x++)
            hash = ( hash << 1 | hash >>> 31 ) ^ genome[x].hashCode();

	return hash;
	}

    public void readIndividual(final EvolutionState state, 
			       final LineNumberReader reader)
	throws IOException, CloneNotSupportedException
	{
	// First, was I evaluated?
	int linenumber = reader.getLineNumber();
	String s = reader.readLine();
	if (s==null || s.length() < EVALUATED_PREAMBLE.length()) // uh oh
	    state.output.fatal("Reading Line " + linenumber + ": " +
			       "Bad 'Evaluated?' line.");
	DecodeReturn d = new DecodeReturn(s, EVALUATED_PREAMBLE.length());
	Code.decode(d);
	if (d.type!=DecodeReturn.T_BOOLEAN)
	    state.output.fatal("Reading Line " + linenumber + ": " +
			       "Bad 'Evaluated?' line.");
	evaluated = (d.l!=0);

	// Next, what's my fitness?
	fitness.readFitness(state,reader);

	// get the size of the genome
	int lll = Integer.parseInt( reader.readLine() );

	genome = new VectorGene[ lll ];

	for( int i = 0 ; i < genome.length ; i++ )
	    genome[i].readGene(state,reader);

	}

    public void printIndividualForHumans(final EvolutionState state,
					 final int log, 
					 final int verbosity)
	{
	state.output.println(EVALUATED_PREAMBLE + (evaluated ? "true" : "false"), 
	  verbosity, log);
	fitness.printFitnessForHumans(state,log,verbosity);
	for( int i = 0 ; i < genome.length ; i++ )
            genome[i].printGeneForHumans(state,log,verbosity);
	state.output.println( "", verbosity, log );
	}

    public void printIndividual(final EvolutionState state,
				final int log, 
				final int verbosity)
	{
	state.output.println(EVALUATED_PREAMBLE + Code.encode(evaluated), 
			     verbosity, log);
	fitness.printFitness(state,log,verbosity);
	state.output.println( ""+genome.length, verbosity, log );
	for( int i = 0 ; i < genome.length ; i++ )
            genome[i].printGene(state,log,verbosity);
	state.output.println( "", verbosity, log );
	}

    public void printIndividual(final EvolutionState state,
				final PrintWriter writer)
	{
	writer.println(EVALUATED_PREAMBLE + Code.encode(evaluated));
	fitness.printFitness(state,writer);
	writer.println( genome.length );
	for( int i = 0 ; i < genome.length ; i++ )
            genome[i].printGene(state,writer);
	writer.println();
	}

    public boolean equals(Object ind)
	{
	if (!(this.getClass().equals(ind.getClass()))) return false; // SimpleRuleIndividuals are special.
	GeneVectorIndividual i = (GeneVectorIndividual)ind;
	if( genome.length != i.genome.length )
	    return false;
	for( int j = 0 ; j < genome.length ; j++ )
	    if( !genome[j].equals(i.genome[j]) )
		return false;
	return true;
	}

    public Object getGenome()
        { return genome; }
    public void setGenome(Object gen)
        { genome = (VectorGene[]) gen; }
    public long genomeLength()
        { return genome.length; }
    }
