package ec.vector;
import java.io.Serializable;
import ec.*;
import ec.util.*;
import java.util.*;
import java.io.*;

 /* 
 * VectorGene.java
 * 
 * Created: Tue Feb 20 13:19:00 2001
 * By: Liviu Panait and Sean Luke
 */

/**
 * VectorGene is an abstract class for describing genes in a GeneVectorIndividual.
 * At the very least, you'll need to implement the hashCode, equals, reset, 
 * printGeneToString, and readGeneFromString methods.
 * 
 * To keep things simple, VectorGenes don't have constraints objects (unless you
 * add some in a subclass).
 *
 
 <p><b>Default Base</b><br>
 vector.vector-gene

 * @author Liviu Panait and Sean luke
 * @version 1.0 
 */
public abstract class VectorGene implements Prototype
    {
    public static final String P_VECTORGENE = "vector-gene";

    /** Generates a hash code for this vector gene -- the vector gene for this is that the hash code
        must be the same for two vector genes that are equal to each other genetically. */
    public abstract int hashCode();
    
    /** Unlike the standard form for Java, this function should return true if this
        vector gene is "genetically identical" to the other vector gene. */
    public abstract boolean equals( final Object other );

    /**
       The reset method randomly reinitializes the vector gene.
    */
    public abstract void reset(final EvolutionState state, final int thread);

    /**
       Mutate the vector gene.  The default form just resets the vector gene.
    */
    public void mutate(final EvolutionState state, final int thread)
	{
	reset(state,thread);
	}

    /**
       Nice printing.  The default form simply calls printGene, but you might want to override this.
     */
    public void printGeneForHumans( final EvolutionState state, final int verbosity, final int log )
        { printGene(state,verbosity,log); }

    /** Prints the vector gene to a string in a fashion readable by readGeneFromString. 
        Override this.  The default form does nothing. */
    public String printGeneToString(final EvolutionState state)
        { return "Override Me!"; }

    /** Reads a vector gene from a string, which may contain a final '\n'.
        Override this method.  The default form does nothing. */
    public void readGeneFromString(final String string, final EvolutionState state)
        { return; }

    /**
        Prints the vector gene in a way that can be read by readGene().  The default form simply
        calls printGeneToString(state).   Override this vector gene to do custom writing to the log,
        or just override printGeneToString(...), which is probably easier to do.
     */
    public void printGene( final EvolutionState state, final int verbosity, final int log )
        { state.output.println(printGeneToString(state),verbosity,log);}

    /**
        Prints the vector gene in a way that can be read by readGene().  The default form simply
        calls printGeneToString(state).   Override this vector gene to do custom writing,
        or just override printGeneToString(...), which is probably easier to do.
     */
    public void printGene( final EvolutionState state, final PrintWriter writer )
        { writer.println(printGeneToString(state)); }

    /**
        Reads a vector gene printed by printGene(...).  The default form simply reads a line into
        a string, and then calls readGeneFromString() on that line.  Override this vector gene to do
        custom reading, or just override readGeneFromString(...), which is probably easier to do.
     */
    public void readGene(final EvolutionState state,
			 final LineNumberReader reader)
	throws IOException, CloneNotSupportedException
        { readGeneFromString(reader.readLine(),state); }


    public Parameter defaultBase()
        {
        return VectorDefaults.base().push(P_VECTORGENE);
        }

    public Object protoClone() throws CloneNotSupportedException
        {
	return super.clone();
        }

    public final Object protoCloneSimple()
        {
        try { return protoClone(); }
        catch (CloneNotSupportedException e) 
            { throw new InternalError(); } // never happens
        } 

    public void setup(EvolutionState state, Parameter base)
	{
	}

    }
