package ec.vector;

import ec.*;
import ec.util.*;

/*
 * VectorIndividual.java
 * Created: Tue Mar 13 15:03:12 EST 2001
 */

/**
 * VectorIndividual is the abstract superclass of simple individual representations
 * which consist of vectors of values (booleans, integers, floating-point, etc.)
 *
 * <p>This class contains two methods, defaultCrossover and defaultMutate, which can
 * be overridden if all you need is a simple crossover and a simple mutate mechanism.
 * the VectorCrossoverPipeline and VectorMutationPipeline classes use these methods to do their
 * handiwork.  For more sophisticated crossover and mutation, you'll need to write
 * a custom breeding pipeline.
 *
 * <p>VectorIndividual defines three common types of crossover which you should
 * implement in your defaultCrossover method: one-point, two-point, and any-point
 * (otherwise known as "uniform") crossover.
 *
 * <p>VectorIndividual is typically used for fixed-length vector representations;
 * however, it can also be used with variable-length representations.  Two methods have
 * been provided in all subclasses of VectorIndividual to help you there: split and
 * join, which you can use to break up and reconnect VectorIndividuals in a variety
 * of ways.  Note that you may want to override the reset() method to create individuals
 * with different initial lengths.
 *
 * <p>VectorIndividuals should belong to the species VectorSpecies (or some subclass of it).
 *
 
 <p><b>Parameters</b><br>
 <table>
 <tr><td valign=top><tt>genome-size</tt><br>
 <font size=-1>int &gt;= 1</font></td>
 <td valign=top>(size of the genome)</td></tr>

 <tr><td valign=top><tt>chunk-size</tt><br>
 <font size=-1>1 &lt;= int &lt;= genome-size</font></td>
 <td valign=top>(the chunk size for crossover (crossover will only occur on chunk boundaries))</td></tr>

 <tr><td valign=top><tt>crossover-type</tt><br>
 <font size=-1>string, one of: one, two, any</font></td>
 <td valign=top>(default crossover type (one-point, two-point, or any-point (uniform) crossover)</td></tr>

 <tr><td valign=top><tt>crossover-prob</tt><br>
 <font size=-1>0.0 &gt;= float &gt;= 1.0 </font></td>
 <td valign=top>(probability that a gene will get crossed over during any-point crossover)</td></tr>

 <tr><td valign=top><tt>mutation-prob</tt><br>
 <font size=-1>0.0 &lt;= float &lt;= 1.0 </font></td>
 <td valign=top>(probability that a gene will get mutated over default mutation)</td></tr>

 </table>

 * @author Sean Luke
 * @version 1.0
 */

public abstract class VectorIndividual extends Individual
    {
    public final static int C_ONE_POINT = 0;
    public final static int C_TWO_POINT = 1;
    public final static int C_ANY_POINT = 3;
    public final static String P_CROSSOVERTYPE = "crossover-type";
    public final static String P_CHUNKSIZE = "chunk-size";
    public final static String V_ONE_POINT = "one";
    public final static String V_TWO_POINT = "two";
    public final static String V_ANY_POINT = "any";
    public final static String P_MUTATIONPROB = "mutation-prob";
    public final static String P_CROSSOVERPROB = "crossover-prob";
    public final static String P_GENOMESIZE = "genome-size";
    /**
       Evaluated string to appear when printed
     */
    public static final String EVALUATED_PREAMBLE = "Evaluated: ";
    
    /** Probability that a gene will mutate */
    public float mutationProbability;
    /** Probability that a gene will cross over -- ONLY used in V_ANY_POINT crossover */
    public float crossoverProbability;
    /** What kind of crossover do we have? */
    public int crossoverType;
    /** How big of a genome should we create on initialization? */
    public int genomeSize;
    /** How big of chunks should we define for crossover? */
    public int chunksize;

    public void setup(final EvolutionState state, final Parameter base)
	{
	Parameter def = defaultBase();
    
	genomeSize = state.parameters.getInt(base.push(P_GENOMESIZE),def.push(P_GENOMESIZE),1);
	if (genomeSize==0)
	    state.output.error("VectorIndividual must have a genome size > 0");
    
        chunksize = state.parameters.getIntWithDefault(base.push(P_CHUNKSIZE),def.push(P_CHUNKSIZE),1);
        if (chunksize <= 0 || chunksize > genomeSize)
            state.output.fatal("VectorIndividual must have a chunksize which is > 0 and < genomeSize",
                base.push(P_CHUNKSIZE),def.push(P_CHUNKSIZE));
        if (genomeSize % chunksize != 0)
            state.output.fatal("VectorIndividual must have a genomeSize which is a multiple of chunksize",
                base.push(P_CHUNKSIZE),def.push(P_CHUNKSIZE));

	mutationProbability = state.parameters.getFloat(
	    base.push(P_MUTATIONPROB),def.push(P_MUTATIONPROB),0.0,1.0);
	if (mutationProbability==-1.0)
	    state.output.error("VectorIndividual must have a mutation probability between 0.0 and 1.0 inclusive",
			       base.push(P_MUTATIONPROB),def.push(P_MUTATIONPROB));
    
	String ctype = state.parameters.getStringWithDefault(base.push(P_CROSSOVERTYPE),
							     def.push(P_CROSSOVERTYPE), "");
	crossoverType = C_ONE_POINT;
	if (ctype==null)
	    state.output.warning("No crossover type given for VectorIndividual, assuming one-point crossover",
				 base.push(P_CROSSOVERTYPE),def.push(P_CROSSOVERTYPE));
	else if (ctype.equalsIgnoreCase(V_ONE_POINT))
	    crossoverType=C_ONE_POINT;  // redundant
	else if (ctype.equalsIgnoreCase(V_TWO_POINT))
	    crossoverType=C_TWO_POINT;
	else if (ctype.equalsIgnoreCase(V_ANY_POINT))
	    crossoverType=C_ANY_POINT;
	else state.output.error("VectorIndividual given a bad crossover type: " + ctype,
				base.push(P_CROSSOVERTYPE),def.push(P_CROSSOVERTYPE));
    
	if (crossoverType==C_ANY_POINT)
	    {
	    crossoverProbability = state.parameters.getFloat(
		base.push(P_CROSSOVERPROB),def.push(P_CROSSOVERPROB),0.0,1.0);
	    if (crossoverProbability==-1.0)
		state.output.error("If it's going to use any-point crossover, VectorIndividual must have a crossover probability between 0.0 and 1.0 inclusive",
				   base.push(P_CROSSOVERPROB),def.push(P_CROSSOVERPROB));
	    }
	else crossoverProbability = (float)0.0;
	state.output.exitIfErrors();
	}

    /** Destructively crosses over the individual with another in some default manner.  In most
        implementations provided in ECJ, one-, two-, and any-point crossover is done with a 
        for loop, rather than a possibly more efficient approach like arrayCopy().  The disadvantage
        is that arrayCopy() takes advantage of a CPU's bulk copying.  The advantage is that arrayCopy()
        would require a scratch array, so you'd be allocing and GCing an array for every crossover.
        Dunno which is more efficient.  */
    public void defaultCrossover(EvolutionState state, int thread, 
				 VectorIndividual ind) { }

    /** Destructively mutates the individual in some default manner.  The default version calls reset()*/
    public void defaultMutate(EvolutionState state, int thread) { reset(state,thread); }

    /** Initializes the individual. */
    public abstract void reset(EvolutionState state, int thread);

    /** Returns the gene array.  If you know the type of the array, you can cast it and work on
        it directly.  Otherwise, you can still manipulate it in general, because arrays (like
        all objects) respond to clone() and can be manipulated with arrayCopy without bothering
        with their type.  This might be useful in creating special generalized crossover operators
        -- we apologize in advance for the fact that Java doesn't have a template system.  :-( 
        The default version returns null. */
    public Object getGenome() { return null; }
    
    /** Sets the gene array.  See getGenome().  The default version does nothing.
        @see getGenome()
    */
    public void setGenome(Object gen) { }

    /** Returns the length of the gene array.  By default, this method returns 0. */
    public long genomeLength() { return 0; }

    /** Splits the genome into n pieces, according to points, which *must* be sorted. 
        pieces.length must be 1 + points.length.  The default form does nothing -- be careful
        not to use this method if it's not implemented!  It should be trivial to implement it
        for your genome -- just like at the other implementations.  */
    public void split(int[] points, Object[] pieces) { }

    /** Joins the n pieces and sets the genome to their concatenation.  The default form does nothing. 
      It should be trivial to implement it
        for your genome -- just like at the other implementations.  */
    public void join(Object[] pieces) { }
    
    public long size() { return genomeLength(); }
    }
