package sim.app.hexabugs;
import sim.engine.*;
import sim.util.*;
import sim.field.grid.*;
import ec.util.*;
import java.io.*;

public /*strictfp*/ class HexaBugs extends SimState
    {
    public static double minIdealTemp = 17000;
    public static double maxIdealTemp = 31000;
    public static double minOutputHeat = 6000;
    public static double maxOutputHeat = 10000;

    public static double evaporationRate = 0.993;
    public static double diffusionRate = 1.0;
    public static final double MAX_HEAT = 32000;
    public static double randomMovementProbability = 0.1;

    public int gridHeight = 100;
    public int gridWidth = 100;
    public int bugCount = 100;
    HexaBug[] bugs;

    public double getMinimumIdealTemperature() { return minIdealTemp; }
    public void setMinimumIdealTemperature( double temp ) { if( temp <= maxIdealTemp ) minIdealTemp = temp; }
    public double getMaximumIdealTemperature() { return maxIdealTemp; }
    public void setMaximumIdealTemperature( double temp ) { if( temp >= minIdealTemp ) maxIdealTemp = temp; }
    public double getMinimumOutputHeat() { return minOutputHeat; }
    public void setMinimumOutputHeat( double temp ) { if( temp <= maxOutputHeat ) minOutputHeat = temp; }
    public double getMaximumOutputHeat() { return maxOutputHeat; }
    public void setMaximumOutputHeat( double temp ) { if( temp >= minOutputHeat ) maxOutputHeat = temp; }
    public double getEvaporationConstant() { return evaporationRate; }
    public void setEvaporationConstant( double temp ) { if( temp >= 0 && temp <= 1 ) evaporationRate = temp; }
    public double getDiffusionConstant() { return diffusionRate; }
    public void setDiffusionConstant( double temp ) { if( temp >= 0 && temp <= 1 ) diffusionRate = temp; }
    public double getRandomMovementProbability() { return randomMovementProbability; }
    public void setRandomMovementProbability( double t )
        {
        if (t >= 0 && t <= 1)
            {
            randomMovementProbability = t;
            for( int i = 0 ; i < bugCount ; i++ )
                bugs[i].setRandomMovementProbability( randomMovementProbability );
            }
        }
    public double getMaximumHeat() { return MAX_HEAT; }
    public int getGridHeight() { return gridHeight; }
    public int getGridWidth() { return gridWidth; }
    public int getBugCount() { return bugCount; }

    public DoubleGrid2D valgrid = new DoubleGrid2D(gridWidth, gridHeight,0);
    public DoubleGrid2D valgrid2 = new DoubleGrid2D(gridWidth, gridHeight, 0);
    public SparseGrid2D buggrid = new SparseGrid2D(gridWidth, gridHeight);

    /** Creates a HexaBugs simulation with the given random number seed. */
    public HexaBugs(long seed)
        {
        // we build a schedule that has two orders per tick: for the bugs, then the Hexa decreaser
        super(new MersenneTwisterFast(seed), new Schedule(2));
        bugs = new HexaBug[bugCount];
        }
        
    /** Resets and starts a simulation */
    public void start()
        {
        super.start();  // clear out the schedule
        
        // make new grids
        valgrid = new DoubleGrid2D(gridWidth, gridHeight,0);
        valgrid2 = new DoubleGrid2D(gridWidth, gridHeight, 0);
        buggrid = new SparseGrid2D(gridWidth, gridHeight);  // we're doing toroidal, so specify dimensions

        // make the decreaser 
        
        Steppable decreaser = new Steppable()
            {
            public void step(SimState state)
                {
                // locals are faster than instance variables
                final DoubleGrid2D _valgrid = valgrid;
                final DoubleGrid2D v = valgrid;  // shorter
                final double[][] _valgrid_field = valgrid.field;
                final double[][] _valgrid2_field = valgrid2.field;
                final int _gridWidth = gridWidth;
                final int _gridHeight = gridHeight;
                
                double average;
                DoubleBag temp = new DoubleBag();
               // for each x and y position
                for(int x=0;x< _gridWidth;x++)
                    for(int y=0;y< _gridHeight;y++)
                        {
                        // this is faster...
                        
                        average = 
                            (_valgrid_field[x][y] + 
                            _valgrid_field[v.stx(v.ulx(x,y))][v.sty(v.uly(x,y))] +
                            _valgrid_field[v.stx(v.urx(x,y))][v.sty(v.ury(x,y))] + 
                            _valgrid_field[v.stx(v.dlx(x,y))][v.sty(v.dly(x,y))] + 
                            _valgrid_field[v.stx(v.drx(x,y))][v.sty(v.dry(x,y))] + 
                            _valgrid_field[v.stx(v.upx(x,y))][v.sty(v.upy(x,y))] + 
                            _valgrid_field[v.stx(v.downx(x,y))][v.sty(v.downy(x,y))]) / 7.0;
                        
                        // ... but this is simpler to write and more clear...
                        /*
                        _valgrid.getNeighborsHexagonalDistance(x,y,1,true,temp,null,null);
                        for( int i = 0 ; i < temp.numObjs ; i++ )
                        average += temp.objs[i];
                        average /= (1+temp.numObjs);
                        */
                        
                        // load the new value into HexaBugs.this.valgrid2
                        _valgrid2_field[x][y] = evaporationRate * 
                            (_valgrid_field[x][y] + diffusionRate * 
                                (average - _valgrid_field[x][y]));
                        }
                
                // copy HexaBugs.this.valgrid2 to HexaBugs.this.valgrid
                _valgrid.setTo(valgrid2);
                }
            };
        
        // Schedule the Hexa bugs -- we could instead use a RandomSequence, which would be faster
        // But we spend no more than 3% of our total runtime in the scheduler max, so it's not worthwhile
        for(int x=0;x<bugCount;x++)
            {
            bugs[x] = new HexaBug(random.nextDouble() * (maxIdealTemp - minIdealTemp) + minIdealTemp,
                        random.nextDouble() * (maxOutputHeat - minOutputHeat) + minOutputHeat, 
                        MAX_HEAT, randomMovementProbability);
            buggrid.setObjectLocation(bugs[x],random.nextInt(gridWidth),random.nextInt(gridHeight));
            schedule.scheduleRepeating(bugs[x]);
            }
                            
        // Schedule the decreaser to happen after the HexaBugs
        schedule.scheduleRepeating(Schedule.EPOCH,1,decreaser,1);
        }
    
    
    
    
    
    
    
    /** Runs HexaBugs headless (no GUI).  Checkpoints out to files every 500 simulation timesteps.
        If no command-line arguments are provided, a new HexaBugs is started.
        Optionally the user can say "-checkpoint [filename]" and HexaBugs will load a checkpoint
        and continue from that point. */
        
    // This example is overly elaborate, with checkpointing and stuff.
    // But to run a simulation, the basic thing you need to do is:
    //
    //
    // 1. Make a SimState		HexaBugs = new HexaBugs(23);  // seed the RNG with 23, what the heck
    // 2. Tell it we're starting	HexaBugs.start();
    // 3. Loop:				for(...however long I want to run...)
    // 4. 	Step the schedule		if (!HexaBugs.schedule.step(HexaBugs)) break;
    // 5. Finish up			HexaBugs.finish();
    //
    // ...and that's it!
    
    public static void main(String[] args)
        {
        HexaBugs hexaBugs = null;
        
        // should we load from checkpoint?  I wrote this little chunk of code to
        // check for this to give you the general idea.
        
        for(int x=0;x<args.length-1;x++)  // "-checkpoint" can't be the last string
            if (args[x].equals("-checkpoint"))
                {
                SimState state = SimState.readFromCheckpoint(new File(args[x+1]));
                if (state == null)   // there was an error -- it got printed out to the screen, so just quit
                    System.exit(1);
                else if (!(state instanceof HexaBugs))  // uh oh, wrong simulation stored in the file!
                    {
                    System.out.println("Checkpoint contains some other simulation: " + state);
                    System.exit(1);
                    }
                else // we're ready to lock and load!  
                    hexaBugs = (HexaBugs)state;
                }
        
        // ...or should we start fresh?
        if (hexaBugs==null)  // no checkpoint file requested
            {
            hexaBugs = new HexaBugs(System.currentTimeMillis());  // make a new HexaBugs.  Seed the RNG with the time
            hexaBugs.start();  // prep the bugs!
            System.out.println("Starting HexaBugs.  Running for 5000 steps.");
            }
        
        long time;
        while((time = hexaBugs.schedule.time()) < 5000)
            {
            // step the schedule.  This is where everything happens.
            if (!hexaBugs.schedule.step(hexaBugs)) break;   // it won't happen that we end prematurely,
                                                            // but it's worth checking for!

            if (time%100==0 && time!=0) 
		System.out.println("Time Step " + time);
        
            // checkpoint
            if (time%500==0 && time!=0)
                {
                String s = "hb." + time + ".checkpoint";
                System.out.println("Checkpointing to file: " + s);
		hexaBugs.writeToCheckpoint(new File(s));
                }
            }
            
        hexaBugs.finish();  // we don't use this, but it's good style
        }
        

    }
    
    
    
    
    
