package sim.app.lightcycles;
import sim.engine.*;
import sim.field.grid.*;
import ec.util.*;
import java.io.*;

public class LightCycles extends SimState
{
    public int gridHeight;
    public int gridWidth;
    public int cycleCount;
    
    // The intGrid holds the walls drawn behind the cycles...
    public IntGrid2D grid;
    // while the sparsegrid holds the cycles themselves.
    public SparseGrid2D cycleGrid;

    /** Creates a LightCycles simulation with the given random number seed. */
    public LightCycles(long seed)
    {
        this(seed, 100, 100, 10);
    }
        
    public LightCycles(long seed, int width, int height, int count)
    {
        super(new MersenneTwisterFast(seed), new Schedule(2));
        gridWidth = width; gridHeight = height; cycleCount = count;
        createGrids();
    }

    private void createGrids()
    {
        grid = new IntGrid2D(gridWidth, gridHeight,0);
        cycleGrid = new SparseGrid2D(gridWidth, gridHeight);
    }
    
    /** Resets and starts a simulation */
    public void start()
    {
        super.start();  // clear out the schedule
        
        // make new grids
        createGrids();

        // Create the cycles, add to both grid and schedule
        for(int x=0;x<cycleCount;x++)
        {
            Cycle c = new Cycle(x+5, random.nextInt(4)+1);
            cycleGrid.setObjectLocation(c, random.nextInt(gridWidth), random.nextInt(gridHeight));
            c.stopper = schedule.scheduleRepeating(c);
        }
    }
    
    public static void main(String[] args)
    {
        LightCycles cycles = null;
        
        // should we load from checkpoint?
        for(int x=0;x<args.length-1;x++)  // "-checkpoint" can't be the last string
            if (args[x].equals("-checkpoint"))
            {
                SimState state = SimState.readFromCheckpoint(new File(args[x+1]));
                if (state == null)   // there was an error -- it got printed out to the screen, so just quit
                    System.exit(1);
                else if (!(state instanceof LightCycles))  // uh oh, wrong simulation stored in the file!
                {
                    System.out.println("Checkpoint contains some other simulation: " + state);
                    System.exit(1);
                }
                else // lock and load!  
                    cycles = (LightCycles)state;
            }
        
        // ...or should we start fresh?
        if (cycles==null)  // no checkpoint file requested
        {
            cycles = new LightCycles(System.currentTimeMillis());  // make a new lightcycles.  Seed the RNG with the time
            cycles.start();
            System.out.println("Starting LightCycles.  Running for 5000 steps.");
        }
        
        long time;
        while((time = cycles.schedule.time()) < 5000)
        {
            // step the schedule.  This is where everything happens.
            if (!cycles.schedule.step(cycles)) 
                break;   				// it won't happen that we end prematurely,
                                                        // but it's worth checking for!

            if (time%100==0 && time!=0) 
		System.out.println("Time Step " + time);
        
            // checkpoint
            if (time%500==0 && time!=0)
            {
                String s = "lc." + time + ".checkpoint";
                System.out.println("Checkpointing to file: " + s);
		cycles.writeToCheckpoint(new File(s));
            }
        }
            
        cycles.finish();  // we don't use this, but it's good style
    }
}
    