package sim.display3d;

import sim.engine.*;
import sim.portrayal3d.*;
import sim.portrayal3d.simple.*;
import sim.util.gui.*;
import sim.util.media.*;
import sim.display.*;

import java.io.*;
import java.util.*;
import java.awt.*;
import java.awt.geom.*;
import java.awt.event.*;
import java.awt.image.*;

import javax.swing.*;
import javax.vecmath.*;
import javax.media.*;
import javax.media.j3d.*;

import com.sun.j3d.utils.universe.*;
import com.sun.j3d.utils.behaviors.vp.*;


/**
 * @author Gabriel Balan 
 * 
 * @see sim.display.Display2D
 **/
public class Display3D extends JComponent implements Steppable
{
    ArrayList portrayals = new ArrayList();
    JScrollPane display;
    JViewport port;
    public JFrame frame;
    Stoppable stopper;
    GUIState simulation;
    JComponent header;
	JButton movieButton;
	JButton snapshotButton;
	JButton optionButton;
	
    long interval = 1;
    protected Object intervalLock = new Object();
    protected void setInterval(long i) { synchronized(intervalLock) { interval = i; } }
    protected long getInterval() { synchronized(intervalLock) { return interval; } }
    
    private BranchGroup mRoot = null;
    private SimpleUniverse mUniverse = null;
	private CapturingCanvas3D mCanvas = null;
	private GraphicsContext3D mGC = null;
	private BranchGroup mAxes = null;
	   
    long lastTimestamp = Schedule.BEFORE_SIMULATION;
    public MovieMaker movmaker;    

	boolean tapeMovie = false;
	private File movieFile;
	private int fps=10;
	private Format movieFormat;

	
    JPopupMenu popup;
    JToggleButton togglebutton;  // for popup

	/** 
	 * Creates a frame holding the Display3D.  This is the best method to create the frame,
	 * rather than making a frame and putting the Display3D in it.  If you prefer the latter,
	 * then you need to handle two things.  
	 * -First, you need to set the Display3D's "frame" field to it
	 * -Second, when the frame is disposed, you need to call quit() on the Display3D.  
	 **/
	public JFrame createFrame()
	{
		JFrame frame = new JFrame()
		{
			public void dispose()
			{
				quit();       // shut down the movies
				super.dispose();
			}
		};
            
		frame.setResizable(true);
        
		frame.getContentPane().setLayout(new BorderLayout());
		frame.getContentPane().add(this,BorderLayout.CENTER);
        this.frame = frame;
		frame.setTitle(simulation.getName() + " Display");
		frame.pack();
		return frame;
	}
	
	
	
	int subgraphCount = 0;
	class Portrayal3DHolder
    {
		public Portrayal3D portrayal;
		public String name;
		public JCheckBoxMenuItem menuItem;
		public int subgraphIndex;
		public String toString() { return name; }
		public Portrayal3DHolder(Portrayal3D p, String n, boolean selected)
		{
		    portrayal=p; 
		    name=n;
		    menuItem = new JCheckBoxMenuItem(name,selected);
		    subgraphIndex = subgraphCount;
		    subgraphCount++;
			menuItem.addActionListener(new ActionListener()
			{
				public void actionPerformed(ActionEvent e)
		        {
		        	mVisibilityMask.flip(subgraphIndex);
			        mLayerSwitch.setChildMask(mVisibilityMask);
		        }
			});

		}
    }


    /** Resets the Display3D so it reschedules itself.  This is
     *  useful when reusing the Display3D.  
     */
	public void reset()
	{
        // now reschedule myself
        if (stopper!=null) stopper.stop();
        if (getInterval() < 1) setInterval(1);  // just in case...
        stopper = simulation.scheduleImmediateRepeat(true,this);
	}


	public void attach(Portrayal3D portrayal, String name){attach(portrayal, name, true);}
	public void attach(Portrayal3D portrayal, String name, boolean selected)
    {
		if(mVisibilityMask != null)
			System.err.println("Visibility BitSet already constructed. This was not supposed to happen.");
		Portrayal3DHolder p = new Portrayal3DHolder(portrayal,name, selected);
		portrayals.add(p);
		popup.add(p.menuItem);
		
    }
    
    public ArrayList detatchAll()
    {    
        ArrayList old = portrayals;
        popup.removeAll();
        portrayals = new ArrayList();
        mVisibilityMask = null;
        subgraphCount = 0;
        return old;

    }
       
    public Display3D(final int width, final int height, GUIState simulation, long interval)
    {
        setInterval(interval);
        this.simulation = simulation;
        reset();  // must happen AFTER state is assigned
        
        header = new JComponent() { };
        header.setLayout(new BoxLayout(header,BoxLayout.X_AXIS));
        
        togglebutton = new JToggleButton(Display2D.LAYERS_ICON);
        togglebutton.setBorder(BorderFactory.createEmptyBorder(4,4,4,4));
        header.add(togglebutton);
        
        //Create the popup menu.
        JCheckBoxMenuItem menuItem;
        popup = new JPopupMenu();
        popup.setLightWeightPopupEnabled(false);

        //Add listener to components that can bring up popup menus.
        togglebutton.addMouseListener(new MouseAdapter()
		{
			public void mousePressed(MouseEvent e)
		    {
			    popup.show(e.getComponent(),
			                togglebutton.getLocation().x,
			                togglebutton.getLocation().y+
			                    togglebutton.getSize().height);
		    }
			public void mouseReleased(MouseEvent e) 
		    {
			    togglebutton.setSelected(false);
		    }
		});

        
		movieButton = new JButton(Display2D.MOVIE_OFF_ICON);
		movieButton.setBorder(BorderFactory.createEmptyBorder(4,4,4,4));
		movieButton.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent e)
	        {
				if (movmaker==null) startMovie();
				else 			    stopMovie();
	        }
		});
		header.add(movieButton);
		
		snapshotButton = new JButton(Display2D.CAMERA_ICON);
		snapshotButton.setBorder(BorderFactory.createEmptyBorder(4,4,4,4));
		snapshotButton.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent e)
			{
				takeSnapshot();
			}
        });
		header.add(snapshotButton);

		optionButton = new JButton(Display2D.OPTIONS_ICON);
		optionButton.setBorder(BorderFactory.createEmptyBorder(4,4,4,4));
		optionButton.setToolTipText("Options");
		optionButton.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent e)
			{
				optionsFrame.setVisible(true);
			}
		});
		header.add(optionButton);
        
        header.add(new NumberTextField("  Skip: ", 1, false)
                    {
                    public double newValue(double newValue)
                        {
                        int val = (int) newValue;
                        if (val < 1) val = 1;
                        // reset with a new interval
                        setInterval(val);
                        reset();                        
                        return val;
                        }
                    });
        
        setLayout(new BorderLayout());
        add(header,BorderLayout.NORTH);
        
		createOptionsPanel();

    }
    



	Transform3D globalRotation = null;
   	TransformGroup mTransformGroupLevel0 = null;
   	TransformGroup mTransformGroupLevel1 = null;
   	Switch mLayerSwitch = null;
   	BitSet mVisibilityMask = null;
	   	
    public void createSceneGraph(boolean inspectingEnabled)
    {
		if(mUniverse!= null)
		{
//	    	System.out.println("Re create Scene graph");
   		   	mCanvas.stopRenderer();
			mUniverse.removeAllLocales();
			remove(mCanvas);
			mAxes = null;
		}
		else
//	    	System.out.println("Create Scene graph for the first time");
	
		//attach axes
		attach(new Axes(0.01f, true),"Axes", false);

    	mCanvas = new CapturingCanvas3D(SimpleUniverse.getPreferredConfiguration());
        
		mGC = mCanvas.getGraphicsContext3D();			
		add(mCanvas, BorderLayout.CENTER);
		
    	mUniverse = new SimpleUniverse(mCanvas);
    	//take the viewing point a step back
    	mUniverse.getViewingPlatform().setNominalViewingTransform();

		mRoot = new BranchGroup();
    	
    	
		globalRotation = new Transform3D();
		mTransformGroupLevel0 = new TransformGroup();
		//for spin behavior
		mTransformGroupLevel0.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);
		mTransformGroupLevel1 = new TransformGroup();
		mLayerSwitch = new Switch(Switch.CHILD_MASK);
		mLayerSwitch.setCapability(Switch.ALLOW_SWITCH_WRITE);
		mLayerSwitch.setCapability(Switch.ALLOW_CHILDREN_READ);
		mVisibilityMask = new BitSet(subgraphCount);
		mVisibilityMask.set(0, subgraphCount);
		//the default value for axes is "hidden"
		mVisibilityMask.clear(subgraphCount-1);
		mLayerSwitch.setChildMask(mVisibilityMask);
    	

///////////////// MODEL /////////////////
        Iterator iter = portrayals.iterator();
        while (iter.hasNext())
        {
            Portrayal3D p = ((Portrayal3DHolder)(iter.next())).portrayal;
			Object obj = (p instanceof FieldPortrayal3D)? ((FieldPortrayal3D)p).getField(): null;
            mLayerSwitch.addChild(p.getModel(obj,null));
        }

        mTransformGroupLevel1.addChild(mLayerSwitch);
		mTransformGroupLevel1.setTransform(globalRotation);
		mTransformGroupLevel0.addChild(mTransformGroupLevel1);
		
		if(inspectingEnabled)
		{
			BoundingSphere bounds = new BoundingSphere(new Point3d(0.0,0.0,0.0), 100.0);
			mSelectBehavior =  new SelectionBehavior(mCanvas, mRoot, bounds, simulation);
			selectBehCheckBox.setSelected(true);
		}
		else
		{
			selectBehCheckBox.setEnabled(false);
			selectBehCheckBox.setToolTipText("The application specifically disabled picking/selecting.");
		}

//		AxisAngle4d axis =	new AxisAngle4d(Math.PI/2,Math.PI/2,Math.PI/2,.5);
//		Transform3D axisT3D = new Transform3D();
//		axisT3D.setRotation(axis);

		mSpinRotator = new RotationInterpolator(new Alpha(0, 8000), 
												mTransformGroupLevel0); 
		
		//note that Alpha's loop count is ZERO
		//beacuse I want the spin behaivor turned off.
		//Don't forget to put a -1 instead if you
		//want endless spinning. 
//		mSpinRotator.getAlpha().pause();
		Component[] ca = spinBehPanel.getComponents();
		for(int i=0;i<ca.length; i++)
			ca[i].setEnabled(false);//because there's no rotation (the "0" loopCount)
		mRoot.addChild(mSpinRotator);
		
		
		mTransformGroupLevel1.setCapability(BranchGroup.ALLOW_CHILDREN_WRITE);
		mTransformGroupLevel1.setCapability(BranchGroup.ALLOW_CHILDREN_EXTEND);
		
        mRoot.addChild(mTransformGroupLevel0);
        //in order to add/remove spinBehabior, I need these:
		mRoot.setCapability(BranchGroup.ALLOW_CHILDREN_EXTEND);
		mRoot.setCapability(BranchGroup.ALLOW_CHILDREN_WRITE);
		mRoot.setCapability(BranchGroup.ALLOW_BOUNDS_READ);

		mSpinRotator.setSchedulingBounds(mRoot.getBounds());											

//    	mRoot.compile();//		it works faster un-compiled !!!
	   	mUniverse.addBranchGraph(mRoot);
	   	mCanvas.startRenderer();
    }

	public void refreshBoundsOnBehaviors()
	{
		mSpinRotator.setSchedulingBounds(mRoot.getBounds());
		if(mOrbitBehavior != null)
			mOrbitBehavior.setSchedulingBounds(mRoot.getBounds());
		if(mSelectBehavior != null)
			mSelectBehavior.setSchedulingBounds(mRoot.getBounds());
	}

	/** 
	 * It is private so it won't get replaced. I used to have it wrap in a BranchGroup and add/remove it 
	 * dynamically, but i guess it simpler this way. Given good arguments, I could switch back.
	 * 
	 * Teh RotationInterpolator can be obtained though, @see getSpinnterpolator(). Now you can do what ever you like:
	 *  
	 * float getMaximumAngle()
	 * float getMinimumAngle() 
	 * void setMaximumAngle(float angle) 
	 * void setMinimumAngle(float angle)	
	 * void setTransformAxis(Transform3D axisOfTransform)
	 * Transform3D getTransformAxis()
	 * void setAlpha(Alpha alpha) 
	 * Alpha getAlpha()
	 * Bounds getSchedulingBounds()
	 * void setSchedulingBounds(Bounds region)
	 * 
	 * it might be a good idea to set the right capabilities immediately after createScene
	 **/
	private RotationInterpolator mSpinRotator = null;
	public RotationInterpolator getSpinnterpolator(){ return mSpinRotator;}

	public Transform3D getYaxis(){return new Transform3D();}
	public Transform3D getXaxis(){Transform3D xAxis = new Transform3D();	xAxis.rotZ((float)Math.PI/2); return xAxis;}
	public Transform3D getZaxis(){Transform3D zAxis = new Transform3D(); 	zAxis.rotX(-(float)Math.PI/2); return zAxis;}

	
	public void setSpinBehaviorEnabled(boolean b)
	{
		if(b)
			mSpinRotator.getAlpha().resume(); 
		else 
			mSpinRotator.getAlpha().pause();
	}
	
	private OrbitBehavior mOrbitBehavior = null;
	/**
	 * Oribit behavior is moving the point of view
	 * around the scene.
	 */
	public void setOrbitBehaviorEnabled(boolean b)
	{
		if(b)
		{
			if(mOrbitBehavior == null)
			{
				mOrbitBehavior=  new OrbitBehavior(mCanvas, OrbitBehavior.REVERSE_ALL);
				orbitRotateCheckBox.setSelected(true);
				orbitTranslateCheckBox.setSelected(true);
				orbitZoomCheckBox.setSelected(true);
				mOrbitBehavior.setSchedulingBounds(mRoot.getBounds());
			}
			mUniverse.getViewingPlatform().setViewPlatformBehavior(mOrbitBehavior);
		}
		else
			if(mOrbitBehavior!=null)
				mUniverse.getViewingPlatform().setViewPlatformBehavior(null);
		orbitBehCheckBox.setSelected(b);
	}
	
	private SelectionBehavior mSelectBehavior = null; 
	public void setSelectBehaviorEnabled(boolean b)
	{
		if(mSelectBehavior==null)
			throw new RuntimeException("SelectBehavior is turned off. Call createScene(true) to allow picking/selecting/etc");
		mSelectBehavior.setEnable(b);
		selectBehCheckBox.setSelected(b);
	}

    public void destroy()
    {
    	if(mUniverse != null)
	    	mUniverse.removeAllLocales();
    }

    public void step(final SimState state)
    {
		long timestamp = simulation.state.schedule.time();	
		if (timestamp % getInterval() != 0)
			return;    	
    	
		//update model ONLY on what is actully on screen. 
		if(!frame.isVisible() || frame.getState()==Frame.ICONIFIED)
			return;
		boolean waitForRenderer = false;
        Iterator iter = portrayals.iterator();
		while(iter.hasNext())
		{
			Portrayal3DHolder ph = 	(Portrayal3DHolder)iter.next();
			if(mVisibilityMask.get(ph.subgraphIndex))
			{
				//update model ONLY on what is actully on screen. 
				ph.portrayal.getModel(
						(ph.portrayal instanceof FieldPortrayal3D)? ((FieldPortrayal3D)ph.portrayal).getField(): null,
						(TransformGroup)mLayerSwitch.getChild(ph.subgraphIndex));
				waitForRenderer = true;
				/** TODO sometimes, this is not enough.
				 * the models are called update upon, but 
				 * nothing actully changes; as a result,
				 * the canvas does not paint a new frame.
				 * => some change must be artificially perfomed
				 * to force post draw notification. Otherwise
				 * the simulation gets stuck at the wait(0)
				 **/
			}
		}
		if(!waitForRenderer)
			return;
		synchronized(mCanvas)
		{
			try
			{
				if (!Thread.currentThread().isInterrupted())
		            mCanvas.wait(0);
			}catch(InterruptedException ex){
				Thread.currentThread().interrupt();
			}
		
			/* 
			 * So far, the Canvas3D is not rendering off-screen, so new frames are not
			 * produced when the display is hidden. Therefore, no point in copying the
			 * same old frame over and over into the movie.
			 * 
			 * Canvas3D could be used in off-screen mode to make a movie without rendering
			 * the frames on the screen, but I really don't think this is the bottle neck.
			 * 
			 * Meanwhile the simulation goes faster when this is hidden, 
			 * because the renderer is short-circuited
			 */
			if(tapeMovie) movmaker.add(mCanvas.getFrameAsImage());
		}
    }
    
	public static String ensureFileEndsWith(String filename, String ending)
	{
		// do we end with the string?
		if (filename.regionMatches(false,filename.length()-ending.length(),ending,0,ending.length()))
		    return filename;
		else return filename + ending;
	}

	private Rectangle2D getImageSize()
	{
//	    Dimension s = getPreferredSize();
		Dimension s = getSize();
	    Rectangle2D clip =  new Rectangle2D.Double(0,0,s.width,s.height);
	    if(mCanvas == null)
	    	return null;
	    Rectangle bounds = mCanvas.getGraphics().getClipBounds();
	    if(bounds != null)
		    clip = clip.createIntersection(bounds);
		return clip;
	}


    /** Ought only be done from the main event loop */
    public void takeSnapshot()
	{
		Rectangle2D imgBounds = getImageSize();
		mCanvas.setWritingParams(getImageSize(), false);
		// NOW pop up the save window
		FileDialog fd = new FileDialog(frame, 
									   "Save Snapshot as 24-bit PNG...", 
									   FileDialog.SAVE);
		fd.setFile("Untitled.png");
		fd.show();
		if (fd.getFile()!=null)
			try
			{
				File snapShotFile = new File(fd.getDirectory(), ensureFileEndsWith(fd.getFile(),".png"));
				PngEncoder tmpEncoder = new PngEncoder(mCanvas.getFrameAsImage(), false,PngEncoder.FILTER_NONE,9);
				OutputStream stream = new BufferedOutputStream(new FileOutputStream(snapShotFile));
				stream.write(tmpEncoder.pngEncode());
				stream.close();
			}
			catch (FileNotFoundException e) { } // fail
			catch (IOException e) { /* could happen on close? */} // fail
	}

    public void startMovie()
	{
		if (movmaker!=null) return;  // already running
		movmaker = new MovieMaker(frame);
		
		Rectangle2D imgBounds = getImageSize();
		mCanvas.setWritingParams(imgBounds, true);
		BufferedImage typicalImage = mCanvas.getFrameAsImage();
		
		if(!movmaker.start(typicalImage))
		{
			movmaker = null;  // fail
			return;
		}
		movieButton.setIcon(Display2D.MOVIE_ON_ICON); 
		tapeMovie = true; 
		simulation.scheduleAtExtreme(new Steppable()   // to stop movie when simulation is stopped
		{
			public void step(SimState state) { stopMovie(); }
		}, true);
	}
        
    public void stopMovie()
	{
		if (movmaker == null) return;  // already stopped
		movmaker.stop();
		movmaker = null;
		tapeMovie = false;
		mCanvas.stopMovie();
		if (movieButton!=null)  // hasn't been destroyed yet
			movieButton.setIcon(Display2D.MOVIE_OFF_ICON);
	}

	/** Quits the Display3D. Called by finalize().  
	 * Otherwise you should call this method before destroying the Display3D.
	 * 	Right now the only thing this method does is stop and clean up the movie.
	 **/
	public void quit()
	{
		stopMovie();
	}
	/** @see Display2D.finalize() */
	protected void finalize() throws Throwable
	{
		super.finalize();
		quit();
	}
	
	private JPanel optionsPanel = new JPanel();
	private JCheckBox orbitBehCheckBox = new JCheckBox("Orbit");
	private JPanel orbitBehPanel = new JPanel();
	private JCheckBox orbitRotateCheckBox = new JCheckBox("Rotate");
	private JCheckBox orbitTranslateCheckBox = new JCheckBox("Translate");
	private JCheckBox orbitZoomCheckBox = new JCheckBox("Zoom");
	private JCheckBox selectBehCheckBox = new JCheckBox("Select", false);
	private JCheckBox spinBehCheckBox = new JCheckBox("Rotate");
	private JPanel spinBehPanel = new JPanel();
	private JRadioButton polyPoint = new JRadioButton("Vertices", false);
	private JRadioButton polyLine = new JRadioButton("Edges", false);
	private JRadioButton polyFill = new JRadioButton("Fill", true);
	
	private JRadioButton polyCullNone = new JRadioButton("None", false);
	private JRadioButton polyCullFront = new JRadioButton("Front", false);
	private JRadioButton polyCullBack = new JRadioButton("Back", true);
	
	private JFrame optionsFrame;
	
	private NumberTextField rotAxis_X = new NumberTextField(null,0, 1, Math.PI/4)
	{	public double newValue(double newValue)
		{
			mSpinRotator.setTransformAxis(getTransformForAxis(newValue, rotAxis_Y.getValue(), rotAxis_Z.getValue()));
			return newValue;
		}
	};
	private NumberTextField rotAxis_Y = new NumberTextField(null,0, 1, Math.PI/4)
	{	public double newValue(double newValue)
		{
			mSpinRotator.setTransformAxis(getTransformForAxis(rotAxis_X.getValue(),newValue, rotAxis_Z.getValue()));
			return newValue;
		}
	};	
	private NumberTextField rotAxis_Z = new NumberTextField(null,0, 1, Math.PI/4)
	{	public double newValue(double newValue)
		{
			mSpinRotator.setTransformAxis(getTransformForAxis(rotAxis_X.getValue(),rotAxis_Y.getValue(),newValue));
			return newValue;
		}
	};
	private NumberTextField spinDuration = new NumberTextField(null,0, true)
	{	public double newValue(double newValue)
		{
			long value = (long)newValue;
			mSpinRotator.getAlpha().setIncreasingAlphaDuration(value);
			mSpinRotator.getAlpha().setLoopCount(-1); 
//			setAlpha(new Alpha(-1, value));
			return value;
		}
	};

	private void createOptionsPanel()
	{
		
		JPanel behaviorsPanel = new JPanel();
		behaviorsPanel.setBorder(new javax.swing.border.TitledBorder("Behaviors"));
		orbitBehCheckBox.addItemListener(new ItemListener()
		{
			public void itemStateChanged(ItemEvent e)
			{
				boolean b= orbitBehCheckBox.isSelected();
				setOrbitBehaviorEnabled(b);
				orbitRotateCheckBox.setEnabled(b);
				orbitTranslateCheckBox.setEnabled(b);
				orbitZoomCheckBox.setEnabled(b);
			}
		});
		orbitBehPanel.setBorder(new javax.swing.border.TitledBorder(""));
		orbitRotateCheckBox.addItemListener(new ItemListener()
		{
			public void itemStateChanged(ItemEvent e)
			{	mOrbitBehavior.setRotateEnable(orbitRotateCheckBox.isSelected());	}
		});
		orbitTranslateCheckBox.addItemListener(new ItemListener()
		{
			public void itemStateChanged(ItemEvent e)
			{	mOrbitBehavior.setTranslateEnable(orbitTranslateCheckBox.isSelected());	}
		});		
		orbitZoomCheckBox.addItemListener(new ItemListener()
		{
			public void itemStateChanged(ItemEvent e)
			{	mOrbitBehavior.setZoomEnable(orbitZoomCheckBox.isSelected());	}
		});		
		orbitBehPanel.setLayout(new GridLayout(3,1));
		orbitBehPanel.add(orbitRotateCheckBox);
		orbitBehPanel.add(orbitTranslateCheckBox);
		orbitBehPanel.add(orbitZoomCheckBox);
		
		selectBehCheckBox.addItemListener(new ItemListener()
		{
			public void itemStateChanged(ItemEvent e)
			{	setSelectBehaviorEnabled(selectBehCheckBox.isSelected()); }
		});		
		
		rotAxis_X.valField.setColumns(20);
		rotAxis_Y.valField.setColumns(20);
		rotAxis_Z.valField.setColumns(20);
		spinDuration.valField.setColumns(20);

		
		spinBehPanel.setLayout(new GridBagLayout());
		spinBehPanel.setBorder(new javax.swing.border.TitledBorder(""));
		GridBagConstraints gbc = new GridBagConstraints();
		gbc.fill = GridBagConstraints.BOTH;
			
		gbc.gridx=0;gbc.gridy=0;
		spinBehPanel.add(new JLabel("axis.x"), gbc);
		gbc.gridx=1;gbc.gridy=0;
		spinBehPanel.add(rotAxis_X, gbc);
		gbc.gridx=0;gbc.gridy=1;
		spinBehPanel.add(new JLabel("axis.y"), gbc);
		gbc.gridx=1;gbc.gridy=1;
		spinBehPanel.add(rotAxis_Y, gbc);
		gbc.gridx=0;gbc.gridy=2;
		spinBehPanel.add(new JLabel("axis.z"), gbc);
		gbc.gridx=1;gbc.gridy=2;
		spinBehPanel.add(rotAxis_Z, gbc);
		gbc.gridx=0;gbc.gridy=3;
		spinBehPanel.add(new JLabel("duration"), gbc);
		gbc.gridx=1;gbc.gridy=3;
		spinBehPanel.add(spinDuration, gbc);
		
		
		
		
		spinBehCheckBox.addItemListener(new ItemListener()
		{
			public void itemStateChanged(ItemEvent e)
			{
				boolean b = spinBehCheckBox.isSelected();	
				setSpinBehaviorEnabled(b);

				Component[] ca = spinBehPanel.getComponents();
				for(int i=0;i<ca.length; i++)
					ca[i].setEnabled(b);
			}
		});			
		
		behaviorsPanel.setLayout(new GridBagLayout());
		gbc.fill = GridBagConstraints.BOTH;
		gbc.gridx=0;gbc.gridy=0;
		behaviorsPanel.add(orbitBehCheckBox, gbc);
		gbc.gridx=1;gbc.gridy=0;
		behaviorsPanel.add(orbitBehPanel,  gbc);
		gbc.gridx=0;gbc.gridy=1;
		behaviorsPanel.add(spinBehCheckBox,  gbc);
		gbc.gridx=1;gbc.gridy=1;
		behaviorsPanel.add(spinBehPanel, gbc);
		gbc.gridx=0;gbc.gridy=2;
		behaviorsPanel.add(selectBehCheckBox, gbc);
		
		
		
		JPanel polyPanel = new JPanel();
		polyPanel.setBorder(new javax.swing.border.TitledBorder("Polygon Attributes"));
		ButtonGroup polyLineGroup = new ButtonGroup();
		polyLineGroup.add(polyPoint);
		polyLineGroup.add(polyLine);
		polyLineGroup.add(polyFill);
		ButtonGroup polyCullingGroup = new ButtonGroup();
		polyCullingGroup.add(polyCullNone);
		polyCullingGroup.add(polyCullFront);
		polyCullingGroup.add(polyCullBack);
		
		Box polyLinebox = Box.createVerticalBox();
		polyLinebox.add(Box.createGlue());
		polyLinebox.add (new JLabel ("Line Attributes"));
		polyLinebox.add (polyPoint);
		polyLinebox.add (polyLine);
		polyLinebox.add (polyFill);
		polyLinebox.add(Box.createGlue());

		Box polyCullbox = Box.createVerticalBox();
		polyCullbox.add(Box.createGlue());
		polyCullbox.add (new JLabel ("Culling Attributes"));
		polyCullbox.add (polyCullNone);
		polyCullbox.add (polyCullFront);
		polyCullbox.add (polyCullBack);
		polyCullbox.add(Box.createGlue());

		polyPanel.add(polyLinebox);
		polyPanel.add(polyCullbox);
		
		ActionListener polyAttListener = new ActionListener()
		{	public void actionPerformed(ActionEvent e)	{updatePolyAttributes();}};
		
		polyPoint.addActionListener(polyAttListener);		
		polyLine.addActionListener(polyAttListener);		
		polyPoint.addActionListener(polyAttListener);		
		polyFill.addActionListener(polyAttListener);		
		polyCullNone.addActionListener(polyAttListener);		
		polyCullFront.addActionListener(polyAttListener);		
		polyCullBack.addActionListener(polyAttListener);
		
		
		optionsPanel.setLayout(new GridBagLayout());
		gbc.gridx=0;gbc.gridy=0;
		optionsPanel.add(behaviorsPanel, gbc);
		gbc.gridx=0;gbc.gridy=1;
		optionsPanel.add(polyPanel, gbc);
		
		optionsFrame = new JFrame("3D Options");
		optionsFrame.getContentPane().add(optionsPanel);
		optionsFrame.pack();
		optionsFrame.setResizable(false);
		optionsFrame.setDefaultCloseOperation(JFrame.HIDE_ON_CLOSE);
	} 

	private void updatePolyAttributes()
	{
		int lineMode = PolygonAttributes.POLYGON_FILL;
		int cullingMode = PolygonAttributes.CULL_BACK;
		if(polyLine.isSelected())
			lineMode = PolygonAttributes.POLYGON_LINE;
		else
			if(polyPoint.isSelected())
				lineMode = PolygonAttributes.POLYGON_POINT;
				
		if(polyCullFront.isSelected())
			cullingMode = PolygonAttributes.CULL_FRONT;
		else
			if(polyCullNone.isSelected())
				cullingMode = PolygonAttributes.CULL_NONE;
		
		Iterator iter = portrayals.iterator();
		 while(iter.hasNext())
		 {
			 Portrayal3DHolder ph = 	(Portrayal3DHolder)iter.next();
			 Portrayal3D p = ph.portrayal;
			 if(p instanceof ConfigurablePolyPortrayal)
			 {
			 	ConfigurablePolyPortrayal cp = (ConfigurablePolyPortrayal)p;
			 	PolygonAttributes pa = cp.getPolyAttributes();
			 	pa.setCullFace(cullingMode);
			 	pa.setPolygonMode(lineMode);
			 }
		 }
	}
	
	static public Transform3D getTransformForAxis(double dx, double dy, double dz)
	{
//		System.out.println(">>\t"+dx+"\t"+dy+"\t"+dz);
		Transform3D t = new Transform3D();
		Transform3D t1 = new Transform3D();
		t.rotX(Math.atan2(dz,dy));
		t1.rotZ(-Math.atan2(dx,Math.sqrt(dy*dy+dx*dz)));
		t.mul(t1);
		return t;
	} 
	static public void printT3D(Transform3D t)
	{
		float[] d = new float[16];
		t.get(d);
		for(int i=0;i<4;i++)
		{
			for(int j=0;j<4;j++)
				System.out.print(d[i*4+j]+"\t");
			System.out.println();
		}
			
	}
}
    
