package sim.field.grid;

import sim.util.IntBag;

/*
 A concrete implementation of the Grid2D methods; used by several subclasses.
 Note that you should avoid calling these methods from an object of type Grid2D; instead
 try to call them from something more concrete (AbstractGrid2D or SparseGrid2D).
 Otherwise they will not get inlined.  For example,

<pre><tt>
Grid2D foo = ... ;
foo.tx(4);  // will not get inlined

AbstractGrid2D bar = ...;
bar.tx(4);  // WILL get inlined
</tt></pre>

*/

public abstract class AbstractGrid2D implements Grid2D
    {
    // this should never change except via setTo
    protected int width;
    // this should never change except via setTo
    protected int height;

    public final int getWidth() { return width; }
    
    public final int getHeight() { return height; }
    
    public final int tx(final int x) { return (x % width + width) % width; }
    
    public final int ty(final int y) { return (y % height + height) % height; }
    
    public final int stx(final int x) 
        { if (x >= 0) { if (x < width) return x; return x - width; } return x + width; }
        
    public final int sty(final int y) 
        { if (y >= 0) { if (y < height) return y ; return y - height; } return y + height; }
        
    public final int ulx(final int x, final int y) { return x - 1; }

    public final int uly(final int x, final int y) { if ((x & 1) == 0) return y - 1; return y; }

    public final int urx(final int x, final int y) { return x + 1; }

    public final int ury(final int x, final int y) { if ((x & 1) == 0) return y - 1; return y; }
    
    public final int dlx(final int x, final int y) { return x - 1; }

    public final int dly(final int x, final int y) { if ((x & 1) == 0) return y ; return y + 1; }
    
    public final int drx(final int x, final int y) { return x + 1; }

    public final int dry(final int x, final int y) { if ((x & 1) == 0) return y ; return y + 1; }

    public final int upx(final int x, final int y) { return x; }

    public final int upy(final int x, final int y) { return y - 1; }

    public final int downx(final int x, final int y) { return x; }

    public final int downy(final int x, final int y) { return y + 1; }

    /*
     * Gets all neighbors of a location that satisfy max( abs(x-X) , abs(y-Y) ) <= d
     * Returns the x and y positions of the neighbors (including original location).
     */
    public final void getNeighborsMaxDistance( final int x, final int y, final int dist, final boolean toroidal, IntBag xPos, IntBag yPos )
    {
        // won't work for negative distances
        if( dist < 0 )
        {
            throw new RuntimeException( "Runtime exception in method getNeighborsMaxDistance: Distance must be positive" );
        }

        if( xPos == null || yPos == null )
        {
            throw new RuntimeException( "Runtime exception in method getNeighborsMaxDistance: xPos and yPos should not be null" );
        }

        xPos.clear();
        yPos.clear();

        // for toroidal environments the code will be different because of wrapping arround
        if( toroidal )
        {
            // compute xmin and xmax for the neighborhood
            for( int x0 = x-dist ; x0 <= x+dist ; x0++ )
            {
                final int x_0 = stx(x0);
                // compute ymin and ymax for the neighborhood
                for( int y0 = y-dist ; y0 <= y+dist ; y0++ )
                {
                    final int y_0 = sty(y0);
                    xPos.add( x_0 );
                    yPos.add( y_0 );
                }
            }
        }
        else // not toroidal
        {
            // compute xmin and xmax for the neighborhood such that they are within boundaries
            for( int x0 = ((x-dist>=0)?x-dist:0) ; x0 <= ((x+dist<=width-1)?x+dist:width-1) ; x0++ )
            {
                final int x_0 = x0;
                // compute ymin and ymax for the neighborhood such that they are within boundaries
                for( int y0 = ((y-dist>=0)?y-dist:0) ; y0 <= ((y+dist<=height-1)?y+dist:height-1) ; y0++ )
                {
                    final int y_0 = y0;
                    xPos.add( x_0 );
                    yPos.add( y_0 );
                }
            }
        }
    }


    /*
     * Gets all neighbors of a location that satisfy abs(x-X) + abs(y-Y) <= d
     * Returns the x and y positions of the neighbors (including original location).
     */
    public final void getNeighborsHamiltonianDistance( final int x, final int y, final int dist, final boolean toroidal, IntBag xPos, IntBag yPos )
    {
        // won't work for negative distances
        if( dist < 0 )
        {
            throw new RuntimeException( "Runtime exception in method getNeighborsHamiltonianDistance: Distance must be positive" );
        }

        if( xPos == null || yPos == null )
        {
            throw new RuntimeException( "Runtime exception in method getNeighborsHamiltonianDistance: xPos and yPos should not be null" );
        }

        xPos.clear();
        yPos.clear();

        // for toroidal environments the code will be different because of wrapping arround
        if( toroidal )
        {
            // compute xmin and xmax for the neighborhood
            for( int x0 = x-dist ; x0 <= x+dist ; x0++ )
            {
                final int x_0 = stx(x0);
                // compute ymin and ymax for the neighborhood; they depend on the curreny x0 value
                for( int y0 = y-(dist-((x0-x>=0)?x0-x:x-x0)) ; y0 <= y+(dist-((x0-x>=0)?x0-x:x-x0)) ; y0++ )
                {
                    final int y_0 = sty(y0);
                    xPos.add( x_0 );
                    yPos.add( y_0 );
                }
            }
        }
        else // not toroidal
        {
            // compute xmin and xmax for the neighborhood such that they are within boundaries
            for( int x0 = ((x-dist>=0)?x-dist:0) ; x0 <= ((x+dist<=width-1)?x+dist:width-1) ; x0++ )
            {
                final int x_0 = x0;
                // compute ymin and ymax for the neighborhood such that they are within boundaries
                // they depend on the curreny x0 value
                for( int y0 = ((y-(dist-((x0-x>=0)?x0-x:x-x0))>=0)?y-(dist-((x0-x>=0)?x0-x:x-x0)):0) ;
                     y0 <= ((y+(dist-((x0-x>=0)?x0-x:x-x0))<=height-1)?y+(dist-((x0-x>=0)?x0-x:x-x0)):height-1) ;
                     y0++ )
                {
                    final int y_0 = y0;
                    xPos.add( x_0 );
                    yPos.add( y_0 );
                }
            }
        }
    }

    /*
     * Gets all neighbors of a location in hexagonal world
     * Returns the x and y positions of the neighbors (including original location).
     */
    public final void getNeighborsHexagonalDistance( final int x, final int y, final int dist, final boolean toroidal, IntBag xPos, IntBag yPos )
    {
        // won't work for negative distances
        if( dist < 0 )
        {
            throw new RuntimeException( "Runtime exception in method getNeighborsHexagonalDistance: Distance must be positive" );
        }

        if( xPos == null || yPos == null )
        {
            throw new RuntimeException( "Runtime exception in method getNeighborsHamiltonianDistance: xPos and yPos should not be null" );
        }

        xPos.clear();
        yPos.clear();

        if( toroidal && height%2==1 )
            throw new RuntimeException( "Runtime exception in getNeighborsHexagonalDistance: toroidal hexagonal environment should have even heights" );

        if( toroidal )
        {
            for( int y0 = y-dist ; y0 <= y+dist ; y0 = downy(x,y0) )
            {
                xPos.add( stx(x) );
                yPos.add( sty(y0) );
            }
            int x0, ymin, ymax;

            x0 = x;
            ymin = y-dist;
            ymax = y+dist;
            for( int i = 1 ; i <= dist ; i++ )
            {
                final int temp_ymin = ymin;
                ymin = dly( x0, ymin );
                ymax = uly( x0, ymax );
                x0 = dlx( x0, temp_ymin );
                for( int y0 = ymin ; y0 <= ymax ; y0 = downy(x0,y0) )
                {
                    xPos.add( stx(x0) );
                    yPos.add( sty(y0) );
                }
            }
            x0 = x;
            ymin = y-dist;
            ymax = y+dist;
            for( int i = 1 ; i <= dist ; i++ )
            {
                final int temp_ymin = ymin;
                ymin = dry( x0, ymin );
                ymax = ury( x0, ymax );
                x0 = drx( x0, temp_ymin );
                for( int y0 = ymin ; y0 <= ymax ; y0 = downy(x0,y0) )
                {
                    xPos.add( stx(x0) );
                    yPos.add( sty(y0) );
                }
            }
        }
        else // not toroidal
        {
            if( x < 0 || x >= width || y < 0 || y >= height )
                throw new RuntimeException( "Runtime exception in method getNeighborsHexagonalDistance: invalid initial position" );

            for( int y0 = y-dist ; y0 <= ((y+dist<height)?y+dist:height-1) ; y0 = downy(x,y0) )
            {
                xPos.add( x );
                yPos.add( y0 );

            }
            int x0, ymin, ymax;

            x0 = x;
            ymin = y-dist;
            ymax = y+dist;
            for( int i = 1 ; i <= dist ; i++ )
            {
                final int temp_ymin = ymin;
                ymin = dly( x0, ymin );
                ymax = uly( x0, ymax );
                x0 = dlx( x0, temp_ymin );
                if( x0 >= 0 )
                    for( int y0 = ymin ; y0 <= ((ymax<height)?ymax:height-1) ; y0 = downy(x0,y0) )
                    {
                        if( y0 >= 0 )
                        {
                            xPos.add( x0 );
                            yPos.add( y0 );
                        }
                    }
            }

            x0 = x;
            ymin = y-dist;
            ymax = y+dist;
            for( int i = 1 ; i <= dist ; i++ )
            {
                final int temp_ymin = ymin;
                ymin = dry( x0, ymin );
                ymax = ury( x0, ymax );
                x0 = drx( x0, temp_ymin );
                if( x0 < width )
                    for( int y0 = ymin ; y0 <= ((ymax<height)?ymax:height) ; y0 = downy(x0,y0) )
                    {
                        if( y0 >= 0 )
                        {
                            xPos.add( x0 );
                            yPos.add( y0 );
                        }
                    }
            }
        }
    }

    }
