package sim.field.grid;

import sim.util.IntBag;

/*
 A concrete implementation of the Grid3D methods; used by several subclasses.
 Note that you should avoid calling these methods from an object of type Grid3D; instead
 try to call them from something more concrete (AbstractGrid3D or SparseGrid3D).  
 Otherwise they will not get inlined.  For example,

<pre><tt>
Grid3D foo = ... ;
foo.tx(4);  // will not get inlined

AbstractGrid3D bar = ...;
bar.tx(4);  // WILL get inlined
</tt></pre>

*/

public abstract class AbstractGrid3D implements Grid3D
    {
    // this should never change except via setTo
    protected int width;
    // this should never change except via setTo
    protected int height;
    // this should never change except via setTo
    protected int length;

    public final int getWidth() { return width; }
    
    public final int getHeight() { return height; }
    
    public final int getLength() { return length; }
    
    public final int tx(final int x) { return (x % width + width) % width; }

    public final int ty(final int y) { return (y % height + height) % height; }
    
    public final int tz(final int z) { return (z % length + length) % length; }

    public final int stx(final int x) 
        { if (x >= 0) { if (x < width) return x; return x - width; } return x + width; }
    
    public final int sty(final int y) 
        { if (y >= 0) { if (y < height) return y ; return y - height; } return y + height; }

    public final int stz(final int z) 
        { if (z >= 0) { if (z < length) return z ; return z - length; } return z + length; }

    /*
     * Gets all neighbors of a location that satisfy max( abs(x-X) , abs(y-Y), abs(z-Z) ) <= d
     * Returns the x, y and z positions of the neighbors.
     */
    public final void getNeighborsMaxDistance( final int x, final int y, final int z, final int dist, final boolean toroidal, IntBag xPos, IntBag yPos, IntBag zPos )
    {
        // won't work for negative distances
        if( dist < 0 )
        {
            throw new RuntimeException( "Runtime exception in method getNeighborsMaxDistance: Distance must be positive" );
        }

        if( xPos == null || yPos == null || zPos == null )
        {
            throw new RuntimeException( "Runtime exception in method getNeighborsMaxDistance: xPos and yPos should not be null" );
        }

        xPos.clear();
        yPos.clear();
        zPos.clear();

        // for toroidal environments the code will be different because of wrapping arround
        if( toroidal )
        {
            // compute xmin and xmax for the neighborhood
            for( int x0 = x-dist ; x0 <= x+dist ; x0++ )
            {
                final int x_0 = stx(x0);
                // compute ymin and ymax for the neighborhood
                for( int y0 = y-dist ; y0 <= y+dist ; y0++ )
                {
                    final int y_0 = sty(y0);
                    for( int z0 = z-dist ; z0 <= z+dist ; z0++ )
                    {
                        final int z_0 = stz(z0);
                        if( x_0 != x || y_0 != y || z_0 != z )
                        {
                            xPos.add( x_0 );
                            yPos.add( y_0 );
                            zPos.add( z_0 );
                        }
                    }
                }
            }
        }
        else // not toroidal
        {
            // compute xmin and xmax for the neighborhood such that they are within boundaries
            for( int x0 = ((x-dist>=0)?x-dist:0) ; x0 <= ((x+dist<=width-1)?x+dist:width-1) ; x0++ )
            {
                final int x_0 = x0;
                // compute ymin and ymax for the neighborhood such that they are within boundaries
                for( int y0 = ((y-dist>=0)?y-dist:0) ; y0 <= ((y+dist<=height-1)?y+dist:height-1) ; y0++ )
                {
                    final int y_0 = y0;
                    for( int z0 = ((z-dist>=0)?z-dist:0) ; z0 <= ((z+dist<=length-1)?z+dist:length-1) ; z0++ )
                    {
                        final int z_0 = z0;
                        if( x_0 != x || y_0 != y || z_0 != z )
                        {
                            xPos.add( x_0 );
                            yPos.add( y_0 );
                            zPos.add( z_0 );
                        }
                    }
                }
            }
        }
    }


    /*
     * Gets all neighbors of a location that satisfy abs(x-X) + abs(y-Y) + abs(z-Z) <= d
     * Returns the x, y and z positions of the neighbors.
     */
    public final void getNeighborsHamiltonianDistance( final int x, final int y, final int z, final int dist, final boolean toroidal, IntBag xPos, IntBag yPos, IntBag zPos )
    {
        // won't work for negative distances
        if( dist < 0 )
        {
            throw new RuntimeException( "Runtime exception in method getNeighborsHamiltonianDistance: Distance must be positive" );
        }

        if( xPos == null || yPos == null || zPos == null )
        {
            throw new RuntimeException( "Runtime exception in method getNeighborsHamiltonianDistance: xPos and yPos should not be null" );
        }

        xPos.clear();
        yPos.clear();
        zPos.clear();

        // for toroidal environments the code will be different because of wrapping arround
        if( toroidal )
        {
            // compute xmin and xmax for the neighborhood
            for( int x0 = x-dist ; x0 <= x+dist ; x0++ )
            {
                final int x_0 = stx(x0);
                // compute ymin and ymax for the neighborhood; they depend on the curreny x0 value
                for( int y0 = y-(dist-((x0-x>=0)?x0-x:x-x0)) ; y0 <= y+(dist-((x0-x>=0)?x0-x:x-x0)) ; y0++ )
                {
                    final int y_0 = sty(y0);
                    for( int z0 = z-(dist-((x0-x>=0)?x0-x:x-x0)-((y0-y>=0)?y0-y:y-y0)) ;
                         z0 <= z+(dist-((x0-x>=0)?x0-x:x-x0)-((y0-y>=0)?y0-y:y-y0)) ;
                         z0++ )
                    {
                        final int z_0 = stz(z0);
                        if( x_0 != x || y_0 != y || z_0 != z )
                        {
                            xPos.add( x_0 );
                            yPos.add( y_0 );
                            zPos.add( z_0 );
                        }
                    }
                }
            }
        }
        else // not toroidal
        {
            // compute xmin and xmax for the neighborhood such that they are within boundaries
            for( int x0 = ((x-dist>=0)?x-dist:0) ; x0 <= ((x+dist<=width-1)?x+dist:width-1) ; x0++ )
            {
                final int x_0 = x0;
                // compute ymin and ymax for the neighborhood such that they are within boundaries
                // they depend on the curreny x0 value
                for( int y0 = ((y-(dist-((x0-x>=0)?x0-x:x-x0))>=0)?y-(dist-((x0-x>=0)?x0-x:x-x0)):0) ;
                     y0 <= ((y+(dist-((x0-x>=0)?x0-x:x-x0))<=height-1)?y+(dist-((x0-x>=0)?x0-x:x-x0)):height-1) ;
                     y0++ )
                {
                    final int y_0 = y0;
                    for( int z0 = ((z-(dist-((x0-x>=0)?x0-x:x-x0)-((y0-y>=0)?y0-y:y-y0))>=0)?z-(dist-((x0-x>=0)?x0-x:x-x0)-((y0-y>=0)?y0-y:y-y0)):0) ;
                         z0 <= ((z+(dist-((x0-x>=0)?x0-x:x-x0)-((y0-y>=0)?y0-y:y-y0))<=length-1)?z+(dist-((x0-x>=0)?x0-x:x-x0)-((y0-y>=0)?y0-y:y-y0)):length-1) ;
                         z0++ )
                    {
                        final int z_0 = z0;
                        if( x_0 != x || y_0 != y || z_0 != z )
                        {
                            xPos.add( x_0 );
                            yPos.add( y_0 );
                            zPos.add( z_0 );
                        }
                    }
                }
            }
        }
    }

    }

