package sim.field.grid;

import sim.util.*;

/**
    A wrapper for 3D arrays of doubles.

    <p>This object expects that the 3D arrays are rectangular.  You are encouraged to access the array
    directly, but fast-inlined methods are provided for you if you want something safer.  The object
    implements all of the Grid3D interface.  See Grid3D for rules on how to properly implement toroidal
    grids.
    
    <p>The width and height and length (z dimension) of the object are provided to avoid having to say field[x].length, etc.  
*/

public /*strictfp*/ class DoubleGrid3D extends AbstractGrid3D
    {
    public double[/**x*/][/**y*/][/**z*/] field;
    
    public DoubleGrid3D (int xdim, int ydim, int zdim)
        {
        super();
        width = xdim;
        height = ydim;
        length = zdim;
        field = new double[xdim][ydim][zdim];
        }
    
    public DoubleGrid3D (int xdim, int ydim, int zdim, double initialValue)
        {
        this(xdim,ydim, zdim);
        setTo(initialValue);
        }
            
    public DoubleGrid3D (DoubleGrid3D values)
        {
        super();
        setTo(values);
        }

    // actually quite fast because it gets inlined
    public final double set(final int x, final int y, final int z, final double val)
        {
        double returnval = field[x][y][z];
        field[x][y][z] = val;
        return returnval;
        }
    
    // actually quite fast because it gets inlined
    public final double get(final int x, final int y, final int z)
        {
        return field[x][y][z];
        }


    public final double max()
        {
        double max = Double.NEGATIVE_INFINITY;
       	double[][] fieldx = null;
       	double[] fieldxy = null;
		final int width = this.width;
		final int height = this.height;
		final int length = this.length;      	
        for(int x=0;x<width;x++)
        {
        	fieldx = field[x];
            for(int y=0;y<height;y++)
            {
            	fieldxy= fieldx[y];
                for(int z=0;z<length;z++)
	                if (max < fieldxy[z]) max = fieldxy[z];
            }
        }
        return max;
        }

    public final double min()
        {
        double min = Double.POSITIVE_INFINITY;
       	double[][] fieldx = null;
       	double[] fieldxy = null;
		final int width = this.width;
		final int height = this.height;
		final int length = this.length;      	
        for(int x=0;x<width;x++)
        {
        	fieldx = field[x];
            for(int y=0;y<height;y++)
            {
            	fieldxy = fieldx[y];
                for(int z=0;z<length;z++)
                if (min > fieldxy[z]) min = fieldxy[z];
            }
        }
        return min;
        }
        
    public final double mean()
        {
        long count = 0;
        double mean = 0;
       	double[][] fieldx = null;
       	double[] fieldxy = null;        
		final int width = this.width;
		final int height = this.height;
		final int length = this.length;      	
        for(int x=0;x<width;x++)
        {
        	fieldx = field[x];
            for(int y=0;y<height;y++)
            {
            	fieldxy = fieldx[y];
                for(int z=0;z<length;z++)
                { mean += fieldxy[z]; count++; }
            }
        }
        return (count == 0 ? 0 : mean / count);
        }
        
    public final DoubleGrid3D setTo(double thisMuch)
        {
       	double[][] fieldx = null;
       	double[] fieldxy = null;        
		final int width = this.width;
		final int height = this.height;
		final int length = this.length;      	
        for(int x=0;x<width;x++)
        {
        	fieldx = field[x];
            for(int y=0;y<height;y++)
            {
            	fieldxy = fieldx[y];
                for(int z=0;z<length;z++)
                fieldxy[z]=thisMuch;
            }
        }
        return this;
        }

    public final DoubleGrid3D setTo(DoubleGrid3D values)
        {
        if (width != values.width || height != values.height || length != values.length )
            {
            final int width = this.width = values.width;
            final int height = this.height = values.height;
            final int length = this.length = values.length;
            field = new double[width][height][];
	       	double[][] fieldx = null;
            for(int x = 0 ; x < width; x++)
            {
            	fieldx = field[x];
                for( int y = 0 ; y < height ; y++ )
                    fieldx[y] = (double []) (values.field[x][y].clone());
            }
            }
        else
            {
            for(int x =0 ; x < width; x++)
                for( int y = 0 ; y < height ; y++ )
                    System.arraycopy(values.field[x][y],0,field[x][y],0,length);
            }

        return this;
        }

    public final DoubleGrid3D upperBound(double toNoMoreThanThisMuch)
        {
       	double[][] fieldx = null;
       	double[] fieldxy = null;     
		final int width = this.width;
		final int height = this.height;
		final int length = this.length;      	
       	   
        for(int x=0;x<width;x++)
        {
        	fieldx = field[x];
            for(int y=0;y<height;y++)
            {
            	fieldxy = fieldx[y];
                for(int z=0;z<length;z++)
                if (fieldxy[z] > toNoMoreThanThisMuch)
                    fieldxy[z] = toNoMoreThanThisMuch;
            }
        }
        return this;
        }

    public final DoubleGrid3D lowerBound(double toNoLowerThanThisMuch)
        {
       	double[][] fieldx = null;
       	double[] fieldxy = null;                	
		final int width = this.width;
		final int height = this.height;
		final int length = this.length;      	
        for(int x=0;x<width;x++)
        {
        	fieldx = field[x];
            for(int y=0;y<height;y++)
            {
            	fieldxy = fieldx[y];
                for(int z=0;z<length;z++)
                if (field[x][y][z] < toNoLowerThanThisMuch)
                    field[x][y][z] = toNoLowerThanThisMuch;
            }
        }
        return this;
        }

    public final DoubleGrid3D add(double withThisMuch)
        {
        if (withThisMuch==0.0) return this;
       	double[][] fieldx = null;
       	double[] fieldxy = null;        
		final int width = this.width;
		final int height = this.height;
		final int length = this.length;      	

        for(int x=0;x<width;x++)
        {
        	fieldx = field[x];
            for(int y=0;y<height;y++)
            {
            	fieldxy = fieldx[y];
                for(int z=0;z<length;z++)
	                fieldxy[z]+=withThisMuch;
            }
        }
        return this;
        }
        
    public final DoubleGrid3D add(IntGrid3D withThis)
        {
        int[][][] otherField = withThis.field;
       	int[][] ofieldx = null;
       	int[] ofieldxy = null;        
       	double[][] fieldx = null;
       	double[] fieldxy = null;        
		final int width = this.width;
		final int height = this.height;
		final int length = this.length;      	
        
        for(int x=0;x<width;x++)
        {
        	fieldx = field[x];
        	ofieldx = otherField[x];
            for(int y=0;y<height;y++)
        	{
        		ofieldxy = ofieldx[y];
        		fieldxy = fieldx[y];
                for(int z=0;z<length;z++)
	                fieldxy[z]+=ofieldxy[z];
        	}
        }
        return this;
        }

    public final DoubleGrid3D add(DoubleGrid3D withThis)
        {
        double[][][] otherField = withThis.field;
       	double[][] ofieldx = null;
       	double[] ofieldxy = null;        
       	double[][] fieldx = null;
       	double[] fieldxy = null;        
		final int width = this.width;
		final int height = this.height;
		final int length = this.length;      	
        for(int x=0;x<width;x++)
        {
        	fieldx = field[x];
        	ofieldx = otherField[x];
            for(int y=0;y<height;y++)
            {
            	fieldxy = fieldx[y];
            	ofieldxy = ofieldx[y];
                for(int z=0;z<length;z++)
             	   fieldxy[z]+=ofieldxy[z];
            }
        }
        return this;
        }

    public final DoubleGrid3D multiply(double byThisMuch)
        {
        if (byThisMuch==1.0) return this;
       	double[][] fieldx = null;
       	double[] fieldxy = null;                	
		final int width = this.width;
		final int height = this.height;
		final int length = this.length;      	
        for(int x=0;x<width;x++)
        {
        	fieldx = field[x];
            for(int y=0;y<height;y++)
            {
            	
            	fieldxy = fieldx[y];
                for(int z=0;z<length;z++)
	                field[x][y][z]*=byThisMuch;
            }
        }
        return this;
        }
    
    public final DoubleGrid3D multiply(IntGrid3D withThis)
        {
        int[][][] otherField = withThis.field;
       	int[][] ofieldx = null;
       	int[] ofieldxy = null;        
       	double[][] fieldx = null;
       	double[] fieldxy = null;        
		final int width = this.width;
		final int height = this.height;
		final int length = this.length;      	
        
        for(int x=0;x<width;x++)
        {
        	fieldx = field[x];
        	ofieldx = otherField[x];
            for(int y=0;y<height;y++)
        	{
        		ofieldxy = ofieldx[y];
        		fieldxy = fieldx[y];
                for(int z=0;z<length;z++)
	                fieldxy[z]*=ofieldxy[z];
        	}
        }
        return this;
        }

    public final DoubleGrid3D multiply(DoubleGrid3D withThis)
        {
        double[][][] otherField = withThis.field;
       	double[][] ofieldx = null;
       	double[] ofieldxy = null;        
       	double[][] fieldx = null;
       	double[] fieldxy = null;        
		final int width = this.width;
		final int height = this.height;
		final int length = this.length;      	
        for(int x=0;x<width;x++)
        {
        	fieldx = field[x];
        	ofieldx = otherField[x];
            for(int y=0;y<height;y++)
            {
            	fieldxy = fieldx[y];
            	ofieldxy = ofieldx[y];
                for(int z=0;z<length;z++)
                fieldxy[z]*=ofieldxy[z];
            }
        }
        return this;
        }

    public final DoubleGrid3D floor()
        {
      	double[][] fieldx = null;
       	double[] fieldxy = null;                	
		final int width = this.width;
		final int height = this.height;
		final int length = this.length;      	
        for(int x=0;x<width;x++)
        {
        	fieldx = field[x];
            for(int y=0;y<height;y++)
            {        	
            	fieldxy = fieldx[y];
                for(int z=0;z<length;z++)
	                fieldxy[z] = /*Strict*/Math.floor(fieldxy[z]);
            }
        }
        return this;
        }

    public final DoubleGrid3D ceiling()
        {
      	double[][] fieldx = null;
       	double[] fieldxy = null;                	
		final int width = this.width;
		final int height = this.height;
		final int length = this.length;      	
        for(int x=0;x<width;x++)
        {
        	fieldx = field[x];
            for(int y=0;y<height;y++)
            {
            	fieldxy = fieldx[y];
                for(int z=0;z<length;z++)
                	fieldxy[z] = /*Strict*/Math.ceil(fieldxy[z]);
            }
        }
        return this;
        }
    
    /** round towards zero */
    public final DoubleGrid3D  truncate()
        {
      	double[][] fieldx = null;
       	double[] fieldxy = null;                	
		final int width = this.width;
		final int height = this.height;
		final int length = this.length;      	
        for(int x=0;x<width;x++)
        {
        	fieldx = field[x];
            for(int y=0;y<height;y++)
            {
            	
            	fieldxy = fieldx[y];
                for(int z=0;z<length;z++)
                	fieldxy[z] = (long) fieldxy[z];  
        // is this right?
        //        if (field[x][y][z] > 0.0) 
        //            /*Strict*/Math.floor(field[x][y][z]);
        //        else
        //            /*Strict*/Math.ceil(field[x][y][z]);
            }
        }
        return this;
        }

    public final DoubleGrid3D  rint()
        {
      	double[][] fieldx = null;
       	double[] fieldxy = null;                	
		final int width = this.width;
		final int height = this.height;
		final int length = this.length;      	
        for(int x=0;x<width;x++)
        {
        	fieldx = field[x];
            for(int y=0;y<height;y++)
            {
            	
            	fieldxy = fieldx[y];
                for(int z=0;z<length;z++)
                	fieldxy[z] = /*Strict*/Math.rint(fieldxy[z]);
            }
        }
        return this;
        }
    
    /*
     * Gets all neighbors of a location that satisfy max( abs(x-X) , abs(y-Y), abs(z-Z) ) <= d
     * Returns the neighbors and their x, y and z positions; xPos, yPos and zPos can be null if you don't care about the x, y and z positions.
     */
    public final void getNeighborsMaxDistance( final int x, final int y, final int z, final int dist, final boolean toroidal, DoubleBag result, IntBag xPos, IntBag yPos, IntBag zPos )
    {
        if( xPos == null )
            xPos = new IntBag();
        if( yPos == null )
            yPos = new IntBag();
        if( zPos == null )
            zPos = new IntBag();

        getNeighborsMaxDistance( x, y, z, dist, toroidal, xPos, yPos, zPos );

        result.clear();
        for( int i = 0 ; i < xPos.numObjs ; i++ )
            result.add( field[xPos.objs[i]][yPos.objs[i]][zPos.objs[i]] );
    }

    /*
     * Gets all neighbors of a location that satisfy abs(x-X) + abs(y-Y) + abs(z-Z) <= d
     * Returns the neighbors and their x, y and z positions; xPos, yPos and zPos can be null if you don't care about the x, y and z positions.
     */
    public final void getNeighborsHamiltonianDistance( final int x, final int y, final int z, final int dist, final boolean toroidal, DoubleBag result, IntBag xPos, IntBag yPos, IntBag zPos )
    {
        if( xPos == null )
            xPos = new IntBag();
        if( yPos == null )
            yPos = new IntBag();
        if( zPos == null )
            zPos = new IntBag();

        getNeighborsHamiltonianDistance( x, y, z, dist, toroidal, xPos, yPos, zPos );

        result.clear();
        for( int i = 0 ; i < xPos.numObjs ; i++ )
            result.add( field[xPos.objs[i]][yPos.objs[i]][zPos.objs[i]] );
    }

    }
