package sim.field.grid;
import sim.util.*;

/**
    A wrapper for 3D arrays of ints.

    <p>This object expects that the 3D arrays are rectangular.  You are encouraged to access the array
    directly, but fast-inlined methods are provided for you if you want something safer.  The object
    implements all of the Grid3D interface.  See Grid3D for rules on how to properly implement toroidal
    grids.
    
    <p>The width and height and length (z dimension) of the object are provided to avoid having to say field[x].length, etc.  
*/

public /*strictfp*/ class IntGrid3D extends AbstractGrid3D
    {
    public int[/**x*/][/**y*/][/**z*/] field;
    
    public IntGrid3D (int xdim, int ydim, int zdim)
        {
        super();
        width = xdim;
        height = ydim;
        length = zdim;
        field = new int[xdim][ydim][zdim];
        }
    
    public IntGrid3D (int xdim, int ydim, int zdim, int initialValue)
        {
        this(xdim,ydim, zdim);
        setTo(initialValue);
        }
    
    public IntGrid3D (IntGrid3D values)
        {
        super();
        setTo(values);
        }

    // actually quite fast because it gets inlined
    public final int set(final int x, final int y, final int z, final int val)
        {
        int returnval = field[x][y][z];
        field[x][y][z] = val;
        return returnval;
        }
    
    // actually quite fast because it gets inlined
    public final int get(final int x, final int y, final int z)
        {
        return field[x][y][z];
        }

    public final int max()
        {
        int max = Integer.MIN_VALUE;
        int[][]fieldx = null;
        int[] fieldxy = null;
		final int width = this.width;
		final int height = this.height;
		final int length = this.length;      	
        
        for(int x=0;x<width;x++)
        {
        	fieldx = field[x];
            for(int y=0;y<height;y++)
            {
            	fieldxy = fieldx[y];
                for(int z=0;z<length;z++)
                if (max < fieldxy[z]) max = fieldxy[z];
            }
        }
        return max;
        }

    public final int min()
        {
        int min = Integer.MAX_VALUE;
        int[][]fieldx = null;
        int[] fieldxy = null;
		final int width = this.width;
		final int height = this.height;
		final int length = this.length;      	
        for(int x=0;x<width;x++)
        {
        	fieldx = field[x];
            for(int y=0;y<height;y++)
            {
            	fieldxy = fieldx[y];
                for(int z=0;z<length;z++)
                if (min > fieldxy[z]) min = fieldxy[z];
            }
        }
        return min;
        }
        
    public final double mean()
        {
        long count = 0;
        double mean = 0;
        int[][]fieldx = null;
        int[] fieldxy = null;
		final int width = this.width;
		final int height = this.height;
		final int length = this.length;      	
        for(int x=0;x<width;x++)
        {
        	fieldx = field[x];
            for(int y=0;y<height;y++)
            {
            	fieldxy = fieldx[y];
                for(int z=0;z<length;z++)
                { mean += fieldxy[z]; count++; }
            }
        }
        return (count == 0 ? 0 : mean / count);
        }
        
    public final IntGrid3D setTo(int thisMuch)
        {
        int[][]fieldx = null;
        int[] fieldxy = null;
   		final int width = this.width;
		final int height = this.height;
		final int length = this.length;
        for(int x=0;x<width;x++)
        {
        	fieldx = field[x];
            for(int y=0;y<height;y++)
            {
            	fieldxy = fieldx[y];
                for(int z=0;z<length;z++)
	                fieldxy[z]=thisMuch;
            }
        }
        return this;
        }

    public final IntGrid3D setTo(IntGrid3D values)
        {
        if (width != values.width || height != values.height || length != values.length )
            {
			final int width = this.width = values.width;
            final int height = this.height = values.height;
            final int length = this.length = values.length;
            field = new int[width][height][];
	        int[][]fieldx = null;        
	        int[][]ofieldx = null;        
            for(int x = 0 ; x < width; x++)
	            {
	            fieldx = field[x];
	            ofieldx = values.field[x];
                for( int y = 0 ; y < height ; y++ )
                    fieldx[y] = (int []) (ofieldx[y].clone());
    	        }
            }
        else
            {
	        int[][]fieldx = null;        
	        int[][]ofieldx = null;
            for(int x =0 ; x < width; x++)
	            {
	            fieldx = field[x];
	            ofieldx = values.field[x];	
                for( int y = 0 ; y < height ; y++ )
                    System.arraycopy(ofieldx[y],0,fieldx[y],0,length);
    	        }
            }

        return this;
        }

    public final IntGrid3D upperBound(int toNoMoreThanThisMuch)
        {
        int[][]fieldx = null;
        int[] fieldxy = null;
   		final int width = this.width;
		final int height = this.height;
		final int length = this.length;      	
        for(int x=0;x<width;x++)
        {
        	fieldx = field[x];
            for(int y=0;y<height;y++)
            {
            	fieldxy = fieldx[y];
                for(int z=0;z<length;z++)
	                if (fieldxy[z] > toNoMoreThanThisMuch)
    	                fieldxy[z] = toNoMoreThanThisMuch;
            }
        }
        return this;
        }

    public final IntGrid3D lowerBound(int toNoLowerThanThisMuch)
        {
        int[][]fieldx = null;
        int[] fieldxy = null;
   		final int width = this.width;
		final int height = this.height;
		final int length = this.length;      	
        for(int x=0;x<width;x++)
        {
        	fieldx = field[x];
            for(int y=0;y<height;y++)
            {
            	fieldxy = fieldx[y];
                for(int z=0;z<length;z++)
	                if (fieldxy[z] < toNoLowerThanThisMuch)
    	                fieldxy[z] = toNoLowerThanThisMuch;
            }
        }
        return this;
        }

    public final IntGrid3D add(int withThisMuch)
        {
        if (withThisMuch==0.0) return this;
        int[][]fieldx = null;
        int[] fieldxy = null;
   		final int width = this.width;
		final int height = this.height;
		final int length = this.length;      	
        for(int x=0;x<width;x++)
        {
        	fieldx = field[x];
            for(int y=0;y<height;y++)
            {
            	fieldxy = fieldx[y];
                for(int z=0;z<length;z++)
                fieldxy[z]+=withThisMuch;
            }
        }
        return this;
        }
        
    public final IntGrid3D add(IntGrid3D withThis)
        {
        int[][]fieldx = null;
        int[] fieldxy = null;
		int[][][] ofield = withThis.field;
        int[][]ofieldx = null;
        int[] ofieldxy = null;
   		final int width = this.width;
		final int height = this.height;
		final int length = this.length;      	
		        	
        for(int x=0;x<width;x++)
        {
        	fieldx = field[x];
        	ofieldx = ofield[x];
            for(int y=0;y<height;y++)
            {
	        	fieldxy = fieldx[y];
	        	ofieldxy = ofieldx[y];
                for(int z=0;z<length;z++)
             	   fieldxy[z]+=ofieldxy[z];
            }
        }
        return this;
        }

    public final IntGrid3D multiply(int byThisMuch)
        {
        if (byThisMuch==1.0) return this;
        int[][]fieldx = null;
        int[] fieldxy = null;
   		final int width = this.width;
		final int height = this.height;
		final int length = this.length;      	
        for(int x=0;x<width;x++)
        {
        	fieldx = field[x];
            for(int y=0;y<height;y++)
            {
            	fieldxy = fieldx[y];
                for(int z=0;z<length;z++)
	                fieldxy[z]*=byThisMuch;
            }       
		}
        return this;
        }
    
    public final IntGrid3D multiply(IntGrid3D withThis)
        {
        int[][]fieldx = null;
        int[] fieldxy = null;
		int[][][] ofield = withThis.field;
        int[][]ofieldx = null;
        int[] ofieldxy = null;
   		final int width = this.width;
		final int height = this.height;
		final int length = this.length;
        for(int x=0;x<width;x++)
        {
        	fieldx = field[x];
        	ofieldx = ofield[x];
            for(int y=0;y<height;y++)
            {
	        	fieldxy = fieldx[y];
	        	ofieldxy = ofieldx[y];
                for(int z=0;z<length;z++)
	                fieldxy[z]*=ofieldxy[z];
            }
        }
        return this;
        }

    /*
     * Gets all neighbors of a location that satisfy max( abs(x-X) , abs(y-Y), abs(z-Z) ) <= d
     * Returns the neighbors and their x, y and z positions; xPos, yPos and zPos can be null if you don't care about the x, y and z positions.
     */
    public final void getNeighborsMaxDistance( final int x, final int y, final int z, final int dist, final boolean toroidal, IntBag result, IntBag xPos, IntBag yPos, IntBag zPos )
    {
        if( xPos == null )
            xPos = new IntBag();
        if( yPos == null )
            yPos = new IntBag();
        if( zPos == null )
            zPos = new IntBag();

        getNeighborsMaxDistance( x, y, z, dist, toroidal, xPos, yPos, zPos );

        result.clear();
        for( int i = 0 ; i < xPos.numObjs ; i++ )
            result.add( field[xPos.objs[i]][yPos.objs[i]][zPos.objs[i]] );
    }

    /*
     * Gets all neighbors of a location that satisfy abs(x-X) + abs(y-Y) + abs(z-Z) <= d
     * Returns the neighbors and their x, y and z positions; xPos, yPos and zPos can be null if you don't care about the x, y and z positions.
     */
    public final void getNeighborsHamiltonianDistance( final int x, final int y, final int z, final int dist, final boolean toroidal, IntBag result, IntBag xPos, IntBag yPos, IntBag zPos )
    {
        if( xPos == null )
            xPos = new IntBag();
        if( yPos == null )
            yPos = new IntBag();
        if( zPos == null )
            zPos = new IntBag();

        getNeighborsHamiltonianDistance( x, y, z, dist, toroidal, xPos, yPos, zPos );

        result.clear();
        for( int i = 0 ; i < xPos.numObjs ; i++ )
            result.add( field[xPos.objs[i]][yPos.objs[i]][zPos.objs[i]] );
    }

    }
