package sim.field.grid;
import sim.field.*;
import sim.util.*;

/**
    A storage facility for sparse objects in discrete 2D space, using HashMaps.  SparseGrid2D differs from ObjectGrid2D
    in several respects:
    
    <ul>
    <li>SparseGrid2D can store more than one object at a location.  ObjectGrid2D cannot.
    <li>ObjectGrid2D can store an object at more than one location (though it's bad form!).
    <li>SparseGrid2D can efficiently (O(1)) tell you the location of an object.
    <li>SparseGrid2D can efficiently (O(#objs)) scan through all objects.  The best you can do with ObjectGrid2D is search its array (which might have many empty slots).
    <li>Storing an object, finding its location, or changing its location, in a SparseGrid2D is O(1) but requires several HashMap lookups and/or removes, which has a significant constant overhead.
    </ul>

    <p>Generally speaking, if you have a grid of objects, one per location, you should use an ObjectGrid2D.  If you have a large grid occupied by a few objects, or those objects can pile up on the same grid location, you should use a SparseGrid2D.
    
    <p>In either case, you might consider storing the location of an object IN THE OBJECT ITSELF if you need to query for the object location often -- it's faster than the hashtable lookup in SparseGrid2D, and certainly faster than searching the entire array of an ObjectGrid2D.
    
    <p>The object-access method names for SparseGrid2D are intentionally different than the get/set and g/s standard set by ObjectGrid2D, IntGrid2D, and DoubleGrid2D.  This is to remind you that you're accessing Bags of objects and not objects directly.
*/

public class SparseGrid2D extends SparseField implements Grid2D
    {
    protected int width;
    protected int height;
    
    public SparseGrid2D(int width, int height)
        {
        this.width = width;
        this.height = height;
        }
        
    public final int getWidth() { return width; }
    
    public final int getHeight() { return height; }
    
    public final int tx(final int x) { return (x % width + width) % width; }

    public final int ty(final int y) { return (y % height + height) % height; }

    public final int stx(final int x) 
        { if (x >= 0) { if (x < width) return x; return x - width; } return x + width; }

    public final int sty(final int y) 
        { if (y >= 0) { if (y < height) return y ; return y - height; } return y + height; }

    public final int ulx(final int x, final int y) { return x - 1; }

    public final int uly(final int x, final int y) { if ((x & 1) == 0) return y - 1; return y; }

    public final int urx(final int x, final int y) { return x + 1; }

    public final int ury(final int x, final int y) { if ((x & 1) == 0) return y - 1; return y; }
        
    public final int dlx(final int x, final int y) { return x - 1; }

    public final int dly(final int x, final int y) { if ((x & 1) == 0) return y ; return y + 1; }
    
    public final int drx(final int x, final int y) { return x + 1; }

    public final int dry(final int x, final int y) { if ((x & 1) == 0) return y ; return y + 1; }

    public final int upx(final int x, final int y) { return x; }

    public final int upy(final int x, final int y) { return y - 1; }

    public final int downx(final int x, final int y) { return x; }

    public final int downy(final int x, final int y) { return y + 1; }
    
    public final Bag getObjectsAtLocation(final int x, final int y)
        {
        return getObjectsAtLocation(new Int2D(x,y));
        }

    public final Double2D getObjectLocationAsDouble2D(Object obj)
        {
        return new Double2D((Int2D) super.getRawObjectLocation(obj));
        }

    public Int2D getObjectLocation(Object obj)
        {
        return (Int2D) super.getRawObjectLocation(obj);
        }
    
    public final Bag removeObjectsAtLocation(final int x, final int y)
        {
        return removeObjectsAtLocation(new Int2D(x,y));
        }

    public Bag removeObjectsAtLocation(Int2D location)
        {
        return super.removeObjectsAtLocation(location);
        }
    
    public final boolean setObjectLocation(final Object obj, final int x, final int y)
        {
        return setObjectLocation(obj,new Int2D(x,y));
        }
    
    public boolean setObjectLocation(Object obj, final Int2D location)
        {
        return super.setObjectLocation(obj, location);
        }

    /*
     * Gets all neighbors of a location that satisfy max( abs(x-X) , abs(y-Y) ) <= d
     * Returns the x and y positions of the neighbors (including original location).
     */
    public final void getNeighborsMaxDistance( final int x, final int y, final int dist, final boolean toroidal, IntBag xPos, IntBag yPos )
    {
        // won't work for negative distances
        if( dist < 0 )
        {
            throw new RuntimeException( "Runtime exception in method getNeighborsMaxDistance: Distance must be positive" );
        }

        if( xPos == null || yPos == null )
        {
            throw new RuntimeException( "Runtime exception in method getNeighborsMaxDistance: xPos and yPos should not be null" );
        }

        xPos.clear();
        yPos.clear();

        // for toroidal environments the code will be different because of wrapping arround
        if( toroidal )
        {
            // compute xmin and xmax for the neighborhood
            for( int x0 = x-dist ; x0 <= x+dist ; x0++ )
            {
                final int x_0 = stx(x0);
                // compute ymin and ymax for the neighborhood
                for( int y0 = y-dist ; y0 <= y+dist ; y0++ )
                {
                    final int y_0 = sty(y0);
                    xPos.add( x_0 );
                    yPos.add( y_0 );
                }
            }
        }
        else // not toroidal
        {
            // compute xmin and xmax for the neighborhood such that they are within boundaries
            for( int x0 = ((x-dist>=0)?x-dist:0) ; x0 <= ((x+dist<=width-1)?x+dist:width-1) ; x0++ )
            {
                final int x_0 = x0;
                // compute ymin and ymax for the neighborhood such that they are within boundaries
                for( int y0 = ((y-dist>=0)?y-dist:0) ; y0 <= ((y+dist<=height-1)?y+dist:height-1) ; y0++ )
                {
                    final int y_0 = y0;
                    xPos.add( x_0 );
                    yPos.add( y_0 );
                }
            }
        }
    }


    /*
     * Gets all neighbors of a location that satisfy abs(x-X) + abs(y-Y) <= d
     * Returns the x and y positions of the neighbors (including original location).
     */
    public final void getNeighborsHamiltonianDistance( final int x, final int y, final int dist, final boolean toroidal, IntBag xPos, IntBag yPos )
    {
        // won't work for negative distances
        if( dist < 0 )
        {
            throw new RuntimeException( "Runtime exception in method getNeighborsHamiltonianDistance: Distance must be positive" );
        }

        if( xPos == null || yPos == null )
        {
            throw new RuntimeException( "Runtime exception in method getNeighborsHamiltonianDistance: xPos and yPos should not be null" );
        }

        xPos.clear();
        yPos.clear();

        // for toroidal environments the code will be different because of wrapping arround
        if( toroidal )
        {
            // compute xmin and xmax for the neighborhood
            for( int x0 = x-dist ; x0 <= x+dist ; x0++ )
            {
                final int x_0 = stx(x0);
                // compute ymin and ymax for the neighborhood; they depend on the curreny x0 value
                for( int y0 = y-(dist-((x0-x>=0)?x0-x:x-x0)) ; y0 <= y+(dist-((x0-x>=0)?x0-x:x-x0)) ; y0++ )
                {
                    final int y_0 = sty(y0);
                    xPos.add( x_0 );
                    yPos.add( y_0 );
                }
            }
        }
        else // not toroidal
        {
            // compute xmin and xmax for the neighborhood such that they are within boundaries
            for( int x0 = ((x-dist>=0)?x-dist:0) ; x0 <= ((x+dist<=width-1)?x+dist:width-1) ; x0++ )
            {
                final int x_0 = x0;
                // compute ymin and ymax for the neighborhood such that they are within boundaries
                // they depend on the curreny x0 value
                for( int y0 = ((y-(dist-((x0-x>=0)?x0-x:x-x0))>=0)?y-(dist-((x0-x>=0)?x0-x:x-x0)):0) ;
                     y0 <= ((y+(dist-((x0-x>=0)?x0-x:x-x0))<=height-1)?y+(dist-((x0-x>=0)?x0-x:x-x0)):height-1) ;
                     y0++ )
                {
                    final int y_0 = y0;
                    xPos.add( x_0 );
                    yPos.add( y_0 );
                }
            }
        }
    }

    /*
     * Gets all neighbors of a location in hexagonal world
     * Returns the x and y positions of the neighbors (including original location).
     */
    public final void getNeighborsHexagonalDistance( final int x, final int y, final int dist, final boolean toroidal, IntBag xPos, IntBag yPos )
    {
        // won't work for negative distances
        if( dist < 0 )
        {
            throw new RuntimeException( "Runtime exception in method getNeighborsHexagonalDistance: Distance must be positive" );
        }

        if( xPos == null || yPos == null )
        {
            throw new RuntimeException( "Runtime exception in method getNeighborsHamiltonianDistance: xPos and yPos should not be null" );
        }

        xPos.clear();
        yPos.clear();

        if( toroidal && height%2==1 )
            throw new RuntimeException( "Runtime exception in getNeighborsHexagonalDistance: toroidal hexagonal environment should have even heights" );

        if( toroidal )
        {
            for( int y0 = y-dist ; y0 <= y+dist ; y0 = downy(x,y0) )
            {
                xPos.add( stx(x) );
                yPos.add( sty(y0) );
            }
            int x0, ymin, ymax;

            x0 = x;
            ymin = y-dist;
            ymax = y+dist;
            for( int i = 1 ; i <= dist ; i++ )
            {
                final int temp_ymin = ymin;
                ymin = dly( x0, ymin );
                ymax = uly( x0, ymax );
                x0 = dlx( x0, temp_ymin );
                for( int y0 = ymin ; y0 <= ymax ; y0 = downy(x0,y0) )
                {
                    xPos.add( stx(x0) );
                    yPos.add( sty(y0) );
                }
            }
            x0 = x;
            ymin = y-dist;
            ymax = y+dist;
            for( int i = 1 ; i <= dist ; i++ )
            {
                final int temp_ymin = ymin;
                ymin = dry( x0, ymin );
                ymax = ury( x0, ymax );
                x0 = drx( x0, temp_ymin );
                for( int y0 = ymin ; y0 <= ymax ; y0 = downy(x0,y0) )
                {
                    xPos.add( stx(x0) );
                    yPos.add( sty(y0) );
                }
            }
        }
        else // not toroidal
        {
            if( x < 0 || x >= width || y < 0 || y >= height )
                throw new RuntimeException( "Runtime exception in method getNeighborsHexagonalDistance: invalid initial position" );

            for( int y0 = y-dist ; y0 <= ((y+dist<height)?y+dist:height-1) ; y0 = downy(x,y0) )
            {
                xPos.add( x );
                yPos.add( y0 );

            }
            int x0, ymin, ymax;

            x0 = x;
            ymin = y-dist;
            ymax = y+dist;
            for( int i = 1 ; i <= dist ; i++ )
            {
                final int temp_ymin = ymin;
                ymin = dly( x0, ymin );
                ymax = uly( x0, ymax );
                x0 = dlx( x0, temp_ymin );
                if( x0 >= 0 )
                    for( int y0 = ymin ; y0 <= ((ymax<height)?ymax:height-1) ; y0 = downy(x0,y0) )
                    {
                        if( y0 >= 0 )
                        {
                            xPos.add( x0 );
                            yPos.add( y0 );
                        }
                    }
            }

            x0 = x;
            ymin = y-dist;
            ymax = y+dist;
            for( int i = 1 ; i <= dist ; i++ )
            {
                final int temp_ymin = ymin;
                ymin = dry( x0, ymin );
                ymax = ury( x0, ymax );
                x0 = drx( x0, temp_ymin );
                if( x0 < width )
                    for( int y0 = ymin ; y0 <= ((ymax<height)?ymax:height) ; y0 = downy(x0,y0) )
                    {
                        if( y0 >= 0 )
                        {
                            xPos.add( x0 );
                            yPos.add( y0 );
                        }
                    }
            }
        }
    }

    /*
     * Gets all neighbors of a location that satisfy max( abs(x-X) , abs(y-Y) ) <= d
     * Returns the neighbors and their x and y positions; xPos and yPos can be null if you don't care about the x and y positions.
     */
    public final void getNeighborsMaxDistance( final int x, final int y, final int dist, final boolean toroidal, Bag result, IntBag xPos, IntBag yPos )
    {
        if( xPos == null )
            xPos = new IntBag();
        if( yPos == null )
            yPos = new IntBag();

        getNeighborsMaxDistance( x, y, dist, toroidal, xPos, yPos );
        getObjectsAtLocations(xPos,yPos,result);
    }

    /*
     * Gets all neighbors of a location that satisfy abs(x-X) + abs(y-Y) <= d
     * Returns the neighbors and their x and y positions; xPos and yPos can be null if you don't care about the x and y positions.
     */
    public final void getNeighborsHamiltonianDistance( final int x, final int y, final int dist, final boolean toroidal, Bag result, IntBag xPos, IntBag yPos )
    {
        if( xPos == null )
            xPos = new IntBag();
        if( yPos == null )
            yPos = new IntBag();

        getNeighborsHamiltonianDistance( x, y, dist, toroidal, xPos, yPos );
        getObjectsAtLocations(xPos,yPos,result);
    }

    /*
     * Gets all neighbors of a location in hexagonal world
     * Returns the neighbors and their x and y positions; xPos and yPos can be null if you don't care about the x and y positions.
     */
    public final void getNeighborsHexagonalDistance( final int x, final int y, final int dist, final boolean toroidal, Bag result, IntBag xPos, IntBag yPos )
    {
        if( xPos == null )
            xPos = new IntBag();
        if( yPos == null )
            yPos = new IntBag();

        getNeighborsHexagonalDistance( x, y, dist, toroidal, xPos, yPos );
        getObjectsAtLocations(xPos,yPos,result);
    }
    
    /** For each <xPos,yPos> location, puts all such objects into the result bag.  Returns the result bag.
        If the provided result bag is null, one will be created and returned. */
    public Bag getObjectsAtLocations(final IntBag xPos, final IntBag yPos, Bag result)
        {
        if (result==null) result = new Bag();
        else result.clear();

        final int len = xPos.numObjs;
        final int[] xs = xPos.objs;
        final int[] ys = yPos.objs;
        for(int i=0; i < len; i++)
            result.addAll(result.numObjs, getObjectsAtLocation(xs[i],ys[i]));
        return result;
        }
    }


