package sim.portrayal.grid;
import sim.portrayal.*;
import sim.field.grid.*;
import java.awt.*;
import java.awt.image.*;

/**
This class works like a ValueGrid2DPortrayal, <b>except</b> that it doesn't use an underlying Portrayal for the object
(instead it always draws a rectangle), and may ignore the getColor() method, so you shouldn't override that method to customize
the color function in nonlinear ways any more.  setColorTable() and setLevels() are still supported.  
Use this class instead of
ValueGrid2DPortrayal unless you need to customize how the values are drawn (other than the color range or lookup table).

<p>Additionally, FastValueGrid2DPortrayal is useful if your grid never changes past its first drawing.  For example, if
you're drawing static obstacles, cities, etc., which never change in value during the invocation of the simulation, then
FastValueGrid2DPortrayal can draw them efficiently by just drawing once into its buffer and re-displaying the buffer over
and over again.  Just pass in true in the constructor.  

<p>If your grid does change but only occasionally, you can also use this technique as well; just manually call reset() 
whenever the grid changes to inform the FastValueGrid2DPortrayal that it needs to re-compute the buffer.  reset() is threadsafe.

<h3>Important Note on Drawing Methods</h3>

FastValueGrid2DPortrayal can draw a grid in two ways.  First, it can draw each of the rects individually ("USE_BUFFER").  Second, it can create a bitmap the size of the grid (one pixel per grid location), poke the colors into the bitmap, then stretch the bitmap over the area and draw it ("DONT_USE_BUFFER").  You can specify the method by calling the <b>setBuffering()</b> method; optionally you can just let FastValueGrid2DPortrayal guess which to use ("DEFAULT").  But you should know what you're doing, as methods can be <i>much</i> faster than each other depending on the situation.  Use the following as guides

<dl><dt><b>MacOS X</b>
<dd>USE_BUFFER is much faster than DONT_USE_BUFFER in all cases, but can draw incorrectly aliased ("fuzzed out") rectangles when writing to media (a movie or a snapshot).  The DEFAULT is for MacOS X is set to USE_BUFFER in ordinary drawing, and DONT_USE_BUFFER when writing to media.
<dt><b>Windows and X Windows</b>
<dd>If you're not using any transparency (alpha), then DONT_USE_BUFFER is a tad faster than USE_BUFFER.  But if you're using transparency, then DONT_USE_BUFFER is <i>very</i> slow -- in this case, try USE_BUFFER.  Note however that in any case USE_BUFFER requires a <i>lot</i> of memory on Windows and X Windows due to poor implementation by Sun.  You'll want to increase the default memory capacity, and expect occasional pauses for full garbage collection.  You can test how often full garbage collection by running with <tt><b>java -verbose:gc ...</b></tt> and looking at the <tt>FULL GC</tt> printouts happen.  Ordinary <tt>GC</tt> you shouldn't worry about.  You can increase the default memory capacity to 50 Megabytes, for example (the default is about 20) by running with <tt><b>java -Xms20M ...</b></tt>  The DEFAULT is for XWindows and Windows to use DONT_USE_BUFFER.  You'll want to change this for sure if you're doing any transparency.
</dl>
*/

public class FastValueGrid2DPortrayal extends ValueGrid2DPortrayal
    {
    /** If assumeImmutableGrid is true, we presume that the grid doesn't change.  This allows us to just
        re-splat the buffer. */
    public FastValueGrid2DPortrayal(String valueName, boolean assumeImmutableGrid)
        {
        super(valueName);
        immutableGrid = assumeImmutableGrid;
        }

    public FastValueGrid2DPortrayal(String valueName)
        {
        this(valueName,false);
        }
        
    /** If assumeImmutableGrid is true, we presume that the grid doesn't change.  This allows us to just
        re-splat the buffer. */
    public FastValueGrid2DPortrayal(boolean assumeImmutableGrid)
        {
        super();
        immutableGrid = assumeImmutableGrid;
        }

    public FastValueGrid2DPortrayal()
        {
        this(false);
        }

    public void reset()
        {
        synchronized(this)
            {
            buffer = null;
            }
        }

    public static final int DEFAULT = 0;
    public static final int USE_BUFFER = 1;
    public static final int DONT_USE_BUFFER = 2;
    int buffering = DEFAULT;
    public int getBuffering() { return buffering; }
    public void setBuffering(int val) { buffering = val; }
    
    // Determines if we should buffer
    boolean shouldBuffer(Graphics2D graphics)
        {
	// We can either draw lots of rects, or we can pixels to a small bitmap, then
	// stretch the bitmap into rects using drawImage.  Which technique is faster depends
	// on the OS unfortunately.  Solaris prefers the bitmap.  Linux prefers the rects
	// very much.  Windows prefers the rects,
	// except for small draws where bitmaps have a slight edge (which we'll not consider).
	// MacOS X prefers bitmaps, but will not stretch and draw to an image buffer
	// without doing fancy-pants interpolation which looks horrible, so we have to check for that.

	// For now we'll do:
	// in X Windows and Windows, DON'T use the buffer
	// in MacOS X, use the buffer for non-image writes ONLY
	
        if (buffering==USE_BUFFER) return true;
        else if (buffering==DONT_USE_BUFFER) return false;
	else if (sim.display.Display2D.isMacOSX)
	    return (graphics.getDeviceConfiguration().
			   getDevice().getType() != GraphicsDevice.TYPE_IMAGE_BUFFER);
	else if (sim.display.Display2D.isWindows)
	    return immutableGrid;
	else // it's Linux or Solaris
	    return false;
        }
        
    boolean immutableGrid = false;
    BufferedImage buffer;

    /** Why is this version final?  Because in some cases we call it and in others we compute
        the color on our own -- thus we don't want to be inconsistent if the user attempts to
        subclass FastValueGrid2DPortrayal and change this method. */
    public final Color getColor(final double level)
        { return super.getColor(level); }
    
    // Should draw itself within the box from (0,0) to (1,1)
    public void draw(Object object, Graphics2D graphics, DrawInfo2D info)
        {
        if (field==null) return;
        
        // first question: determine the range in which we need to draw.
        final int maxX = field.getWidth();
        final int maxY = field.getHeight(); 
        if (maxX == 0 || maxY == 0) return;

        final double xScale = info.draw.width / maxX;
        final double yScale = info.draw.height / maxY;
        int startx = (int)((info.clip.x - info.draw.x) / xScale);
        int starty = (int)((info.clip.y - info.draw.y) / yScale); // assume that the X coordinate is proportional -- and yes, it's _width_
        int endx = /*startx +*/ (int)((info.clip.x - info.draw.x + info.clip.width) / xScale) + /*2*/ 1;  // with rounding, width be as much as 1 off
        int endy = /*starty +*/ (int)((info.clip.y - info.draw.y + info.clip.height) / yScale) + /*2*/ 1;  // with rounding, height be as much as 1 off
        
        // next we determine if this is a DoubleGrid2D or an IntGrid2D
        
        final boolean isDoubleGrid2D = (field instanceof DoubleGrid2D);
        final double[][] doubleField = (isDoubleGrid2D ? ((DoubleGrid2D) field).field : null);
        final int[][] intField = (isDoubleGrid2D ? null : ((IntGrid2D) field).field);

        // local vars a little faster
        final Color[] _colors = colors;
        final boolean hasColorTable = (_colors!=null);
        
        int argb;
        
        if (shouldBuffer(graphics))
            {
            // create new buffer if needed
            boolean newBuffer = false;
            BufferedImage _buffer = null;  // make compiler happy
            
            synchronized(this)
                {
                if (buffer==null || buffer.getWidth() != maxX || buffer.getHeight() != maxY)
                    {
                    buffer = new BufferedImage(maxX,maxY,BufferedImage.TYPE_INT_ARGB);  // transparency allowed
                    newBuffer = true;
                    }
                _buffer = buffer;
                }

            if (newBuffer || !immutableGrid)  // we have to load the buffer
                {
		if (endx > maxX) endx = maxX;
		if (endy > maxY) endy = maxY;
                if( startx < 0 ) startx = 0;
                if( starty < 0 ) starty = 0;
                
                if (immutableGrid)
                    {
                    // must load ENTIRE buffer
                    startx = 0; starty = 0; endx = maxX; endy = maxY;
                    }

                for(int x=startx;x<endx;x++)
                    for(int y=starty;y<endy;y++)
                        {
                        double level = (isDoubleGrid2D ?  doubleField[x][y] : intField[x][y]);
                        
                        if (hasColorTable && (level >= 0) && (level < _colors.length))
                            {
                            argb = _colors[(int)level].getRGB();
                            }
                        else
                            {
                            final double interpolation = (level - minLevel) / (maxLevel - minLevel);
                            final int alpha = (maxAlpha == minAlpha ? minAlpha : 
                                                    (int)(interpolation * (maxAlpha - minAlpha) + minAlpha));
                            final int red = (maxRed == minRed ? minRed : 
                                                    (int)(interpolation * (maxRed - minRed) + minRed));
                            final int green = (maxGreen == minGreen ? minGreen : 
                                                    (int)(interpolation * (maxGreen - minGreen) + minGreen));
                            final int blue = (maxBlue == minBlue ? minBlue : 
                                                    (int)(interpolation * (maxBlue - minBlue) + minBlue));
                            argb = (alpha << 24) | (red << 16) | (green << 8) | blue;
                            }
                        _buffer.setRGB(x,y,argb);
                        }
                }
            graphics.drawImage(_buffer, (int)info.draw.x, (int)info.draw.y, (int)info.draw.width, (int)info.draw.height,null);
            }
        else
            {
            buffer = null;  // GC the buffer in case the user had changed his mind
            
            if (endx > maxX) endx = maxX;
            if (endy > maxY) endy = maxY;
            if( startx < 0 ) startx = 0;
            if( starty < 0 ) starty = 0;
            int _x = 0;
            int _y = 0;
            int _width = 0;
            int _height = 0;
            for(int x=startx;x<endx;x++)
                for(int y=starty;y<endy;y++)
                    {
		    Color c = getColor(isDoubleGrid2D ?  doubleField[x][y] : intField[x][y]);
                    if (c.getAlpha() == 0) continue;
		    graphics.setColor(c);
                   	 
                   _x = (int)(info.draw.x + (xScale) * x);
                   _y = (int)(info.draw.y + (yScale) * y);
                   _width = (int)(info.draw.x + (xScale) * (x+1)) - _x;
                   _height = (int)(info.draw.y + (yScale) * (y+1)) - _y;
                   
                    // draw
                    graphics.fillRect(_x,_y,_width,_height);
                    }
            }
        }
    }
