package sim.portrayal.grid;
import sim.portrayal.*;
import sim.portrayal.simple.*;
import sim.field.grid.*;
import java.awt.*;
import java.awt.geom.*;
import sim.util.*;

/**
This class is capable of portraying the DoubleGrid2D and IntGrid2D fields (and <b>only</b> those two fields -- or subclasses).
It is fairly customizable, and this flexibility
comes at a cost in drawing speed.  If you just need to draw your field as a grid of squares, you might look into
the much simpler, and faster (and aptly named) FastValueGrid2DPortrayal instead.

<p>Like other FieldPortrayal2Ds, this class uses an underlying SimplePortrayal2D to draw each separate element
in the grid.  A default SimplePortrayal2D is provided which draws squares.  In the default, the color for the square is
determined by looking up the value of the square in a user-provided color-table, or if there is none, by
interpolating it between two user-provided colors.  See the setColorTable() and setLevels() methods. 

<p>Here's a trick you might consider in specifying interpolations.  Rather than draw from white to red (for example),
you might consider setting the backdrop of the display to white, and then instead draw from FULLY TRANSPARENT white to
FULLY OPAQUE red.  That is, from Color(0,0,0,0) to Color(255,0,0,255).  Fully transparent colors are not drawn; and not drawing
at all is significantly faster than drawing for no reason!  Plus you can stack multiple ValueGrid2DPortrayals on top
of one another and let the transparency bleed through for a nice effect.  The alpha channel is your friend.

<p>By default the min Level and the max Level are the same (1.0), and the alpha values for both are 0 (totally transparent).
Thus if you want a range, you must specify it.  This is intentional, because this way if you want to use a color
table instead (say, to specify three colors for the integers 0, 1, and 2), you can specify them, and ALL other grid values
will be automatically transparent.

<p>If you would like more control over the color of your values (perhaps to implement a nonlinear function of the colors),
you can override the getColor() function to define your own custom color.

<p>You can also provide your own custom SimplePortrayal2D (use setPortrayalForAll(...) ) to draw elements as you
see fit rather than as rectangles.  Your SimplePortrayal2D should expect objects passed to its draw method
to be of type MutableDouble.  Do not hold onto this array -- it will be reused.
*/

public class ValueGrid2DPortrayal extends FieldPortrayal2D
    {
    public AbstractGrid2D field;
    
    public void setField(Object field)
        {
        if (field instanceof DoubleGrid2D ||
            field instanceof IntGrid2D ) this.field = (AbstractGrid2D) field;
        else throw new RuntimeException("Invalid field for ValueGrid2DPortrayal: " + field);
        }
        
    public Object getField()
        {
        return field;
        }

    public int minRed = 0;
    public int minBlue = 0;
    public int minGreen = 0;
    public int minAlpha = 0;
    public int maxRed = 0;
    public int maxBlue = 0;
    public int maxGreen = 0;
    public int maxAlpha = 0;
    public double maxLevel = 0;
    public double minLevel = 0;
    public final Color clearColor = new Color(0,0,0,0);
    public Color minColor = clearColor;  // used when minLevel = maxLevel
    
    public static final int COLOR_DISCRETIZATION = 257;
    
    /** User-provided color table */
    public Color[] colors;
    
    /** Our cache for linear interpolation -- faster than generating all those Colors over and over again */
    Bag[] colorCache = new Bag[COLOR_DISCRETIZATION];

//    /** Faster but less accurate cache */
//    Color[] colorCache2 = new Color[COLOR_DISCRETIZATION];
    
    SimplePortrayal2D defaultPortrayal = new ValuePortrayal2D(this);

    public String valueName;
    
    public String getValueName() { return valueName; }
    
    public ValueGrid2DPortrayal()
        {
        this("Value");
        }

    public ValueGrid2DPortrayal(String valueName)
    {
        this.valueName = valueName;
        for(int x=0;x<COLOR_DISCRETIZATION;x++) colorCache[x] = new Bag();
//        for(int x=0;x<COLOR_DISCRETIZATION;x++) colorCache2[x] = clearColor;
    }
    
    /** This method is called by the default inspector to filter new values set by the user.
        You should return the "corrected" value if the given value is invalid. The default version
        of this method bases values on the values passed into the setLevels() and setColorTable() methods. */
    public double newValue(int x, int y, double value)
        {
        if (field instanceof IntGrid2D) value = (int) value;
        
        if (colors!=null && value >= 0 && value < colors.length)
            return value;
        else if (value <= maxLevel && value >= minLevel)
            return value;
        
        // at this point we need to reset to current value
        java.awt.Toolkit.getDefaultToolkit().beep();
        if (field != null)
            {
            if (field instanceof DoubleGrid2D)
                return ((DoubleGrid2D)field).field[x][y];
            else return ((IntGrid2D)field).field[x][y];
            }
        else return minLevel; // return *something*
        }

    /** Override this if you'd like to customize the color for values in the portrayal.  The default version
        looks up the value in the colors[] table, else computes the interpolated color and grabs it out of
        a predefined color cache (there can't be more than about 1024 or so interpolated colors, max). 
        */
    
    public Color getColor(double level)
        {
        if (colors != null && level >= 0 && level < colors.length)
            {
            return colors[(int)level];
            }
        else
            {
            if (level > maxLevel) level = maxLevel;
            if (level < minLevel) level = minLevel;
            if (level == minLevel) return minColor;  // so we don't divide by zero (maxLevel - minLevel)
            
            final double interpolation = (level - minLevel) / (maxLevel - minLevel);
            final int alpha = (maxAlpha == minAlpha ? minAlpha : (int)(interpolation * (maxAlpha - minAlpha) + minAlpha));
            if (alpha==0) return clearColor;
            
            final int red = (maxRed == minRed ? minRed : (int)(interpolation * (maxRed - minRed) + minRed));
            final int green = (maxGreen == minGreen ? minGreen : (int)(interpolation * (maxGreen - minGreen) + minGreen));
            final int blue = (maxBlue == minBlue ? minBlue : (int)(interpolation * (maxBlue - minBlue) + minBlue));
            
            final int rgb = (alpha << 24) | (red << 16) | (green << 8) | blue;
            
            // look up color in cache  -- less garbage collection and slightly faster
            Bag colors = colorCache[(int)(interpolation * (COLOR_DISCRETIZATION-1))];
            for(int x=0;x<colors.numObjs;x++)
                {
                Color c = (Color)(colors.objs[x]);
                if (c.getRGB()==rgb)  // it's the right color
                    return c;
                }
            Color c = new Color(rgb,(alpha!=0));
            colors.add(c);
            return c;
            }
        }
        
    /** Specifies that if a value (cast into an int) in the IntGrid2D or DoubleGrid2D falls in the range 0 ... colors.length,
        then that index in the colors table should be used to represent that value.  Otherwise, values in
        setLevels(...) are used.  You can remove the color table by passing in null here.  Returns the old color table. */
    public Color[] setColorTable(Color[] colors)
        {
        Color[] retval = this.colors;
        this.colors = colors;
        return retval;
        }
    
    /** Sets the color levels for the ValueGrid2DPortrayal values for use by the default getColor(...)
        method.  These are overridden by any array provided in setColorTable().  If the value in the IntGrid2D or DoubleGrid2D
        is less than or equal to minLevel, then minColor is used.  If the value is greater than or equal to maxColor, then
        maxColor is used.  Otherwise a linear interpolation from minColor to maxColor is used. */
    public void setLevels(double minLevel, double maxLevel, Color minColor, Color maxColor)
        {
        if (maxLevel < minLevel) throw new RuntimeException("maxLevel cannot be less than minLevel");
        minRed = minColor.getRed(); minGreen = minColor.getGreen(); minBlue = minColor.getBlue(); minAlpha = minColor.getAlpha();
        maxRed = maxColor.getRed(); maxGreen = maxColor.getGreen(); maxBlue = maxColor.getBlue(); maxAlpha = maxColor.getAlpha();
        this.maxLevel = maxLevel; this.minLevel = minLevel;
        this.minColor = minColor;

        // reset cache
        for(int x=0;x<COLOR_DISCRETIZATION;x++) colorCache[x] = new Bag();
//        for(int x=0;x<COLOR_DISCRETIZATION;x++) 
//                    {
//                    colorCache2[x] =
//                    new Color( 	x * (maxColor.getRed()-minColor.getRed()) / (COLOR_DISCRETIZATION-1) + minColor.getRed(), 
//                                x * (maxColor.getGreen()-minColor.getGreen()) / (COLOR_DISCRETIZATION-1) + minColor.getGreen(), 
//                                x * (maxColor.getBlue()-minColor.getBlue()) / (COLOR_DISCRETIZATION-1) + minColor.getBlue(), 
//                                x * (maxColor.getAlpha()-minColor.getAlpha()) / (COLOR_DISCRETIZATION-1) + minColor.getAlpha());
//                    }
        }
  
                          
    public Portrayal getDefaultPortrayal()
        {
        return defaultPortrayal;
        }

    // our object to pass to the portrayal
    final MutableDouble valueToPass = new MutableDouble(0);
        
    protected void hitOrDraw(Graphics2D graphics, DrawInfo2D info, Bag putInHere)
        {
        if (field==null) return;

        // Scale graphics to desired shape -- according to p. 90 of Java2D book,
        // this will change the line widths etc. as well.  Maybe that's not what we
        // want.
        
        // first question: determine the range in which we need to draw.
        // We assume that we will fill exactly the info.draw rectangle.
        // We can do the item below because we're an expensive operation ourselves
        
        final int maxX = field.getWidth(); 
        final int maxY = field.getHeight();
        if (maxX == 0 || maxY == 0) return; 
        
        final double xScale = info.draw.width / maxX;
        final double yScale = info.draw.height / maxY;
        int startx = (int)((info.clip.x - info.draw.x) / xScale);
        int starty = (int)((info.clip.y - info.draw.y) / yScale); // assume that the X coordinate is proportional -- and yes, it's _width_
        int endx = /*startx +*/ (int)((info.clip.x - info.draw.x + info.clip.width) / xScale) + /*2*/ 1;  // with rounding, width be as much as 1 off
        int endy = /*starty +*/ (int)((info.clip.y - info.draw.y + info.clip.height) / yScale) + /*2*/ 1;  // with rounding, height be as much as 1 off
        
        // next we determine if this is a DoubleGrid2D or an IntGrid2D
        
        final boolean isDoubleGrid2D = (field instanceof DoubleGrid2D);
        final double[][] doubleField = (isDoubleGrid2D ? ((DoubleGrid2D) field).field : null);
        final int[][] intField = (isDoubleGrid2D ? null : ((IntGrid2D) field).field);
        
        // the drawinfo that the object's portrayal will use -- we fill in the blanks later
        DrawInfo2D newinfo = new DrawInfo2D(new Rectangle2D.Double(0,0, xScale, yScale), info.clip);

        if (maxX>0 || maxY > 0)
            {
            Portrayal p = getPortrayalForObject(valueToPass);
            if (!(p instanceof SimplePortrayal2D))
                throw new RuntimeException("Unexpected Portrayal " + p + " for object " + 
                                            valueToPass + " -- expected a SimplePortrayal2D");
            SimplePortrayal2D portrayal = (SimplePortrayal2D) p;

            if (endx > maxX) endx = maxX;
            if (endy > maxY) endy = maxY;
            if( startx < 0 ) startx = 0;
            if( starty < 0 ) starty = 0;
            for(int x=startx;x<endx;x++)
                for(int y=starty;y<endy;y++)
                    {
                    // dunno how much of a hit we get for doing this if/then each and every time...
                    valueToPass.val = (isDoubleGrid2D ?  doubleField[x][y] : intField[x][y]);
                    
                    // translate --- the   + newinfo.width/2.0  etc. moves us to the center of the object
                    newinfo.draw.x = (int)(info.draw.x + (xScale) * x);
                    newinfo.draw.y = (int)(info.draw.y + (yScale) * y);
                    newinfo.draw.width = (int)(info.draw.x + (xScale) * (x+1)) - newinfo.draw.x;
                    newinfo.draw.height = (int)(info.draw.y + (yScale) * (y+1)) - newinfo.draw.y;
                    
                    // adjust drawX and drawY to center
                    newinfo.draw.x += newinfo.draw.width / 2.0;
                    newinfo.draw.y += newinfo.draw.height / 2.0;
                    
                    if (graphics == null)
                        {
                        if (portrayal.hitObject(valueToPass, newinfo))
                            putInHere.add(getWrapper(valueToPass.val, x, y));
                        }
                    else
                        portrayal.draw(valueToPass, graphics, newinfo);
                    }
            }
        }

    // ValueGrid2DPortrayal's objects are instances of MutableDouble
    public LocationWrapper getWrapper(double val, int x, int y)
        {
        return new LocationWrapper( new MutableDouble(val),  // something unique to return for getObject()
                                    new Int2D(x, y), this )  // it's location
            {
            public Object getObject()
                {
                Int2D loc = (Int2D) location;
                MutableDouble val = (MutableDouble) this.object;
                // update the current value
                if (field instanceof DoubleGrid2D) val.val = ((DoubleGrid2D)field).field[loc.x][loc.y];
                else val.val = ((IntGrid2D)field).field[loc.x][loc.y];
                return val;
                }
            
            public String getLocationName()
                {
                return ((Int2D)location).toCoordinates();
                }
            };
        }
    }
