package sim.portrayal3d;
import java.awt.Color;
import sim.util.*;

/**
 * @author Catalin Balan
 * 
 * it takes the color-deciding code from ValueGrid2DPortrayal.
 */

public class SimpleColorInterpolator implements ColorCoding 
{
	public int minRed = 0;
	public int minBlue = 0;
	public int minGreen = 0;
	public int minAlpha = 0;
	public int maxRed = 0;
	public int maxBlue = 0;
	public int maxGreen = 0;
	public int maxAlpha = 0;
	public double maxLevel = 0;
	public double minLevel = 0;
	public final Color clearColor = new Color(0,0,0,0);
	public Color minColor = clearColor;  // used when minLevel = maxLevel
    
	public static final int COLOR_DISCRETIZATION = 257;
    
	/** User-provided color table */
	public Color[] colors;
    
	/** Our cache for linear interpolation -- faster than generating all those Colors over and over again */
	Bag[] colorCache = new Bag[COLOR_DISCRETIZATION];

//	/** Faster but less accurate cache */
//	Color[] colorCache2 = new Color[COLOR_DISCRETIZATION];	
	
    public SimpleColorInterpolator()
        {
        setLevels(0,1,Color.black,Color.white);
        }
            
	/** Sets the color levels for the ValueGrid2DPortrayal values for use by the default getColor(...)
		method.  These are overridden by any array provided in setColorTable().  If the value in the IntGrid2D or DoubleGrid2D
		is less than or equal to minLevel, then minColor is used.  If the value is greater than or equal to maxColor, then
		maxColor is used.  Otherwise a linear interpolation from minColor to maxColor is used. */
	public void setLevels(double minLevel, double maxLevel, Color minColor, Color maxColor)
		{
		if (maxLevel < minLevel) throw new RuntimeException("maxLevel cannot be less than minLevel");
		minRed = minColor.getRed(); minGreen = minColor.getGreen(); minBlue = minColor.getBlue(); minAlpha = minColor.getAlpha();
		maxRed = maxColor.getRed(); maxGreen = maxColor.getGreen(); maxBlue = maxColor.getBlue(); maxAlpha = maxColor.getAlpha();
		this.maxLevel = maxLevel; this.minLevel = minLevel;
		this.minColor = minColor;

		// reset cache
		for(int x=0;x<COLOR_DISCRETIZATION;x++) colorCache[x] = new Bag();
//		for(int x=0;x<COLOR_DISCRETIZATION;x++) 
//					{
//					colorCache2[x] =
//					new Color( 	x * (maxColor.getRed()-minColor.getRed()) / (COLOR_DISCRETIZATION-1) + minColor.getRed(), 
//								x * (maxColor.getGreen()-minColor.getGreen()) / (COLOR_DISCRETIZATION-1) + minColor.getGreen(), 
//								x * (maxColor.getBlue()-minColor.getBlue()) / (COLOR_DISCRETIZATION-1) + minColor.getBlue(), 
//								x * (maxColor.getAlpha()-minColor.getAlpha()) / (COLOR_DISCRETIZATION-1) + minColor.getAlpha());
//					}
		}

	public SimpleColorInterpolator(Color[] colors)
		{
		setColorTable(colors);
		}
	
	/** Specifies that if a value (cast into an int) in the IntGrid2D or DoubleGrid2D falls in the range 0 ... colors.length,
		then that index in the colors table should be used to represent that value.  Otherwise, values in
		setLevels(...) are used.  You can remove the color table by passing in null here.  Returns the old color table. */
	public Color[] setColorTable(Color[] colors)
		{
		Color[] retval = this.colors;
		this.colors = colors;
		return retval;
		}
	
	/** Override this if you'd like to customize the color for values in the portrayal.  The default version
		looks up the value in the colors[] table, else computes the interpolated color and grabs it out of
		a predefined color cache (there can't be more than about 1024 or so interpolated colors, max). 
		*/
    
	public Color getColor(double level)
		{
		if (colors != null && level >= 0 && level < colors.length)
			{
			return colors[(int)level];
			}
		else
			{
			if (level > maxLevel) level = maxLevel;
			if (level < minLevel) level = minLevel;
			if (level == minLevel) return minColor;  // so we don't divide by zero (maxLevel - minLevel)
            
			final double interpolation = (level - minLevel) / (maxLevel - minLevel);
                        
			// look up color in cache  -- less garbage collection and slightly faster
			final int alpha = (maxAlpha == minAlpha ? minAlpha : (int)(interpolation * (maxAlpha - minAlpha) + minAlpha));
			if (alpha==0) return clearColor;
                        final int red = (maxRed == minRed ? minRed : (int)(interpolation * (maxRed - minRed) + minRed));
			final int green = (maxGreen == minGreen ? minGreen : (int)(interpolation * (maxGreen - minGreen) + minGreen));
			final int blue = (maxBlue == minBlue ? minBlue : (int)(interpolation * (maxBlue - minBlue) + minBlue));
			final int rgb = (alpha << 24) | (red << 16) | (green << 8) | blue;
			Bag colors = colorCache[(int)(interpolation * (COLOR_DISCRETIZATION-1))];
			for(int x=0;x<colors.numObjs;x++)
				{
				Color c = (Color)(colors.objs[x]);
				if (c.getRGB()==rgb)  // it's the right color
					return c;
				}
			Color c = new Color(rgb,(alpha!=0));
			colors.add(c);
			return c;
                        
                        // ...or much faster but lower color resolution (by factor of 2 or so)
//                      return colorColorCache2[(int)(interpolation * (COLOR_DISCRETIZATION-1))];
			}
		}	
}
