 package sim.portrayal3d;

import sim.util.*;
import sim.field.*;
import sim.portrayal.*;
import sim.portrayal3d.simple.*;

import java.util.*;

import javax.media.j3d.*;
import javax.vecmath.*;
/**
 * Generic functionality for a field portrayal in Java3D mode:
 * It handles the creation and update of the part of the scene
 * it is responsible for, including dynamic addition/deletion 
 * of objects.
 * 
 * The implementors must provide menas for converting the position
 * of the objects in the <code>field</code> into vecmath.Vector3d
 * through <code>getLocationOfObjectAsVector3d</code>
 * 
 *  
 * @author Gabriel Balan
 */
public abstract class SparseFieldPortrayal3D extends FieldPortrayal3D
{
	
	public SparseField field;
	
	public SparseFieldPortrayal3D(SparseField field, double width, double height)
	{
		this(field,0,0,width,height);
	}
	
	public SparseFieldPortrayal3D(SparseField field, double x, double y, double width, double height)
	{
		this(field, x,y,0,width,height,0);
	}
	
	
	public SparseFieldPortrayal3D(SparseField field, double x, double y, double z, double dx, double dy, double dz)
	{	
		super(DefaultTransform.getDefaultTransform(x,y,z,dx,dy,dz));
		portray(field);
	}
	
	public SparseFieldPortrayal3D(SparseField field,Transform3D transf)
	{	
		super(transf);
		portray(field);
	}
	
	
	public void portray(Object grid)
	{
		if (grid instanceof SparseField) 
		    this.field = (SparseField) grid;
		else throw new RuntimeException("SparseFieldPortrayal3D cannot portray the object: " + grid);
	}
	
	/** 
	 * Nice simple white cube as default portrayal
	 * for objects that do not have any other specified to them
	 * 
	 * Note that it is not final, so it can be replaced. It
	 * was chosen for its low triangle-count.
	 */
	static Portrayal3D defaultPortrayal = new CubePortrayal3D();
	
	public Portrayal getDefaultPortrayal()
	{
		return defaultPortrayal;
	}
	
	
	/** tmp Vector3d */ 
	protected Vector3d tmpVect = new Vector3d();
	/** tmp Transform3D 
	 * it is reused, since the TGs are copying it internally*/
	protected Transform3D tmpLocalT = new Transform3D();
	
	/**
	 * Instead of allocating a new Vector3d
	 * for every call, reuse <code>tmpVect</code>, unless
	 * concurrecy is an issue.
	 **/ 
	public abstract Vector3d getLocationOfObjectAsVector3d(Object location);
	

	/**
	 * Create J3D subgraphs for all objects in the field
	 * note that each subgraph points to the simulation
	 * object it represents, for update purposes.
	 **/
	public TransformGroup createModel()
	{
		if (field==null) 
	    	return null;
	    Bag objects = field.getAllObjects();
	
		TransformGroup globalTG = new TransformGroup();	
		
		globalTG.setCapability(TransformGroup.ALLOW_CHILDREN_READ);
		globalTG.setCapability(TransformGroup.ALLOW_CHILDREN_WRITE);
		globalTG.setCapability(TransformGroup.ALLOW_CHILDREN_EXTEND);
	
		if(objects != null)
			for(int z = 0; z<objects.numObjs; z++)
			{
				Vector3d locationV3d = getLocationOfObjectAsVector3d(objects.objs[z]);
				tmpLocalT.setTranslation(locationV3d);
	
				//wrap the TG in a BG so it can be removed dynamically
				BranchGroup localBG = InsertModelForNewObject(objects.objs[z], tmpLocalT);			
				
				globalTG.addChild(localBG);
			}
			
		TransformGroup modelTG = new TransformGroup();
		modelTG.setTransform(overallT);
		modelTG.addChild(globalTG);
		modelTG.setCapability(TransformGroup.ALLOW_CHILDREN_READ);
		return modelTG;
	}
	
	/**
	 * 
	 * This function is called from createMode for each object in the 
	 * field and from the updateModel part of getModel for the
	 * new objects.
	 * 
	 * In order to dynamically add/remove the subtrees associated with
	 * children, I have to wrap their TGs into BranchGroups.
	 **/
	protected BranchGroup InsertModelForNewObject(Object o, Transform3D localT)
	{
		Portrayal p = getPortrayalForObject(o);
		if(! (p instanceof Portrayal3D))
			throw new RuntimeException("Unexpected Portrayal " + p + " for object " + 
						o + " -- expecting a PortrayalJ3D");
		Portrayal3D p3d = (Portrayal3D)p;

		p3d.setParentPortrayal(this);
		TransformGroup localTG = p3d.getModel(o, null);
		localTG.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);
		localTG.setCapability(TransformGroup.ALLOW_TRANSFORM_READ);
		localTG.setTransform(localT);
	
		BranchGroup localBG = new BranchGroup();
		localBG.setCapability(BranchGroup.ALLOW_CHILDREN_READ);
		localBG.setCapability(BranchGroup.ALLOW_DETACH);
		localBG.addChild(localTG);
		localBG.setUserData(o);
		return localBG;
	}

	
	/** 
	 * HashMap use to detect objects deleted /inserted into the field between frames.
	 * It is reused unless the new count is larger than the last capacity
	 **/
	HashMap hm = new HashMap();
	/** Since the hashmap is hiding its capacity, i have to keep track of it myself */
	int oldCapacity = 0;

	public TransformGroup getModel(Object obj, TransformGroup modelTG){
		if(modelTG == null)
			return createModel();
	
		TransformGroup globalTG = (TransformGroup)modelTG.getChild(0);
	
		Bag b = field.getAllObjects();

		if(b.numObjs > oldCapacity)
			hm = new HashMap(oldCapacity = b.numObjs);
		//I assume that hm is empty.
		for(int i=0; i<b.numObjs; ++i)
			hm.put(b.objs[i], b.objs[i]);
		
		
		for(int t= globalTG.numChildren()-1; t>=0; t--)
		{
			BranchGroup localBG = (BranchGroup)globalTG.getChild(t);
			Object fieldObj = localBG.getUserData();
			if(hm.containsKey(fieldObj))
			{
				TransformGroup localTG = (TransformGroup)localBG.getChild(0);
				Portrayal p = getPortrayalForObject(fieldObj);
				if(! (p instanceof Portrayal3D))
					throw new RuntimeException("Unexpected Portrayal " + p + " for object " + 
								fieldObj + " -- expecting a PortrayalJ3D");
	            Portrayal3D p3d = (Portrayal3D)p;
	            hm.remove(fieldObj);
	            TransformGroup localTG2 = p3d.getModel(fieldObj, localTG);
	
				Vector3d locationV3d = getLocationOfObjectAsVector3d(fieldObj);
				tmpLocalT.setTranslation(locationV3d);
				
				localTG.setTransform(tmpLocalT);
	            
	            if(localTG != localTG2)
	            	localBG.setChild(localTG2, 0);
			}
			else
				globalTG.removeChild(t);
		}
		
		Iterator newObjs = hm.values().iterator();	
		while(newObjs.hasNext())
		{
			Object fieldObj = newObjs.next();
			Vector3d locationV3d = getLocationOfObjectAsVector3d(fieldObj);
			tmpLocalT.setTranslation(locationV3d);
			//wrap the TG in a BG so it can be added 
			//and later on removed dynamically
				BranchGroup localBG = InsertModelForNewObject(fieldObj, tmpLocalT);			
				globalTG.addChild(localBG);
		}
		//hm.clear is linear in capacity, so I think
		//it's worth checking first.
		if(!hm.isEmpty())
			hm.clear();
		return modelTG;
    }
}
	    
    