package sim.util;

/** 
Int2D is more or less the same class as java.awt.Point, but it is immutable: once the x and y values are set, they cannot be changed (they're final).  Why use this immutable class when you could just use Point?  Because Point is broken with respect to hash tables.  You use Point as a key in a hash table at your own peril.  Try this: hash an object by a Point as key.  Then change the x value of the original Point.  Ta-da!  The object is lost in the hash table.

<p>One day in the far future, Int2D should also be HIGHLY efficient; since it is immutable, it can be passed by value rather than by pointer by a smart compiler.  Not today, though.  But it's not bad.

<p>This class has an elaborate hash code generation that is much more random than Sun's standard generator, but takes more time.  For very large numbers of objects, this is a good idea, but we may change it to a simpler version in the future.

<p>Int2D.equals(...) can compare by value against other Int2Ds and Double2Ds.
*/
public final class Int2D implements java.io.Serializable
    {
    public final int x;
    public final int y;
   
    public Int2D() { x = 0; y = 0; }
    public Int2D(final java.awt.Point p) { x = p.x; y = p.y; }
    public Int2D(final int x, final int y) { this.x = x; this.y = y; }
    public final int getX() { return x; }
    public final int getY() { return y; }
    public final java.awt.Point getPoint() { return new java.awt.Point(x,y); }
    public String toString() { return "Int2D["+x+","+y+"]"; }
    public String toCoordinates() { return "(" + x + ", " + y + ")"; }

    public final static int hashCodeFor(final int x, int y)
        {
	// basically we need to randomly disperse <int,int> --> int
	// I do that by hashing (shuffling) y, then xoring it with x. So I 
	// need something that will hash y to a nicely random value.
	// this taken from http://www.cris.com/~Ttwang/tech/inthash.htm

        y += ~(y << 15);
        y ^=  (y >>> 10);
        y +=  (y << 3);
        y ^=  (y >>> 6);
        y += ~(y << 11);
        y ^=  (y >>> 16);

	// nifty!  Now mix in x
	
        return x ^ y;
        }

    public final int hashCode()
        {
        // why the redirection to hashCodeFor()?  So we can also use this in Double2D.
        return hashCodeFor(x,y);
        }
        
    public final boolean equals(Object obj)
        {
        if (obj instanceof Int2D)
            {
            Int2D other = (Int2D) obj;
            return (other.x == x && other.y == y);
            }
        else if (obj instanceof Double2D)
            {
            Double2D other = (Double2D) obj;
            return (other.x == x && other.y == y);
            }
        else return false;
        }
    }
