package sim.util;

/*
Int3D stores three values (x, y, and z) but it is immutable: once the x and y and z values are set, they cannot be changed (they're final).  Unlike Double3D or Double2D or Int2D, Int3D has no mutable equivalent in the standard Java libraries.  Like the others, Int3D is immutable primarily to prevent hash tables from breaking.

<p>One day in the far future, Int3D should also be HIGHLY efficient; since it is immutable, it can be passed by value rather than by pointer by a smart compiler.  Not today, though.  But it's not bad.

<p>This class has an elaborate hash code generation that is much more random than Sun's standard generator, but takes more time.  For very large numbers of objects, this is a good idea, but we may change it to a simpler version in the future.

<p>Int3D.equals(...) can compare by value against other Int3Ds and Double3Ds.
*/
public final class Int3D implements java.io.Serializable
    {
    public final int x;
    public final int y;
    public final int z;
    
    public Int3D() { x = 0; y = 0; z = 0;}
    public Int3D(final int x, final int y, final int z) { this.x = x; this.y = y; this.z = z;}
    /** Explicitly assumes the z value is set to 0 */
    public Int3D(final Int2D p) { x = p.x; y = p.y; z = 0; }
    public Int3D(final Int2D p, final int z) { x = p.x; y = p.y; this.z = z; }
    public final int getX() { return x; }
    public final int getY() { return y; }
    public final int getZ() { return z; }
    public String toString() { return "Int3D["+x+","+y+","+z+"]"; }
    public String toCoordinates() { return "(" + x + ", " + y + ", " + z + ")"; }

    public final static int hashCodeFor(final int x, final int y, int z)
        {
	// basically we need to randomly disperse <int,int,int> --> int
	// I do that as x ^ hash(y ^ hash(z) + 17 [or whatever]). Hash function
	// taken from http://www.cris.com/~Ttwang/tech/inthash.htm
        
        z += ~(z << 15);
        z ^=  (z >>> 10);
        z +=  (z << 3);
        z ^=  (z >>> 6);
        z += ~(z << 11);
        z ^=  (z >>> 16);
        
        z ^= y;
        z+=17;
        
        z += ~(z << 15);
        z ^=  (z >>> 10);
        z +=  (z << 3);
        z ^=  (z >>> 6);
        z += ~(z << 11);
        z ^=  (z >>> 16);

	// nifty!  Now mix in x
	
        return x ^ z;
        }
        
    public int hashCode()
        {
        // why the redirection to hashCodeFor()?  So we can also use this in Double3D.
        return hashCodeFor(x,y,z);
        }
        
    public boolean equals(final Object obj)
        {
        if (obj instanceof Int3D)
            {
            Int3D other = (Int3D) obj;
            return (other.x == x && other.y == y && other.z == z);
            }
        else if (obj instanceof Double3D)
            {
            Double3D other = (Double3D) obj;
            return (other.x == x && other.y == y && other.z == z);
            }
        else return false;
        }
    }
