package sim.engine;
import ec.util.*;
import java.io.*;
import java.util.zip.*;

/** SimState represents the simulation proper.  Your simulations generally will contain one top-level object which subclasses from SimState.

<p>A SimState contains the random number generator and the simulator's schedule.  You should not change the schedule to another Schedule object.

<p>When a simulation is begun, SimState's start() method is called.  Then the schedule is stepped some N times.  Last, the SimState's finish() method is called, and the simulation is over.

<p>SimStates are serializable; if you wish to be able to checkpoint your simulation and read from checkpoints, you should endeavor to make all objects in the simulation serializable as well.  Prior to serializing to a checkpoint, preCheckpoint() is called.  Then after serialization, postCheckpoint() is called.  When a SimState is loaded from a checkpoint, awakeFromCheckpoint() is called to give you a chance to make any adjustments.  SimState also implements several methods which call these methods and then serialize the SimState to files and to streams.

<p>A number of the methods below have default empty implementations; but you should remember to <b>always</b> call super.foo() for any such method foo(), as later versions of this library may add features in the SimState's top-level methods.
*/

public class SimState implements java.io.Serializable
    {
    /** The SimState's random number generator */
    public MersenneTwisterFast random;
    
    /** SimState's schedule */
    public Schedule schedule;
    
    public SimState(MersenneTwisterFast random, Schedule schedule)
        {
        this.random = random;
        this.schedule = schedule;
        }
    
    public void setRandom(MersenneTwisterFast random)
        {
        this.random = random;
        }
    
    /** Called immediately prior to starting the simulation, or in-between
        simulation runs.  This gives you a chance to set up initially,
        or reset from the last simulation run. The default version simply
        replaces the Schedule with a completely new one.  */
    public void start()
        {
        schedule.reset();
        }
        
    /** Called either at the proper or a premature end to the simulation. 
        If the user quits the program, this function may not be called.  */
    public void finish()
        {
        }

    /** Called just before the SimState is being checkpointed (serialized out to a file to be
        unserialized and fired up at a future time).  You should override this to prepare 
        your SimState object appropriately. */
    public void preCheckpoint()
        {
        }
    
    /** Called just after the SimState was checkpointed (serialized out to a file to be
        unserialized and fired up at a future time).  You cam override this as you see fit. */
    public void postCheckpoint()
        {
        }

    /** Called after the SimState was created by reading from a checkpointed object.  You should
        set up your SimState in any way necessary (reestablishing file connections, etc.) to fix
        anything that may no longer exist. */
    public void awakeFromCheckpoint()
        {
        }

    /** Serializes out the SimState, and the entire simulation state (not including the graphical interfaces)
        to the provided stream. Calls preCheckpoint() before and postCheckpoint() afterwards.
        Throws an IOException if the stream becomes invalid (prematurely closes, etc.).  Does not close or flush
        the stream. */
    public void writeToCheckpoint(OutputStream stream)
        throws IOException
        {
        preCheckpoint();

        GZIPOutputStream g = 
            new GZIPOutputStream(
		new BufferedOutputStream(stream));

        ObjectOutputStream s = 
            new ObjectOutputStream(g);
            
        s.writeObject(this);
        s.flush();
        g.finish();  // need to force out the gzip stream AND manually flush it.  Java's annoying.  Took a while to find this bug...
        g.flush();
        postCheckpoint();
        }
    
    /** Writes the state to a checkpoint and returns the state.
        If an exception is raised, it is printed and null is returned. */
    public SimState writeToCheckpoint(File file)
        {
        try {
            FileOutputStream f = new FileOutputStream(file);
            writeToCheckpoint(f);
            f.close();
            return this;
            }
        catch(Exception e) { e.printStackTrace(); return null; }
        }
    
    /** Creates a SimState from checkpoint.  If an exception is raised, it is printed and null is returned. */
    public static SimState readFromCheckpoint(File file)
        {
        try {
            FileInputStream f = new FileInputStream(file);
            SimState state = readFromCheckpoint(f);
            f.close();
            return state;
            }
        catch(Exception e) { e.printStackTrace(); return null; }
        }

    /** Creates and returns a new SimState object read in from the provided stream.  Calls awakeFromCheckpoint().
        Throws an IOException if the stream becomes invalid (prematurely closes etc.).  Throws a ClassNotFoundException
        if a serialized object is not found in the CLASSPATH and thus cannot be created.  Throws an OptionalDataException
        if the stream is corrupted.  Throws a ClassCastException if the top-level object is not actually a SimState.
        Does not close or flush the stream. */
    public static SimState readFromCheckpoint(InputStream stream)
	throws IOException, ClassNotFoundException, OptionalDataException, ClassCastException
        {
	ObjectInputStream s = 
	    new ObjectInputStream(
		new GZIPInputStream (
		    new BufferedInputStream(stream)));
	SimState state = (SimState) (s.readObject());
        state.awakeFromCheckpoint();
        return state;
        }
    }

