package sim.field.grid;
import sim.util.IntBag;

/*
  Define basic neighborhood functions for 3D Grids.  The basic interface defines a width and a height
  (not all grids require a width and a height unless you're doing toroidal grids), and basic math for
  toroidal computation.
    
  <H3>Toroidal Computation</H3>
    
  <p>If you're using the Grid to define a toroidal (wrap-around) world, you can use the <b>tx</b>
  and <b>ty</b> and <b>tz</b> methods to simplify the math for you.  For example, to increment in the x direction,
  including wrap-around, you can do:  x = tx(x+1).
    
  <p>If you're sure that the values you'd pass into the toroidal functions would not wander off more than
  a grid dimension in either direction (height, width, length), you can use the slightly faster toroidal functions
  <b>stx</b> and <b>sty</b> and <b>stz</b> instead.  For example, to increment in the x direction,
  including wrap-around, you can do:  x = stx(x+1).  See the documentation on these functions for
  when they're appropriate to use.  Under most common situations, they're okay.

  <p>In HotSpot 1.4.1, stx, sty, and stz are inlined.  In Hotspot 1.3.1, they are not (they contain if-statements).

  <p>While this interface defines various methods common to many grids, you should endeavor not to call these grids casted into this interface: it's slow.  If you call the grids' methods directly by their class, their methods are almost certain to be inlined into your code, which is very fast.
*/

public interface Grid3D extends java.io.Serializable
    {
    /** Get the width */
    public int getWidth();
    
    /** Get the height */
    public int getHeight();
    
    /** Get the length */
    public int getLength();
    
    /** Toroidal x. Defined as: (x%width+width)%width  */
    public int tx(final int x);

    /** Toroidal y.  Defined as: (y%height+height)%height */
    public int ty(final int y);
    
    /** Toroidal z.  Defined as: (z%length+length)%length  */
    public int tz(final int z);

    /** Simple [and fast] toroidal x.  Use this if the values you'd pass in never stray
        beyond (-width ... width * 2) not inclusive.  It's a bit faster than the full
        toroidal computation as it uses if statements rather than two modulos.
        The following definition:<br>
        { if (x >= 0) { if (x < width) return x; return x - width; } return x + width; }
        ...produces the shortest code (28 bytes) and is inlined in Hotspot for 1.4.1. */
    public int stx(final int x);
    
    /** Simple [and fast] toroidal y.  Use this if the values you'd pass in never stray
        beyond (-height ... height * 2) not inclusive.  It's a bit faster than the full
        toroidal computation as it uses if statements rather than two modulos.
        The following definition:<br>
        { if (y >= 0) { if (y < height) return y ; return y - height; } return y + height; }
        ...produces the shortest code (28 bytes) and is inlined in Hotspot for 1.4.1. */
    public int sty(final int y);

    /** Simple [and fast] toroidal z.  Use this if the values you'd pass in never stray
        beyond (-length ... length * 2) not inclusive.  It's a bit faster than the full
        toroidal computation as it uses if statements rather than two modulos.
        The following definition:<br>
        { if (z >= 0) { if (z < length) return z ; return z - length; } return z + length; }
        ...produces the shortest code (28 bytes) and is inlined in Hotspot for 1.4.1. */
    public int stz(final int z);

    /*
     * Gets all neighbors of a location that satisfy max( abs(x-X) , abs(y-Y), abs(z-Z) ) <= d
     * Returns the x, y and z positions of the neighbors.
     */
    public void getNeighborsMaxDistance( final int x, final int y, final int z, final int dist, final boolean toroidal, IntBag xPos, IntBag yPos, IntBag zPos );

    /*
     * Gets all neighbors of a location that satisfy abs(x-X) + abs(y-Y) + abs(z-Z) <= d
     * Returns the x, y and z positions of the neighbors.
     */
    public void getNeighborsHamiltonianDistance( final int x, final int y, final int z, final int dist, final boolean toroidal, IntBag xPos, IntBag yPos, IntBag zPos );
    }

