package sim.field.grid;
import sim.util.*;

/**
   A wrapper for 2D arrays of Objects.

   <p>This object expects that the 2D arrays are rectangular.  You are encouraged to access the array
   directly.  The object
   implements all of the Grid2D interface.  See Grid2D for rules on how to properly implement toroidal
   or hexagonal grids.
    
   <p>The width and height of the object are provided to avoid having to say field[x].length, etc.
    
   <p>If you need arbitrary numbers of Objects to be able to occupy the same location in the grid, or if you have very few Objects and a very large grid, you should probably use SparseGrid3D instead.
*/

public class ObjectGrid2D extends AbstractGrid2D
    {
    public Object[/**x*/][/**y*/] field;
    
    public ObjectGrid2D (int xdim, int ydim)
        {
        width = xdim;
        height = ydim;
        field = new Object[xdim][ydim];
        }
    
    public ObjectGrid2D (int xdim, int ydim, Object initialValue)
        {
        this(xdim,ydim);
        setTo(initialValue);
        }
    
    public ObjectGrid2D (ObjectGrid2D values)
        {
        setTo(values);
        }
            
    public final void set(final int x, final int y, final Object val)
        {
        field[x][y] = val;
        }

    public final Object get(final int x, final int y)
        {
        return field[x][y];
        }

    public final ObjectGrid2D setTo(Object thisObj)
        {
        Object[] fieldx = null;
	final int width = this.width;
	final int height = this.height;
        for(int x=0;x<width;x++)
            {
            fieldx = field[x];
            for(int y = 0; y<height;y++)
                fieldx[y]=thisObj;
            }
        return this;
        }

    public final ObjectGrid2D setTo(final ObjectGrid2D values)
        {
        if (width != values.width || height != values.height)
            {
            final int width = this.width = values.width;
            final int height = this.height = values.height;
            Object[][] field = this.field = new Object[width][];
            Object[][] ofield = values.field;
            for(int x =0 ; x < width; x++)
                field[x] = (Object []) (ofield[x].clone());
            }
        else
            {
            Object[][] field = this.field;
            Object[][] ofield = values.field;
            for(int x =0 ; x < width; x++)
                System.arraycopy(ofield[x],0,field[x],0,height);
            }
        return this;
        }

    /*
     * Gets all neighbors of a location that satisfy max( abs(x-X) , abs(y-Y) ) <= d
     * Returns the neighbors and their x and y positions; xPos and yPos can be null if you don't care about the x and y positions.
     */
    public final void getNeighborsMaxDistance( final int x, final int y, final int dist, final boolean toroidal, Bag result, IntBag xPos, IntBag yPos )
	{
        if( xPos == null )
            xPos = new IntBag();
        if( yPos == null )
            yPos = new IntBag();

        getNeighborsMaxDistance( x, y, dist, toroidal, xPos, yPos );

        result.clear();
        for( int i = 0 ; i < xPos.numObjs ; i++ )
            result.add( field[xPos.objs[i]][yPos.objs[i]] );
	}

    /*
     * Gets all neighbors of a location that satisfy abs(x-X) + abs(y-Y) <= d
     * Returns the neighbors and their x and y positions; xPos and yPos can be null if you don't care about the x and y positions.
     */
    public final void getNeighborsHamiltonianDistance( final int x, final int y, final int dist, final boolean toroidal, Bag result, IntBag xPos, IntBag yPos )
	{
        if( xPos == null )
            xPos = new IntBag();
        if( yPos == null )
            yPos = new IntBag();

        getNeighborsHamiltonianDistance( x, y, dist, toroidal, xPos, yPos );

        result.clear();
        for( int i = 0 ; i < xPos.numObjs ; i++ )
            result.add( field[xPos.objs[i]][yPos.objs[i]] );
	}

    /*
     * Gets all neighbors of a location in hexagonal world
     * Returns the neighbors and their x and y positions; xPos and yPos can be null if you don't care about the x and y positions.
     */
    public final void getNeighborsHexagonalDistance( final int x, final int y, final int dist, final boolean toroidal, Bag result, IntBag xPos, IntBag yPos )
	{
        if( xPos == null )
            xPos = new IntBag();
        if( yPos == null )
            yPos = new IntBag();

        getNeighborsHexagonalDistance( x, y, dist, toroidal, xPos, yPos );

        result.clear();
        for( int i = 0 ; i < xPos.numObjs ; i++ )
            result.add( field[xPos.objs[i]][yPos.objs[i]] );
	}

    }
