package sim.field.grid;
import sim.field.*;
import sim.util.*;

/**
   A storage facility for sparse objects in discrete 3D space, using HashMaps.  SparseGrid3D differs from ObjectGrid3D
   in several respects:
    
   <ul>
   <li>SparseGrid3D can store more than one object at a location.  ObjectGrid3D cannot.
   <li>ObjectGrid3D can store an object at more than one location (though it's bad form!).
   <li>SparseGrid3D can efficiently (O(1)) tell you the location of an object.
   <li>SparseGrid3D can efficiently (O(#objs)) scan through all objects.  The best you can do with ObjectGrid3D is search its array (which might have many empty slots).
   <li>Storing an object, finding its location, or changing its location, in a SparseGrid3D is O(1) but requires several HashMap lookups and/or removes, which has a significant constant overhead.
   </ul>

   <p>Generally speaking, if you have a grid of objects, one per location, you should use an ObjectGrid3D.  If you have a large grid occupied by a few objects, or those objects can pile up on the same grid location, you should use a SparseGrid3D.
    
   <p>In either case, you might consider storing the location of an object IN THE OBJECT ITSELF if you need to query for the object location often -- it's faster than the hashtable lookup in SparseGrid3D, and certainly faster than searching the entire array of an ObjectGrid3D.
*/

public class SparseGrid3D extends SparseField
    {
    protected int width;
    protected int height;
    protected int length;
    
    public SparseGrid3D(int width, int height, int length)
        {
        this.width = width;
        this.height = height;
        this.length = length;
        }
        
    public final int getWidth() { return width; }
    
    public final int getHeight() { return height; }
    
    public final int getLength() { return length; }
    
    public final int tx(final int x) { return (x % width + width) % width; }

    public final int ty(final int y) { return (y % height + height) % height; }

    public final int tz(final int z) { return (z % length + length) % length; }

    public final int stx(final int x) 
        { if (x >= 0) { if (x < width) return x; return x - width; } return x + width; }

    public final int sty(final int y) 
        { if (y >= 0) { if (y < height) return y ; return y - height; } return y + height; }

    public final int stz(final int z) 
        { if (z >= 0) { if (z < length) return z ; return z - length; } return z + length; }

    public final Bag getObjectsAtLocation(final int x, final int y, final int z)
        {
        return getObjectsAtLocation(new Int3D(x,y,z));
        }

    public final Double3D getObjectLocationAsDouble3D(Object obj)
        {
        return new Double3D((Int3D) super.getRawObjectLocation(obj));
        }

    public Int3D getObjectLocation(Object obj)
        {
        return (Int3D) super.getRawObjectLocation(obj);
        }
    
    public final Bag removeObjectsAtLocation(final int x, final int y, final int z)
        {
        return removeObjectsAtLocation(new Int3D(x,y,z));
        }

    public Bag removeObjectsAtLocation(Int3D location)
        {
        return super.removeObjectsAtLocation(location);
        }
        
    public final boolean setObjectLocation(final Object obj, final int x, final int y, final int z)
        {
        return setObjectLocation(obj,new Int3D(x,y,z));
        
        }
    public boolean setObjectLocation(Object obj, final Int3D location)
        {
        return super.setObjectLocation(obj, location);
        }

    /*
     * Gets all neighbors of a location that satisfy max( abs(x-X) , abs(y-Y), abs(z-Z) ) <= d
     * Returns the x, y and z positions of the neighbors.
     */
    public final void getNeighborsMaxDistance( final int x, final int y, final int z, final int dist, final boolean toroidal, IntBag xPos, IntBag yPos, IntBag zPos )
	{
        // won't work for negative distances
        if( dist < 0 )
	    {
            throw new RuntimeException( "Runtime exception in method getNeighborsMaxDistance: Distance must be positive" );
	    }

        if( xPos == null || yPos == null || zPos == null )
	    {
            throw new RuntimeException( "Runtime exception in method getNeighborsMaxDistance: xPos and yPos should not be null" );
	    }

        xPos.clear();
        yPos.clear();
        zPos.clear();

        // for toroidal environments the code will be different because of wrapping arround
        if( toroidal )
	    {
            // compute xmin and xmax for the neighborhood
            for( int x0 = x-dist ; x0 <= x+dist ; x0++ )
		{
                final int x_0 = stx(x0);
                // compute ymin and ymax for the neighborhood
                for( int y0 = y-dist ; y0 <= y+dist ; y0++ )
		    {
                    final int y_0 = sty(y0);
                    for( int z0 = z-dist ; z0 <= z+dist ; z0++ )
			{
                        final int z_0 = stz(z0);
                        if( x_0 != x || y_0 != y || z_0 != z )
			    {
                            xPos.add( x_0 );
                            yPos.add( y_0 );
                            zPos.add( z_0 );
			    }
			}
		    }
		}
	    }
        else // not toroidal
	    {
            // compute xmin and xmax for the neighborhood such that they are within boundaries
            for( int x0 = ((x-dist>=0)?x-dist:0) ; x0 <= ((x+dist<=width-1)?x+dist:width-1) ; x0++ )
		{
                final int x_0 = x0;
                // compute ymin and ymax for the neighborhood such that they are within boundaries
                for( int y0 = ((y-dist>=0)?y-dist:0) ; y0 <= ((y+dist<=height-1)?y+dist:height-1) ; y0++ )
		    {
                    final int y_0 = y0;
                    for( int z0 = ((z-dist>=0)?z-dist:0) ; z0 <= ((z+dist<=length-1)?z+dist:length-1) ; z0++ )
			{
                        final int z_0 = z0;
                        if( x_0 != x || y_0 != y || z_0 != z )
			    {
                            xPos.add( x_0 );
                            yPos.add( y_0 );
                            zPos.add( z_0 );
			    }
			}
		    }
		}
	    }
	}


    /*
     * Gets all neighbors of a location that satisfy abs(x-X) + abs(y-Y) + abs(z-Z) <= d
     * Returns the x, y and z positions of the neighbors.
     */
    public final void getNeighborsHamiltonianDistance( final int x, final int y, final int z, final int dist, final boolean toroidal, IntBag xPos, IntBag yPos, IntBag zPos )
	{
        // won't work for negative distances
        if( dist < 0 )
	    {
            throw new RuntimeException( "Runtime exception in method getNeighborsHamiltonianDistance: Distance must be positive" );
	    }

        if( xPos == null || yPos == null || zPos == null )
	    {
            throw new RuntimeException( "Runtime exception in method getNeighborsHamiltonianDistance: xPos and yPos should not be null" );
	    }

        xPos.clear();
        yPos.clear();
        zPos.clear();

        // for toroidal environments the code will be different because of wrapping arround
        if( toroidal )
	    {
            // compute xmin and xmax for the neighborhood
            for( int x0 = x-dist ; x0 <= x+dist ; x0++ )
		{
                final int x_0 = stx(x0);
                // compute ymin and ymax for the neighborhood; they depend on the curreny x0 value
                for( int y0 = y-(dist-((x0-x>=0)?x0-x:x-x0)) ; y0 <= y+(dist-((x0-x>=0)?x0-x:x-x0)) ; y0++ )
		    {
                    final int y_0 = sty(y0);
                    for( int z0 = z-(dist-((x0-x>=0)?x0-x:x-x0)-((y0-y>=0)?y0-y:y-y0)) ;
                         z0 <= z+(dist-((x0-x>=0)?x0-x:x-x0)-((y0-y>=0)?y0-y:y-y0)) ;
                         z0++ )
			{
                        final int z_0 = stz(z0);
                        if( x_0 != x || y_0 != y || z_0 != z )
			    {
                            xPos.add( x_0 );
                            yPos.add( y_0 );
                            zPos.add( z_0 );
			    }
			}
		    }
		}
	    }
        else // not toroidal
	    {
            // compute xmin and xmax for the neighborhood such that they are within boundaries
            for( int x0 = ((x-dist>=0)?x-dist:0) ; x0 <= ((x+dist<=width-1)?x+dist:width-1) ; x0++ )
		{
                final int x_0 = x0;
                // compute ymin and ymax for the neighborhood such that they are within boundaries
                // they depend on the curreny x0 value
                for( int y0 = ((y-(dist-((x0-x>=0)?x0-x:x-x0))>=0)?y-(dist-((x0-x>=0)?x0-x:x-x0)):0) ;
                     y0 <= ((y+(dist-((x0-x>=0)?x0-x:x-x0))<=height-1)?y+(dist-((x0-x>=0)?x0-x:x-x0)):height-1) ;
                     y0++ )
		    {
                    final int y_0 = y0;
                    for( int z0 = ((z-(dist-((x0-x>=0)?x0-x:x-x0)-((y0-y>=0)?y0-y:y-y0))>=0)?z-(dist-((x0-x>=0)?x0-x:x-x0)-((y0-y>=0)?y0-y:y-y0)):0) ;
                         z0 <= ((z+(dist-((x0-x>=0)?x0-x:x-x0)-((y0-y>=0)?y0-y:y-y0))<=length-1)?z+(dist-((x0-x>=0)?x0-x:x-x0)-((y0-y>=0)?y0-y:y-y0)):length-1) ;
                         z0++ )
			{
                        final int z_0 = z0;
                        if( x_0 != x || y_0 != y || z_0 != z )
			    {
                            xPos.add( x_0 );
                            yPos.add( y_0 );
                            zPos.add( z_0 );
			    }
			}
		    }
		}
	    }
	}

    /*
     * Gets all neighbors of a location that satisfy max( abs(x-X) , abs(y-Y), abs(z-Z) ) <= d
     * Returns the neighbors and their x, y and z positions; xPos, yPos and zPos can be null if you don't care about the x, y and z positions.
     */
    public final void getNeighborsMaxDistance( final int x, final int y, final int z, final int dist, final boolean toroidal, Bag result, IntBag xPos, IntBag yPos, IntBag zPos )
	{
        if( xPos == null )
            xPos = new IntBag();
        if( yPos == null )
            yPos = new IntBag();
        if( zPos == null )
            zPos = new IntBag();

        getNeighborsMaxDistance( x, y, z, dist, toroidal, xPos, yPos, zPos );
        result = getObjectsAtLocations(xPos,yPos,zPos,result);
	}

    /*
     * Gets all neighbors of a location that satisfy abs(x-X) + abs(y-Y) + abs(z-Z) <= d
     * Returns the neighbors and their x, y and z positions; xPos, yPos and zPos can be null if you don't care about the x, y and z positions.
     */
    public final void getNeighborsHamiltonianDistance( final int x, final int y, final int z, final int dist, final boolean toroidal, Bag result, IntBag xPos, IntBag yPos, IntBag zPos )
	{
        if( xPos == null )
            xPos = new IntBag();
        if( yPos == null )
            yPos = new IntBag();
        if( zPos == null )
            zPos = new IntBag();

        getNeighborsHamiltonianDistance( x, y, z, dist, toroidal, xPos, yPos, zPos );
        result = getObjectsAtLocations(xPos,yPos,zPos,result);
	}

    /** For each <xPos,yPos,zPos> location, puts all such objects into the result bag.  Returns the result bag.
        If the provided result bag is null, one will be created and returned. */
    public Bag getObjectsAtLocations(final IntBag xPos, final IntBag yPos, final IntBag zPos, Bag result)
        {
        if (result==null) result = new Bag();
        else result.clear();
        
        final int len = xPos.numObjs;
        final int[] xs = xPos.objs;
        final int[] ys = yPos.objs;
        final int[] zs = zPos.objs;
        for(int i=0; i < len; i++)
            {
            // a little efficiency: add if we're 1, addAll if we're > 1, 
            // do nothing if we're 0
            Bag temp = getObjectsAtLocation(xs[i],ys[i],zs[i]);
            if (temp!=null)
                {
                int n = temp.numObjs;
                if (n==1) result.add(temp.objs[0]);
                else if (n > 1) result.addAll(temp);
                }
            }
        return result;
        }

    }


