package sim.util;
import java.util.*;
import java.lang.reflect.*;

// stars down the side maintain formatting
/**
 * A very simple class for getting and setting object properties.  You create this class by passing in the object you'd like to modify.  The only properties that are considered are ones which are simple (or boxed)
 *  booleans, bytes, shorts, ints, longs, floats, doubles, characters, or strings.  
 *  A property Foo exists in a class if there is a getFoo() or isFoo() method.  ReadWrite properties
 *  are ones for which there is ALSO a setFoo(prop) method.
 *
 *  <P>Alternatively, you can get a class like this by calling Properties.getProperties(...), which will return either a SimpleProperties or a CollectionProperties, depending on which is appropriate and the flags you have passed in.
 *
 *   <p>
 *  This class allows you to set and get properties on the object via boxing the property (java.lang.Integer
 *  for int, for example).  You can also pass in a String, and SimpleProperties will parse the appropriate
 *  value out of the string automatically without you having to bother checking the type.
 *
 *  <p>If any errors arise from generating the properties, setting them, or getting their values, then
 *  the error is printed to the console, but is not thrown: instead typically null is returned.
 *
 *  <p>If the object provided to SimpleProperties is sim.util.Proxiable, then SimpleProperties will call
 *  propertiesProxy() on that object to get the "true" object whose properties are to be inspected.  This
 *  makes it possible for objects to create filter objects which only permit access to certain properties.
 *  Generally speaking, such proxies (and indeed any object whose properties will be inspected) should be
 *  public, non-anonymous classes.  For example, imagine that you've got get/set methods on some property
 *  Foo, but you only want SimpleProperties to recognize the get method.  You can easily do this by making
 *  your class Proxiable and providing a proxy which only gets Foo, not sets it, as so:
 *  
 * <pre><tt>
 * public class MyClass extends Proxiable
 *     {        
 *     int foo;
 *    
 *     public int getFoo() { return foo; }
 *     public void setFoo(int val) { foo = val; }
 *    
 *     public class MyProxy
 *         {
 *	   public int getFoo() { return foo; }
 *	   }
 *   
 *     public Object propertiesProxy() { return new MyProxy(); }
 *     }
 * </tt></pre>
 *
 */

public class SimpleProperties extends Properties implements java.io.Serializable
    {
    Object object;
    
    ArrayList getMethods = new ArrayList();
    ArrayList setMethods = new ArrayList(); // if no setters, that corresponding spot will be null
    
    /** Gathers all properties for the object, including ones defined in superclasses. */
    public SimpleProperties(Object o) { this(o,true,true); }
    
    /** Gathers all properties for the object, possibly including ones defined in superclasses. 
        If includeGetClass is true, then the Class property will be included. 
    */
    public SimpleProperties(Object o, boolean includeSuperclasses, boolean includeGetClass)
        {
        object = o;
        if (o!=null && o instanceof sim.util.Proxiable)
            object = ((sim.util.Proxiable)(o)).propertiesProxy();
        generateProperties(includeSuperclasses,includeGetClass);
        }
    
    void generateProperties(boolean includeSuperclasses, boolean includeGetClass)
        {
        if (object != null) try
            {
            // generate the properties
            Class c = object.getClass();
            Method[] m = (includeSuperclasses ? c.getMethods() : c.getDeclaredMethods());
            for(int x = 0 ; x < m.length; x++)
                {
                if (m[x].getName().startsWith("get") || m[x].getName().startsWith("is")) // corrrect syntax?
                    {
                    int modifier = m[x].getModifiers();
                    if ((includeGetClass || !m[x].getName().equals("getClass")) &&
                        m[x].getParameterTypes().length == 0 &&
                        Modifier.isPublic(modifier)) // no arguments, and public, non-abstract?
                        {
                        //// Add all properties...
                        Class returnType = m[x].getReturnType();
                        if (returnType!= Void.TYPE)
                            {
                            getMethods.add(m[x]);
                            setMethods.add(getWriteProperty(m[x],c));
                            }
                        }
                    }
                }
            }
        catch (Exception e)
            {
            e.printStackTrace();
            }
        }
        
    Method getWriteProperty(Method m, Class c)
        {
        try
            {
            if (m.getName().startsWith("get"))
                {
                return c.getMethod("set" + (m.getName().substring(3)), new Class[] { m.getReturnType() });
                }
            else if (m.getName().startsWith("is"))
                {
                return c.getMethod("set" + (m.getName().substring(2)), new Class[] { m.getReturnType() });
                }
            else return null;
            }
        catch (Exception e)
            {
            return null;
            }
        }
    
    
    public boolean isVolatile() { return false; }

    /** Returns the number of properties discovered */
    public int numProperties()
        {
        return getMethods.size();
        }

    /** Returns the name of the given property.
        Returns null if the index is out of the range [0 ... numProperties() - 1 ]*/
    public String getName(int index)
        {
        if (index < 0 || index > numProperties()) return null;
        if (((Method)(getMethods.get(index))).getName().startsWith("is"))
            return ((Method)(getMethods.get(index))).getName().substring(2);
        else return ((Method)(getMethods.get(index))).getName().substring(3);
        }
        
    /** Returns whether or not the property can be written as well as read
        Returns false if the index is out of the range [0 ... numProperties() - 1 ]*/
    public boolean isReadWrite(int index)
        {
        if (index < 0 || index > numProperties()) return false;
        if (isComposite(index)) return false;
        return (setMethods.get(index)!=null);
        }

    /** Returns the return type of the property (see the TYPE_... values)
        Returns -1 if the index is out of the range [0 ... numProperties() - 1 ]*/
    public Class getType(int index)
        {
        if (index < 0 || index > numProperties()) return null;
        Class returnType = ((Method)(getMethods.get(index))).getReturnType();

        return getTypeConversion(returnType);
        }

    /** Returns the current value of the property.  Simple values (byte, int, etc.)
        are boxed (into Byte, Integer, etc.).
        Returns null if an error occurs or if the index is out of the range [0 ... numProperties() - 1 ]*/
    public Object getValue(int index)
        {
        if (index < 0 || index > numProperties()) return null;
        try
            {
            return ((Method)(getMethods.get(index))).invoke(object, new Object[0]);
            }
        catch (Exception e)
            {
            e.printStackTrace();
            return null;
            }
        }
    
    Object _setValue(int index, Object value)
        {
        try
            {
            if (setMethods.get(index) == null) return null;
            ((Method)(setMethods.get(index))).invoke(object, new Object[] { value });
            return getValue(index);
            }
        catch (Exception e)
            {
            e.printStackTrace();
            return null;
            }
        }
    }
