package sim.engine;
import sim.util.*;

/**
   Schedule defines a scheduling queue in which events can be scheduled to occur
   after some number of "ticks" of a clock which the schedule maintains.  The current
   tick is given by the <b>time()</b> method.  If the current tick is <tt>BEFORE_SIMULATION</tt> (0),
   then the schedule is set to the "time before time" (the schedule hasn't started running
   yet).  If the current tick is <tt>AFTER_SIMULATION</tt> (9218868437227405312), then the schedule has run
   out of time.  <tt>EPOCH</tt> (1) is defined as the first timestep after <tt>BEFORE_SIMULATION</tt> -- it's
   the first timestep for which you can legally schedule a value.

   <p>An event is defined as a <b>Steppable</b> object. You can schedule events to either 
   occur a single time or to occur repeatedly at some interval.  If the event occurs repeatedly,
   the schedule will provide you with a <b>Stoppable</b> object on which you can call <b>stop()</b>
   to cancel all future repeats of the event.

   <p>The schedule is pulsed by calling its <b>step(...)</b> method.  Each pulse, the schedule
   finds the next tick in which events are scheduled, moves ahead to that tick, and then calls
   all the events scheduled at that tick.    Multiple events may be scheduled for the same clock tick.
   Events at a tick are further subdivided and scheduled according to their <i>ordering</i>.
   You specify the number of orderings in the constructor, and can't change them thereafter.
   If you specify N orderings, then the ordering values are 0 ... N-1.
   Objects for scheduled for lower orderings in a given time tick will be executed before objects with
   higher orderings for the same time tick.  If objects are scheduled for the same time tick and
   have the same ordering value, their execution will be randomly ordered with respect to one another.

   <p>You may not schedule events to occur on or before <tt>BEFORE_SIMULATION</tt> (tick 0).  Likewise, you
   may not schedule events to occur on or after <tt>AFTER_SIMULATION</tt> (tick 9218868437227405312).  This gives you
   9,218,868,437,227,405,311 timesteps to play with; we hope that's sufficient resolution for your simulation.

   <p>Schedule is synchronized and threadsafe.  It's easy enough to de-synchronize the schedule by hand if you
   would like to.  In the worst case (schedule a single repeated steppable that does nothing), a synchronized
   schedule runs at 2/3 the speed of an unsynchronized schedule.  In a typical case (such as HeatBugs), 
   the difference between a synchronized and an unsynchronized schedule is less than 5% efficiency loss
   if it's visible at all.

   <p>You can get the number of times that step(...) has been called on the schedule by calling the getSteps() method.
   This value is incremented just as the Schedule exits its step(...) method and only if the method returned true.
   Additionally, you can get a string version of the current time with the getTimestamp(...) method.

   <h3>Using a Real-Valued Schedule</h3>
   Schedule also has rudimentary facilities for use in pseudo-"real-valued" timesteps rather than discrete ones.
   This means that you can submit and receive time values as doubles rather than as longs.  In this case, 
   BEFORE_SIMULATION = 0.0  and AFTER_SIMULATION = +infinity.  You schedule values in "real time" by converting
   your desired double-valued "times" to and from long ticks using toReals(...) and fromReals(...).  These will
   come up with equivalent long values for your double such that if d1 &lt; d2, then fromReals(d1) &lt; fromReals(d2).
   Understand that this is <b>not a cast</b>:  for example, Double.doubleToRawLongBits(9.0) == 4621256167635550208.
   Your only guarantee is that if you convert your timestamps toReals() and fromReals(), they'll be ordered
   in the same way that your real-valued "desired" timestamps are ordered.  Also note that NaN will always be
   an invalid timestamp.

   <p>You have to be careful using a real-valued schedule.  Notably, you should <b>not use any variant of 
   scheduleRepeating(...)</b> as they will not work correctly.   To schedule events to occur repeatedly, the events
   will have to reschedule themselves.  Further, take care to note that real values <b>lose resolution</b>
   as time goes by.  Thus while you can distinguish between 0.000001 and 0.000002, you cannot distinguish between
   C+0.000001 and C+0.000002, where C is some large constant.  This mostly will bite you if you schedule for
   a real-valued event at time T+1.0, but it turns out that T+1.0 rounds to T (the current time) in the schedule.
   In this case the event will not be scheduled.  So be careful to check to make certain that
   your schedule(...) calls always return true.
*/
    

public class Schedule implements java.io.Serializable
    {
    public static final long BEFORE_SIMULATION = 0L;
    public static final long EPOCH = BEFORE_SIMULATION + 1L;
    public static final long AFTER_SIMULATION = 9218868437227405312L;

    protected Heap[] queue;
    protected Steppable[][] next;
    protected int[] numNext;
    protected Steppable[][] substeps;
    protected int[] numSubsteps;
    
    // the time
    protected long time;
    
    // the number of times step() has been called on m
    protected long steps;
    
    // resets the queues by replacing them, NOT reusing them.  This allows us to 
    // work properly in our schedule if the user resets the queues from within a Steppable;
    // see the comments in step()
    protected void resetQueues(final int numOrders)
        {
        queue = new Heap[numOrders];
        substeps = new Steppable[numOrders][11];
        numSubsteps = new int[numOrders];
        next = new Steppable[numOrders][11];
        numNext = new int[numOrders];
        for(int x=0;x<queue.length;x++) queue[x] = new Heap(); 
        }
    
    public Schedule(final int numOrders)
        {
        resetQueues(numOrders);
        time = BEFORE_SIMULATION;
        steps = 0;
        }
    
    /** Creates a Schedule with a single order */
    public Schedule()
        {
        this(1);
        }
    
    public synchronized long time() { return time; }
    
    /** Returns the current time in string format. If the time is BEFORE_SIMULATION, then beforeSimulationString is
        returned.  If the time is AFTER_SIMULATION, then afterSimulationString is returned.  Otherwise a numerical
        representation of the time is returned. */
    public synchronized String getTimestamp(final String beforeSimulationString, final String afterSimulationString)
        {
        long time = time();
        if (time <= BEFORE_SIMULATION) return "";
        if (time >= AFTER_SIMULATION) return "Done";
        return Long.toString(time); 
        }
    
    public synchronized long getSteps() { return steps; }
    
    /** Converts the long tick value into a reals "time" value (as a double) -- this is NOT a cast.
        If you use doubles, you should ONLY use
        doubles -- do not mix them with longs.  See the warning in the class documentation.
        We can just convert bits to bits because
        the half-open long range (0 ... 9218868437227405312L] is equivalent to the
        half-open double range (0 ... +infinity).  NaN values exist outside this range, so they're
        not a problem. */
    public final double toReals(final long time) { return Double.longBitsToDouble(time); }
    
    /** Converts the reals "time" value into a long tick value -- this is NOT a cast.  If you use doubles, you should ONLY use
        doubles -- do not mix them with longs.  See the warning in the class documentation.
        We can just convert bits to bits because
        the half-open long range (0 ... 9218868437227405312L] is equivalent to the
        half-open double range (0 ... +infinity).  NaN values exist outside this range, so they're
        not a problem. */
    public final long fromReals(final double time) { return Double.doubleToRawLongBits(time); }
    
    // roughly doubles the array size, retaining the existing elements
    protected Steppable[] increaseSubsteps(final Steppable[] substeps)
        {
        return increaseSubsteps(substeps,substeps.length*2+1);
        }
        
    // increases substeps to n length -- which had better be bigger than substeps.length!
    protected Steppable[] increaseSubsteps(final Steppable[] substeps, final int n)
        {
        Steppable[] newsubstep = new Steppable[n];
        System.arraycopy(substeps,0,newsubstep,0,substeps.length);
        return newsubstep;
        }
    
    /** Empties out the schedule and resets it to a pristine state BEFORE_SIMULATION, with steps = 0.*/
    public synchronized void reset()
        {
        resetQueues(queue.length);        // make new queues
        time = BEFORE_SIMULATION;
        steps = 0;
        }
        
    /** Returns true if the schedule has nothing left to do. */
    public synchronized boolean scheduleComplete()
        {
        return _scheduleComplete();
        }
    
    protected boolean _scheduleComplete()
        {
        for(int x=0;x<queue.length;x++)
            if (numNext[x] > 0 || !queue[x].isEmpty())
                return false;
        return true;
        }
    
    /** Steps the schedule, gathering and ordering all the items to step on the next time step (skipping
        blank time steps), and then stepping all of them in the decided order.  
        Returns FALSE if nothing was stepped -- the schedule is exhausted or time has run out. */
    
    public synchronized boolean step(final SimState state)
        {
        final int[] _numSubsteps;
        final Steppable[][] _substeps;
        
        if (time==AFTER_SIMULATION) 
            return false;
        
        long t = AFTER_SIMULATION;
    
        // store the queue substep information here even if the user deletes it
        // from within a step(), so we can continue forward anyway
        _numSubsteps = numSubsteps;
        _substeps = substeps;  

        if (!_scheduleComplete())
            {
            long content;
            
            for(int x=0;x<numNext.length;x++)
                if (numNext[x] > 0)  { t = time + 1L; break; }
            if (t == AFTER_SIMULATION)  // if we've not already determined the minimum...
                for(int x=0;x<queue.length;x++)
                    if( !queue[x].isEmpty() )
                        {
                        content = queue[x].getMinKey();
                        if (t > content) t = content;
                        }
        
            if (t==AFTER_SIMULATION) 
                return false; 
        
            time = t;
            
            // Extract the contents
            // slightly less efficient than previous version (extraneous call
            // to min(), oh well, much simpler to understand)
            for(int x=0;x<queue.length;x++)
                {
                _numSubsteps[x] = 0;
                if (numNext[x] > 0)
                    {
                    int len = _numSubsteps[x] + numNext[x];
                    if (len > _substeps[x].length)
                        _substeps[x] = increaseSubsteps(_substeps[x],len*2+1);  // more than twice needed
                    System.arraycopy(next[x],0,_substeps[x], _numSubsteps[x],numNext[x]);
                    // we're not zeroing array right now, so stuff can't get GC'd.  :-(
                    _numSubsteps[x] += numNext[x];
                    numNext[x] = 0;
                    }
                
                while ( (!queue[x].isEmpty()) && ((content=queue[x].getMinKey())==time) )
                    {
                    if (_numSubsteps[x] == _substeps[x].length)
                        _substeps[x] = increaseSubsteps(_substeps[x]);
                    _substeps[x][_numSubsteps[x]++] = (Steppable)(queue[x].extractMin());
                    }

                // shuffle the queue _substeps
                Steppable temp;
                final Steppable[] s = _substeps[x];
                int len = _numSubsteps[x]-1;
                for(int z=len; z>0 ;z--)
                    {
                    int i = state.random.nextInt(z+1);
                    temp = s[i];
                    s[i] = s[z];
                    s[z] = temp;
                    }
                }
            
            // execute the content -- we use the retained private
            // local variables defined above just in case the user blows away
            // the substep variables with a resetQueues().
            for(int x=0;x< _substeps.length;x++)
                {
                final int l = _numSubsteps[x];
                for(int z=0;z<l;z++)
                    {
                    if (_substeps[x][z]!=null)
                        {
                        _substeps[x][z].step(state);
                        _substeps[x][z] = null;  // let GC
                        }
                    } 
                }
            }
        else
            {
            time = AFTER_SIMULATION;
            return false;
            }
        steps++;
        return true;
        }
        
    /** Schedules the event to occur at the next clock tick, 0 ordering. If this is a valid time
        and event, schedules the event and returns TRUE, else returns FALSE.  */
    
    // synchronized so getting the time can be atomic with the subsidiary scheduleOnce function call
    public final synchronized boolean scheduleOnce(final Steppable event)
        {
        return scheduleOnce(time+1L,0,event);
        }
        
    /** If the current time is less than the time requested, 0 ordering, and the event is valid,
        schedules the event to occur at the requested time and returns TRUE.  Else returns FALSE. */
    
    public final boolean scheduleOnce(final long time, final Steppable event)
        {
        return scheduleOnce(time,0,event);
        }
        
    /** If the current time is less than the time requested, and the event is valid,
        schedules the event to occur
        at the requested time, placed in the 'first' queue, and returns TRUE.  Else returns FALSE.  */
    
    public synchronized boolean scheduleOnce(final long time, final int ordering, final Steppable event)
        {
        if (this.time < time && time < AFTER_SIMULATION && event !=null &&
            ordering < queue.length && ordering >= 0)
            {
            if (time == this.time + 1)  // it's the very next step
                {
                if (numNext[ordering] == next[ordering].length)  // increase next steps
                    next[ordering] = increaseSubsteps(next[ordering]);
                next[ordering][numNext[ordering]] = event;
                numNext[ordering]++;
                }
            else queue[ordering].add(event,time);
            return true;
            }
        else return false;
        }
    
    /** If the event is valid, and the current time is valid, schedules the event to occur once
        every clock tick starting at the next clock tick, 0 ordering, and 
        returns TRUE, else returns FALSE.   Returns a Stoppable which you can use to stop the
        repeated scheduling from occuring any more.
        
        <p> Note that calling stop() on the Stoppable
        will not only stop the repeating, but will <i>also</i> make the Schedule completely
        forget (lose the pointer to) the Steppable scheduled here.  This is particularly useful
        if you need to make the Schedule NOT serialize certain Steppable objects.  <b>Do not use this
        with a real-valued schedule.</b> */
    
    // synchronized so getting the time can be atomic with the subsidiary scheduleRepeating function call
    public final synchronized Stoppable scheduleRepeating(final Steppable event)
        {
        return scheduleRepeating(time+1L,0,event,1L);
        }

    /** If the event is valid, and the current time is less than time, schedules the
        event to occur once every clock tick starting at time, 0 ordering, and 
        returns TRUE, else returns FALSE. Returns a Stoppable which you can use to stop the
        repeated scheduling from occuring any more.
        
        <p> Note that calling stop() on the Stoppable
        will not only stop the repeating, but will <i>also</i> make the Schedule completely
        forget (lose the pointer to) the Steppable scheduled here.  This is particularly useful
        if you need to make the Schedule NOT serialize certain Steppable objects.    <b>Do not use this
        with a real-valued schedule.</b> */

    public final Stoppable scheduleRepeating(final long time, final Steppable event)
        {
        return scheduleRepeating(time,0,event,1L);
        }

    /** If the event is valid, and the current time is valid, schedules the event to occur once
        every interval's worth of clock ticks starting at the next clock tick, 0 ordering, and 
        returns TRUE, else returns FALSE. Returns a Stoppable which you can use to stop the
        repeated scheduling from occuring any more.
        
        <p> Note that calling stop() on the Stoppable
        will not only stop the repeating, but will <i>also</i> make the Schedule completely
        forget (lose the pointer to) the Steppable scheduled here.  This is particularly useful
        if you need to make the Schedule NOT serialize certain Steppable objects.    <b>Do not use this
        with a real-valued schedule.</b> */

    // synchronized so getting the time can be atomic with the subsidiary scheduleRepeating function call
    public final synchronized Stoppable scheduleRepeating(final Steppable event, final long interval)
        {
        return scheduleRepeating(time,0,event,interval);
        }

    /** If the event is valid, and the current time is less than time, schedules the
        event to occur once every interval's worth of clock ticks starting at time, 0 ordering, and 
        returns TRUE, else returns FALSE. Returns a Stoppable which you can use to stop the
        repeated scheduling from occuring any more.
        
        <p> Note that calling stop() on the Stoppable
        will not only stop the repeating, but will <i>also</i> make the Schedule completely
        forget (lose the pointer to) the Steppable scheduled here.  This is particularly useful
        if you need to make the Schedule NOT serialize certain Steppable objects.    <b>Do not use this
        with a real-valued schedule.</b> */

    public final Stoppable scheduleRepeating(final long time, final Steppable event, final long interval)
        {
        return scheduleRepeating(time,0,event,interval);
        }

    /** If the event is valid, and the current time is less than time, schedules the
        event to occur once every interval's worth of clock ticks starting at time, with the given ordering, and 
        returns TRUE, else returns FALSE. Returns a Stoppable which you can use to stop the
        repeated scheduling from occuring any more.
        
        <p> Note that calling stop() on the Stoppable
        will not only stop the repeating, but will <i>also</i> make the Schedule completely
        forget (lose the pointer to) the Steppable scheduled here.  This is particularly useful
        if you need to make the Schedule NOT serialize certain Steppable objects.    <b>Do not use this
        with a real-valued schedule.</b> */

    public Stoppable scheduleRepeating(final long time, final int ordering, final Steppable event, final long interval)
        {
        if (event==null) return null;
        Repeat r = new Repeat(event,interval,ordering);
        if (scheduleOnce(time,ordering,r)) return r; 
        else return null;
        }
    
    /**
       Handles repeated steps.  This is done by wrapping the Steppable with a Repeat object
       which is itself Steppable, and on its step calls its subsidiary Steppable, then reschedules
       itself.  Repeat is stopped by setting its subsidiary to null, and so the next time it's
       scheduled it won't reschedule itself (or call the subsidiary).   A private class for
       Schedule.  */
    
    class Repeat implements Steppable, Stoppable
        {
        protected long interval;
        protected Steppable step;  // if null, does not reschedule
        protected int ordering;
        
        public Repeat(final Steppable step, final long interval, final int ordering)
            {
            this.step = step;
            this.interval = interval;
            this.ordering = ordering;
            }
        
        public synchronized void step(final SimState state)
            {
            if (step!=null)
                {
                // this occurs WITHIN the schedule's synchronized step, so time()
                // and scheduleOnce() will both occur together without the time
                // changing
                scheduleOnce(time()+interval,ordering,this);
                step.step(state);
                }
            }
        
        public synchronized void stop()  
            {
            step = null;
            }

        // explicitly state a UID in order to be 'cross-platform' serializable 
        // because we ARE an inner class and compilers come up with all sorts
        // of different UIDs for inner classes and their parents.
        static final long serialVersionUID = -7903946075763886169L;
        }
        
    // explicitly state a UID in order to be 'cross-platform' serializable
    // because we contain an inner class and compilers come up with all
    // sorts of different UIDs for inner classes and their parents.
    static final long serialVersionUID = 2562838695289414534L;
    }
    


