package sim.util.gui;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.border.*;
import java.awt.*;

/**
   A simple class that lets you specify a label and validate a string.  PropertyField lets you control the values which the user sets by subclassing the class and overriding the newValue(val) method filters all newly user-set values and "corrects" them.  Programmatically set values (by calling setValue(...)) are not filtered through newValue by default.  If you need to filter, you should do setValue(newValue(val));

   <p>You can optionally specify a type for the property to be set.  At present the only value this type can take on is boolean (Boolean.TYPE).  If boolean, then the PropertyField will display a checkbox rather than a text field for the user to enter.  Additionally, if the PropertyField constructed read-only, then a JLabel (rather than text field) or a disabled checkbox will be shown.

   <p>PropertyField can also be provided with an optional label.
*/

public class PropertyField extends JComponent
    {
    public JTextField valField = new JTextField();
    public JCheckBox checkField = new JCheckBox();
    public JButton viewButton = new JButton("View");  // optionally displayed instead of valField (array or Object)
    public JLabel viewLabel = new JLabel();
    public JLabel optionalLabel = new JLabel();
    public JPanel switcher = new JPanel();
    public Border valFieldBorder;
    public Border emptyBorder;
    public String currentValue;
    public boolean isReadWrite;
    
    public int displayState;
    public static final int SHOW_CHECKBOX = 0;
    public static final int SHOW_TEXTFIELD = 1;
    public static final int SHOW_VIEWBUTTON = 2;
    static final String[] stateName = new String[] { "checkbox", "textfield", "viewbutton", "label" };

    public Color defaultColor;
    public Color editedColor = new Color(225,225,255);
    public void setEditedColor(Color c) { editedColor = c; }
    public Color getEditedColor() { return editedColor; }

    public KeyListener listener = new KeyListener()
        {
        public void keyReleased(KeyEvent keyEvent) { }
        public void keyTyped(KeyEvent keyEvent) { }
        public void keyPressed(KeyEvent keyEvent) 
            {
            if (keyEvent.getKeyCode() == KeyEvent.VK_ENTER)
                {
                setValue(newValue( valField.getText() ));
                }
            else if (keyEvent.getKeyCode() == KeyEvent.VK_ESCAPE)  // reset
                {
                setValue(getValue());
                }
            else
                {
                valField.setBackground(editedColor);
                }
            }
        };
        
    /*
      public ChangeListener checkListener = new ChangeListener()
      {
      public void stateChanged (ChangeEvent e)
      {
      setValue(newValue( new Boolean(checkField.isSelected()).toString() ));
      }
      };
    */    
    public ActionListener checkListener = new ActionListener()
        {
        public void actionPerformed ( ActionEvent e )
            {
            setValue(newValue( new Boolean(checkField.isSelected()).toString() ));
            }
        };

    public ActionListener viewButtonListener = new ActionListener()
        {
        public void actionPerformed ( ActionEvent e )
            {
            viewProperty();
            }
        };
    
    /** Sets the value, filtering it through newValue(val) first. */
    public void setValue(String val)
        {
        if(val!=null)
            if(val.equals("true"))
                checkField.setSelected(true);
            else
                checkField.setSelected(false);
            
        valField.setText(""+val);
        viewLabel.setText(""+val);
        currentValue = val;
        valField.setBackground(isReadWrite ? defaultColor : checkField.getBackground());
        }

    /** Returns the most recently set value. */
    public String getValue()
        {
        return currentValue;
        }
    
    public PropertyField()
        {
        this(null,"",true);
        }
        
    /** This looks just like a JTextField */
    public PropertyField(String initialValue)
        {
        this(null,initialValue,true);
        }
    
    /** if isReadWrite is false, then this looks just like a JLabel; else it looks just like a JTextField*/
    public PropertyField(String initialValue, boolean isReadWrite)
        {
        this(null,initialValue,isReadWrite);
        }
    
    public PropertyField(String label, String initialValue)
        {
        this(label,initialValue,true);
        }

    public PropertyField(String label, String initialValue, boolean isReadWrite)
        {
        this(label,initialValue,isReadWrite, SHOW_TEXTFIELD);
        }
    
    /** If show is SHOW_CHECKBOX, a checkbox will be shown (expecting "true" and "false" string values).
        If show is SHOW_VIEWBUTTON, a view button will be shown (expecting a true object).
        If show is SHOW_TEXTFIELD, a textfield will be shown. */
    public PropertyField(String label, String initialValue, boolean isReadWrite, int show)
        {
        // create object
        setLayout(new BorderLayout());
        add(optionalLabel,BorderLayout.WEST);
        add(switcher,BorderLayout.CENTER);
        switcher.setLayout(new CardLayout());
        switcher.add(valField, stateName[SHOW_TEXTFIELD]);
        switcher.add(checkField, stateName[SHOW_CHECKBOX]);
        JPanel p = new JPanel();
        p.setLayout(new BorderLayout());
        p.add(viewButton, BorderLayout.EAST);
        p.add(viewLabel, BorderLayout.CENTER);
        switcher.add(p, stateName[SHOW_VIEWBUTTON]);
        
        valFieldBorder = valField.getBorder();
        Insets i = valFieldBorder.getBorderInsets(valField);
        emptyBorder = new EmptyBorder(i.top,i.left,i.bottom,i.right);
        
        defaultColor = valField.getBackground();
        valField.addKeyListener(listener);
        checkField.addActionListener(checkListener);
        viewButton.addActionListener(viewButtonListener);

        // set values
        setValues(label, initialValue, isReadWrite, show);
        }

    public void setValues(String label, String initialValue, boolean isReadWrite, int show)
        {
        currentValue = initialValue;
        optionalLabel.setText(label);
        
        checkField.setEnabled(isReadWrite);
        valField.setEditable(isReadWrite);
        valField.setBorder(isReadWrite? valFieldBorder : emptyBorder);
        
        this.isReadWrite = isReadWrite;
        setValue(currentValue);
        ((CardLayout)(switcher.getLayout())).show(switcher,stateName[show]);
        }
        
    /** Override this to be informed when a new value has been set.
        The return value should be the value you want the display to show 
        instead. */
    public String newValue(String newValue)
        {
        return newValue;
        }

    /** Override this to be informed when a property is to be viewed in its
        own inspector because the user pressed the "view" button. */
    public void viewProperty()
        {
        //          System.out.println("you're still in PropertyField you dork!");
        }

    public void setToolTipText(String text)
        {
        super.setToolTipText(text);
        valField.setToolTipText(text);
        checkField.setToolTipText(text);
        optionalLabel.setToolTipText(text);
        viewButton.setToolTipText(text);
        viewLabel.setToolTipText(text);
        }
        
    public Dimension getMinimumSize()
        {
        Dimension s = super.getMinimumSize();
        s.height = valField.getMinimumSize().height;
        return s;
        }
    public Dimension getPreferredSize()
        {
        Dimension s = super.getPreferredSize();
        s.height = valField.getPreferredSize().height;
        return s;
        }
    }
