package sim.app.tutorial5;
import sim.engine.*;
import sim.field.continuous.*;
import sim.field.network.*;
import sim.util.*;
import ec.util.*;

public class Tutorial5 extends SimState
    {
    public Continuous2D balls;
    public NetworkField bands;

    public int numBalls = 50;
    public int numBands = 60;
    
    public final static double maxMass = 10.0;
    public final static double minMass = 1.0;
    public final static double minLaxBandDistance = 10.0;
    public final static double maxLaxBandDistance = 50.0;
    public final static double minBandStrength = 5.0;
    public final static double maxBandStrength = 10.0;
    public final static double collisionDistance = 5.0;
    
    public int getNumBalls() { return numBalls; }
    public void setNumBalls(int val) { if (val >= 2 ) numBalls = val; }
    public int getNumBands() { return numBands; }
    public void setNumBands(int val) { if (val >= 0 ) numBands = val; }

    public Tutorial5(long seed)
        {
        super(new MersenneTwisterFast(seed), new Schedule(2));
        balls = new Continuous2D(collisionDistance,100,100);
        bands = new NetworkField();
        }

    public void start()
        {
        super.start();

        balls = new Continuous2D(collisionDistance,100,100);
        bands = new NetworkField();
        
        Steppable[] s = new Steppable[numBalls];
        
        // make the balls
        for(int i=0; i<numBalls;i++)
            {
            // must be final to be used in the anonymous class below
            final Ball ball = new Ball(0,0,random.nextDouble() * (maxMass-minMass) + minMass);
            balls.setObjectLocation(ball,
                                    new Double2D(random.nextDouble() * 100,
                                                 random.nextDouble() * 100));
            bands.addNode(ball);
            schedule.scheduleRepeating(ball);
            
            // schedule the balls to compute their force after everyone's moved
            s[i] = new Steppable()
                {
                public void step(SimState state) { ball.computeForce(state); }
                // see Tutorial 3 for why this is helpful
                static final long serialVersionUID = -4269174171145445918L;
                };
            }
            
        // add the sequence
        schedule.scheduleRepeating(Schedule.EPOCH,1,new Sequence(s),1);
        
        // make the bands
        Bag ballObjs = balls.getAllObjects();
        for(int i=0;i<numBands;i++)
            {
            Band band = new Band(random.nextDouble() * 
                                 (maxLaxBandDistance - minLaxBandDistance) + minLaxBandDistance,
                                 random.nextDouble() *
                                 (maxBandStrength - minBandStrength) + minBandStrength);
            Ball from;
            from = (Ball)(ballObjs.objs[random.nextInt(ballObjs.numObjs)]);

            Ball to = from;
            while(to == from)
                to = (Ball)(ballObjs.objs[random.nextInt(ballObjs.numObjs)]);
            
            bands.addEdge(from,to,band);
            }
        }

    public static void main(String[] args)
        {
        Tutorial5 tutorial5 = null;
        
        // should we load from checkpoint?  I wrote this little chunk of code to
        // check for this to give you the general idea.
        
        for(int x=0;x<args.length-1;x++)  // "-checkpoint" can't be the last string
            if (args[x].equals("-checkpoint"))
                {
                SimState state = SimState.readFromCheckpoint(new java.io.File(args[x+1]));
                if (state == null)   // there was an error -- it got printed out to the screen, so just quit
                    System.exit(1);
                else if (!(state instanceof Tutorial5))  // uh oh, wrong simulation stored in the file!
                    {
                    System.out.println("Checkpoint contains some other simulation: " + state);
                    System.exit(1);
                    }
                else // we're ready to lock and load!  
                    tutorial5 = (Tutorial5)state;
                }
        
        // ...or should we start fresh?
        if (tutorial5==null)  // no checkpoint file requested
            {
            tutorial5 = new Tutorial5(System.currentTimeMillis());
            tutorial5.start();
            }
        
        long time;
        while((time = tutorial5.schedule.time()) < 10000)
            {
            if (time % 10 == 0) System.out.println(time);
            if (!tutorial5.schedule.step(tutorial5))
                break;

            // checkpoint
            if (time%500==0 && time!=0)
                {
                String s = "tutorial5." + time + ".checkpoint";
                System.out.println("Checkpointing to file: " + s);
                tutorial5.writeToCheckpoint(new java.io.File(s));
                }
            }
            
        tutorial5.finish();  // we don't use this, but it's good style
        }
        
    // see Tutorial 3 for why this is helpful
    static final long serialVersionUID = -7164072518609011190L;
    }
