package sim.app.heatbugs;
import sim.engine.*;
import sim.field.grid.*;

/** This agent decreases evaporates and diffuses all the heat at each time step. */

public /*strictfp*/ class Diffuser implements Steppable
    {
    public void step(SimState state)
        {
        HeatBugs heatbugs = (HeatBugs)state;
        
        // this method is where HeatBugs spends 90% of its time, so it's worthwhile
        // optimizing the daylights out of it.  By using the stx() and sty() methods
        // rather than just computing toroidal values ourselves, we lose about
        // 5% in speed.  I don't know if that's overhead having to compile the methods
        // (they get inlined) or if it's due to bad inlining -- I suspect the latter.
        // But the stx() and sty() methods are quite useful compared to the explicit
        // computation, so I'm not willing to hard-inline those myself.
        // Anyway, we can gain all of it back by explicitly pointing to valgrid etc. 
        // with constant locals here, rather than instance variables.  Besides, 
        // locals are a bit faster than instance variables in outer scopes anyway.  So
        // here are five local variables which get us our 5% back and then some.  It's
        // about the most optimization I'm willing to do before we get obtuse.  Could
        // probably squeeze another 5% out before we had to go to linearizing the
        // arrays (which would get us another 25% but require major, unacceptable
        // architectural changes).
        
        // locals are faster than instance variables
        final DoubleGrid2D _valgrid = heatbugs.valgrid;
        final double[][] _valgrid_field = heatbugs.valgrid.field;
        final double[][] _valgrid2_field = heatbugs.valgrid2.field;
        final int _gridWidth = _valgrid.getWidth();
        final int _gridHeight = _valgrid.getHeight();
        
        double average;
        
        // for each x and y position
        for(int x=0;x< _gridWidth;x++)
            for(int y=0;y< _gridHeight;y++)
                {
                average = 0.0;
                // for each neighbor of that position
                for(int dx=-1; dx< 2; dx++)
                    for(int dy=-1; dy<2; dy++)
                        {
                        // compute the toroidal <x,y> position of the neighbor
                        int xx = _valgrid.stx(x+dx);
                        int yy = _valgrid.sty(y+dy);
                                                        
                        // compute average
                        average += _valgrid_field[xx][yy];
                        }
                average /= 9.0;
                
                // load the new value into HeatBugs.this.valgrid2
                _valgrid2_field[x][y] = heatbugs.evaporationRate * 
                    (_valgrid_field[x][y] + heatbugs.diffusionRate * 
                     (average - _valgrid_field[x][y]));
                }
        
        // copy HeatBugs.this.valgrid2 to HeatBugs.this.valgrid
        _valgrid.setTo(heatbugs.valgrid2);
        }
    }

