package sim.app.heatbugs;
import sim.engine.*;
import sim.field.grid.*;

/** A dual-threaded version of Diffuser for use on machines with two processors.
    Holds a ParallelSequence which in turn holds two dummy Steppables which each
    call diffuse(...) with different start and end values.  That way basically we
    split the array diffusion up among two processors, one taking the first half
    of the array and one taking up the second half.  Because Diffuser takes up
    nearly all our time, this results in a dramatic increase in speed on a
    dual-processor machine. */

public /*strictfp*/ class ThreadedDiffuser implements Steppable
    {
    public ParallelSequence diffusers;
        
    public ThreadedDiffuser()
        {
        diffusers = new ParallelSequence(new Steppable[]
            {
            new Steppable ()
                { 
                public void step(SimState state) 
                    {
                    // diffuse top half of field
                    HeatBugs heatbugs = (HeatBugs)state;
                    int _gridWidth = heatbugs.valgrid.getWidth();  // read-only, so threadsafe with other one
                    diffuse(heatbugs, 0, _gridWidth/2);
                    }
                },
            new Steppable ()
                {
                public void step(SimState state) 
                    {
                    // diffuse bottom half of field
                    HeatBugs heatbugs = (HeatBugs)state;
                    int _gridWidth = heatbugs.valgrid.getWidth();  // read-only, so threadsafe with other one
                    diffuse(heatbugs, _gridWidth/2, _gridWidth);
                    }
                }
            });
        }
        
        
    public void step(SimState state)
        {
        diffusers.step(state);
                
        // copy HeatBugs.this.valgrid2 to HeatBugs.this.valgrid
        HeatBugs heatbugs = (HeatBugs)state;
        heatbugs.valgrid.setTo(heatbugs.valgrid2);
        }
        
        
    /** Diffuse heatbugs.valgrid.field[start...end] not including end */
        
    void diffuse(HeatBugs heatbugs, int start, int end)
        {
        // this method is where HeatBugs spends 90% of its time, so it's worthwhile
        // optimizing the daylights out of it.  By using the stx() and sty() methods
        // rather than just computing toroidal values ourselves, we lose about
        // 5% in speed.  I don't know if that's overhead having to compile the methods
        // (they get inlined) or if it's due to bad inlining -- I suspect the latter.
        // But the stx() and sty() methods are quite useful compared to the explicit
        // computation, so I'm not willing to hard-inline those myself.
        // Anyway, we can gain all of it back by explicitly pointing to valgrid etc. 
        // with constant locals here, rather than instance variables.  Besides, 
        // locals are a bit faster than instance variables in outer scopes anyway.  So
        // here are five local variables which get us our 5% back and then some.  It's
        // about the most optimization I'm willing to do before we get obtuse.  Could
        // probably squeeze another 5% out before we had to go to linearizing the
        // arrays (which would get us another 25% but require major, unacceptable
        // architectural changes).
        
        // locals are faster than instance variables
        final DoubleGrid2D _valgrid = heatbugs.valgrid;
        final double[][] _valgrid_field = heatbugs.valgrid.field;
        final double[][] _valgrid2_field = heatbugs.valgrid2.field;
        final int _gridHeight = _valgrid.getHeight();  // read-only, so threadsafe with other diffuser
        
        double average;
        
        // for each x and y position
        for(int x=start ; x< end ;x++)
            for(int y=0;y< _gridHeight;y++)
                {
                average = 0.0;
                // for each neighbor of that position
                for(int dx=-1; dx< 2; dx++)
                    for(int dy=-1; dy<2; dy++)
                        {
                        // compute the toroidal <x,y> position of the neighbor
                        int xx = _valgrid.stx(x+dx);
                        int yy = _valgrid.sty(y+dy);
                                                        
                        // compute average
                        average += _valgrid_field[xx][yy];
                        }
                average /= 9.0;
                
                // load the new value into HeatBugs.this.valgrid2
                _valgrid2_field[x][y] = heatbugs.evaporationRate * 
                    (_valgrid_field[x][y] + heatbugs.diffusionRate * 
                     (average - _valgrid_field[x][y]));
                }
        }
    }

